package com.xforceplus.action.trail.starter.advice;

import com.xforceplus.action.trail.core.context.TrailInfoHolder;
import com.xforceplus.action.trail.core.domain.TrailInfo;
import io.geewit.web.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.MethodParameter;
import org.springframework.http.HttpInputMessage;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.servlet.mvc.method.annotation.RequestBodyAdviceAdapter;

import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.lang.reflect.Type;
import java.util.Arrays;

/**
 * <p>
 * 这个advice仅支持参数为@RequestBody的接口日志，一般为POST、PUT、PATCH等类型的请求
 * 对于Get请求及文件上传，不会走到这个逻辑，通过tenant-security提供的interceptor进行拦截记录简单日志
 * </p>
 *
 * @author chenpengpeng@xforceplus.com
 * @date 2021/6/24
 **/
@Slf4j
@ControllerAdvice
@ConditionalOnProperty(prefix = "action.trail.request-body", name = "enable", havingValue = "true")
public class GlobalRequestBodyAdviceAdapter extends RequestBodyAdviceAdapter {

    private final HttpServletRequest httpServletRequest;

    private static final String PASSWORD = "password";

    @Value("${action.trail.request-body.max.length:4096}")
    private long requestBodyMaxLength;

    public GlobalRequestBodyAdviceAdapter(HttpServletRequest httpServletRequest) {
        this.httpServletRequest = httpServletRequest;
    }

    /**
     * Invoked first to determine if this interceptor applies.
     *
     * @param methodParameter the method parameter
     * @param targetType      the target type, not necessarily the same as the method
     *                        parameter type, e.g. for {@code HttpEntity<String>}.
     * @param converterType   the selected converter type
     * @return whether this interceptor should be invoked or not
     */
    @Override
    public boolean supports(MethodParameter methodParameter, Type targetType, Class<? extends HttpMessageConverter<?>> converterType) {
        return httpServletRequest.getContentLength() < requestBodyMaxLength;
    }

    /**
     * The default implementation returns the InputMessage that was passed in.
     *
     * @param inputMessage
     * @param parameter
     * @param targetType
     * @param converterType
     */
    @Override
    public HttpInputMessage beforeBodyRead(HttpInputMessage inputMessage, MethodParameter parameter, Type
            targetType, Class<? extends HttpMessageConverter<?>> converterType) throws IOException {
        return super.beforeBodyRead(inputMessage, parameter, targetType, converterType);
    }

    /**
     * The default implementation returns the body that was passed in.
     *
     * @param body
     * @param inputMessage
     * @param parameter
     * @param targetType
     * @param converterType
     */
    @Override
    public Object afterBodyRead(Object body, HttpInputMessage inputMessage, MethodParameter parameter, Type
            targetType, Class<? extends HttpMessageConverter<?>> converterType) {
        try {
            logRequest(httpServletRequest, body);
        } catch (Exception exception) {
            log.error("error while logging request info:{}", exception.getMessage());
        }
        return super.afterBodyRead(body, inputMessage, parameter, targetType, converterType);
    }

    /**
     * The default implementation returns the body that was passed in.
     *
     * @param body
     * @param inputMessage
     * @param parameter
     * @param targetType
     * @param converterType
     */
    @Override
    public Object handleEmptyBody(Object body, HttpInputMessage inputMessage, MethodParameter parameter, Type
            targetType, Class<? extends HttpMessageConverter<?>> converterType) {
        return super.handleEmptyBody(body, inputMessage, parameter, targetType, converterType);
    }

    private void logRequest(HttpServletRequest request, Object body) {
        TrailInfo trailInfo = TrailInfoHolder.get();
        if (null == trailInfo) {
            trailInfo = new TrailInfo();
        }
        try {
            String queryString = request.getQueryString();
            if (StringUtils.isNotBlank(request.getQueryString())) {
                queryString = this.filterSensitiveLog(queryString);
                trailInfo.setParams(queryString);
            }
            if (body != null) {
                String requestBody = JsonUtils.toJson(body);
                trailInfo.setRequestBody(requestBody);
                TrailInfoHolder.put(trailInfo);
                log.info(JsonUtils.toJson(trailInfo));
            }
        } catch (Exception e) {
            log.error("requestBody-error", e);
        }
    }

    /**
     * 将queryString中的敏感信息屏蔽掉
     *
     * @param source queryString
     * @return 去掉敏感信息的字符串
     */
    private String filterSensitiveLog(String source) {
        final int keyLength = 2;
        if (StringUtils.isBlank(source)) {
            return null;
        } else {
            if (!source.contains(PASSWORD)) {
                return source;
            } else {
                StringBuilder sb = new StringBuilder();
                String[] elements = source.split(";");
                Arrays.stream(elements).forEach(item -> {
                    String[] kv = item.split("=");
                    if (!kv[0].equals(PASSWORD)) {
                        sb.append(item).append(";");
                    } else {
                        sb.append("pwdlength")
                                .append("=");
                        //仅有key没有value的情况
                        if (kv.length < keyLength) {
                            sb.append(0).append(";");
                        } else {
                            //value为“”的情况
                            if (StringUtils.isNotBlank(kv[1])) {
                                sb.append(kv[1].length());
                            } else {
                                sb.append(0);
                            }
                            sb.append(";");
                        }
                    }
                });
                return sb.toString();
            }
        }
    }
}
