package com.xforceplus.bi.commons.disruptor;

import com.lmax.disruptor.*;
import com.lmax.disruptor.dsl.Disruptor;
import com.lmax.disruptor.dsl.ProducerType;
import com.xforceplus.bi.commons.disruptor.bean.TaskEvent;
import com.xforceplus.bi.commons.disruptor.bean.TaskEventFactory;
import com.xforceplus.bi.commons.disruptor.service.WorkService;
import com.xforceplus.bi.commons.jdk.concurrent.ExecutorHelper;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

import javax.annotation.PreDestroy;

@Slf4j
public class DisruptorWorker {
    /**
     * 并行任务书
     */
    @Getter
    @Setter
    private int workers = 1;

    /**
     * 如果要改变线程执行优先级，override此策略. YieldingWaitStrategy会提高响应并在闲时占用70%以上CPU，
     * 慎用SleepingWaitStrategy会降低响应更减少CPU占用，用于日志等场景.
     *
     * @return WaitStrategy
     */
    private WaitStrategy strategy;

    /**
     * Disruptor 对象
     */
    private Disruptor<TaskEvent> disruptor;

    /**
     * RingBuffer
     */
    private RingBuffer<TaskEvent> ringBuffer;

    /**
     * 错误处理监听
     */
    public DisruptorWorker(int workers) {
        this.workers = workers;
    }

    public DisruptorWorker() {
    }

    public static DisruptorWorker create(int leftMoveWorkers) {
        DisruptorWorker disruptorWorker = new DisruptorWorker(leftMoveWorkers);
        disruptorWorker.start();
        return disruptorWorker;
    }

    /**
     * 初始化
     */
    private void init() {
        workers = workers < 4 ? 4 : workers;
        WorkHandler[] workHandlers = new WorkHandler[workers];
        EventFactory<TaskEvent> eventFactory = new TaskEventFactory();
        for (int i = 0; i < workers; i++) {
            workHandlers[i] = new DisruptorHandler();
        }
        if (strategy == null) {
            strategy = new BusySpinWaitStrategy();
        }
        disruptor = new Disruptor<TaskEvent>(eventFactory,
                1024,
                ExecutorHelper.buildNameThreadFactory("datasource-worker"),
                ProducerType.SINGLE,
                strategy);
        disruptor.handleEventsWithWorkerPool(workHandlers);
    }

    public void start() {
        init();
        ringBuffer = disruptor.start();
    }

    @PreDestroy
    public void close() {
        if (disruptor != null) {
            disruptor.shutdown();
        }
        log.info("Close");
    }

    /**
     * 新增任务
     *
     * @param t 业务任务对象
     */
    public <T> void addTask(T t, WorkService<T> workService) {
        ringBuffer.publishEvent(new EventTranslator<TaskEvent>() {
            @Override
            public void translateTo(TaskEvent event, long sequence) {
                event.setTask(t);
                event.setWorkService(workService);
            }
        });
    }
}
