package com.xforceplus.bi.commons.excel;

import lombok.Data;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

import java.io.Closeable;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

/**
 * Excel阅读器
 */
@Data
public class ExcelReader implements Closeable {

    private Workbook workbook;

    /**
     * 读取Excel
     *
     * @param filePath
     * @param suffix
     */
    public void read(String filePath, String suffix) {
        try {
            InputStream is = new FileInputStream(filePath);
            if (suffix.endsWith(ExcelVersion.V_2003.getExtension())) {
                workbook = new HSSFWorkbook(is);
            } else if (suffix.endsWith(ExcelVersion.V_2007_ABOVE.getExtension())) {
                workbook = new XSSFWorkbook(is);
            }
            if (is != null)
                is.close();
        } catch (Exception e) {
            throw new RuntimeException("读取Excel失败");
        }
    }

    /**
     * 读取单元格中的内容,当做字符串
     *
     * @param cell
     * @return
     */
    public String getCellValue(Cell cell) {
        return getCellValue(cell, true);
    }

    /**
     * 读取单元格中的内容
     *
     * @param cell
     * @param treatAsStr
     * @return
     */
    public String getCellValue(Cell cell, boolean treatAsStr) {
        if (cell == null) {
            return "";
        }
        if (treatAsStr) { // Process it into a string
            cell.setCellType(Cell.CELL_TYPE_STRING);
        }
        if (cell.getCellType() == Cell.CELL_TYPE_BOOLEAN) {
            return String.valueOf(cell.getBooleanCellValue());
        } else if (cell.getCellType() == Cell.CELL_TYPE_NUMERIC) {
            return String.valueOf(cell.getNumericCellValue());
        } else {
            return String.valueOf(cell.getStringCellValue());
        }
    }

    public boolean isDate(Cell cell) {
        String dataFormatString = cell.getCellStyle().getDataFormatString();
        if (dataFormatString.contains("yyyy/mm;@")) {
            return true;
        }
        if (dataFormatString.contains("m/d/yy")) {
            return true;
        }
        if (dataFormatString.contains("yy/m/d")) {
            return true;
        }
        if (dataFormatString.contains("mm/dd/yy")) {
            return true;
        }
        if (dataFormatString.contains("dd-mmm-yy")) {
            return true;
        }
        if (dataFormatString.contains("yyyy/m/d")) {
            return true;
        }
        return false;
    }

    /**
     * 关闭workbook
     *
     * @throws IOException
     */
    @Override
    public void close() throws IOException {
        if (workbook != null) {
            workbook.close();
            workbook = null;
        }
    }
}
