package com.xforceplus.bi.commons.excel;

import com.xforceplus.bi.commons.jdk.io.utils.IOUtil;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.DataFormat;
import org.apache.poi.xssf.streaming.SXSSFCell;
import org.apache.poi.xssf.streaming.SXSSFRow;
import org.apache.poi.xssf.streaming.SXSSFSheet;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFRichTextString;

import java.io.Closeable;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.util.concurrent.atomic.AtomicInteger;

import static org.apache.commons.lang3.StringUtils.EMPTY;
import static org.apache.poi.util.IOUtils.closeQuietly;

/**
 * Excel写入器
 */
public class ExcelWriter implements Closeable {

    /**
     * Excel workbook
     */
    private SXSSFWorkbook workbook;

    /**
     * 当前sheet
     */
    private SXSSFSheet sheet;

    /**
     * 当前行
     */
    private SXSSFRow currentRow;

    /**
     * 行号
     */
    AtomicInteger rowIndex = new AtomicInteger();

    /**
     * File output stream
     */
    private FileOutputStream outputStream;

    /**
     * 生成目标文件的路径
     */
    private String targetPath;

    /**
     * 数字格式,默认保留两位
     */
    private DecimalFormat dataFormat;

    /**
     * 设置数字格式
     *
     * @param pattern
     */
    public void setNumberFormat(String pattern) {
        this.dataFormat = new DecimalFormat(pattern);
    }

    /**
     * ExcelWriter构造器
     *
     * @param fetchSize  流式操作Excel时一次性flush的数量
     * @param targetPath 生成目标文件的路径
     */
    public ExcelWriter(Integer fetchSize, String targetPath) throws FileNotFoundException {
        this.targetPath = targetPath;
        this.workbook = new SXSSFWorkbook(fetchSize);
        outputStream = IOUtil.getOut(targetPath);
    }

    /**
     * 写Excel之前需要一个Sheet
     *
     * @return
     */
    public void createSheet() {
        rowIndex.set(0);
        SXSSFSheet sxssfSheet = workbook.createSheet();
        sxssfSheet.trackAllColumnsForAutoSizing();
        this.sheet = sxssfSheet;
    }

    /**
     * 写头部,不带样式
     *
     * @param heads
     * @throws IOException
     */
    @Deprecated
    public void writeHeads(String[] heads) throws IOException {
        createRow();
        for (int columnIndex = 0; columnIndex < heads.length; columnIndex++) {
            createCell(currentRow, columnIndex, heads[columnIndex], null, false);
        }
    }

    /**
     * 写头部,带样式
     *
     * @param heads
     * @param cellStyle
     * @throws IOException
     */
    public void writeHeads(String[] heads, CellStyle cellStyle) throws IOException {
        createRow();
        for (int columnIndex = 0; columnIndex < heads.length; columnIndex++) {
            createCell(currentRow, columnIndex, heads[columnIndex], cellStyle, false);
        }
    }

    /**
     * 创建一行
     */
    public void createRow() {
        currentRow = sheet.createRow(rowIndex.getAndIncrement());
    }

    /**
     * 创建默认单元格样式
     *
     * @return
     */
    public CellStyle createDefaultCellStyle() {
        return workbook.createCellStyle();
    }

    /**
     * 创建number单元格样式
     *
     * @param numberFormat 数字格式化
     * @return
     */
    public CellStyle createNumberCellStyle(String numberFormat) {
        CellStyle style = workbook.createCellStyle();
        DataFormat format = workbook.createDataFormat();
//        style.setDataFormat(format.getFormat("#,##0.00;-#"));
        style.setDataFormat(format.getFormat(numberFormat));
        return style;
    }

    /**
     * 写单元格
     *
     * @param columnIndex
     * @param value
     * @param needTransferToNumber
     * @throws IOException
     */
    public void writeCell(int columnIndex, Object value, boolean needTransferToNumber, CellStyle cellStyle) {
        String resultValue;
        if (needTransferToNumber && dataFormat != null) {
            resultValue = dataFormat.format(new BigDecimal(value.toString()));
        } else {
            resultValue = (value == null) ? EMPTY : value.toString();
        }
        createCell(currentRow, columnIndex, resultValue, cellStyle, needTransferToNumber);
    }

    /**
     * 刷到文件输出流
     *
     * @throws IOException
     */
    public void flush() throws IOException {
        workbook.write(outputStream);
    }

    @Override
    public void close() {
        closeQuietly(outputStream);
        closeQuietly(workbook);
    }

    /**
     * @param row
     * @param cellNum
     * @param value
     * @param cellStyle
     * @param needTransferToNumber
     */
    private void createCell(SXSSFRow row, int cellNum, String value, CellStyle cellStyle, boolean needTransferToNumber) {
        SXSSFCell cell0 = row.createCell(cellNum);
        cell0.setCellStyle(cellStyle);
        if (needTransferToNumber) {
            cell0.setCellValue(Double.parseDouble(value));
        } else {
            cell0.setCellValue(new XSSFRichTextString(value));
        }

    }

}
