package com.xforceplus.bi.commons.oss;

import com.aliyun.oss.OSSClient;
import com.aliyun.oss.common.comm.ResponseMessage;
import com.aliyun.oss.model.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.io.*;
import java.util.ArrayList;
import java.util.List;

/**
 * bi-oss:
 * endpoint: http://oss-cn-szfinance.aliyuncs.com
 * access-key: LTAI9saIp9guF9Lv
 * secure-key: gRh3xEDlWykH6SPFpyv9bLSeo5KCE5
 * bucket: bi-dataplatform-prod
 * path-export: bi-tool-export
 */
@Slf4j
@Service
public class BiOssClientImpl implements BiOssClient {

    private static final String OSS_ERROR_READ = "读取OSS文件失败";

    @Value("${xforce.bi-oss.endpoint}")
    private String endpoint;

    @Value("${xforce.bi-oss.access-key}")
    private String accessKey;

    @Value("${xforce.bi-oss.secure-key}")
    private String secureKey;

    @Value("${xforce.bi-oss.bucket}")
    private String bucket;

    @Override
    public OSSClient getOssClient() {
        return new OSSClient(endpoint, accessKey, secureKey);
    }

    @Override
    public void release(OSSClient ossClient) {
        if (ossClient != null) {
            try {
                ossClient.shutdown();
            } catch (Throwable thr) {
                log.warn("关闭OSSClient失败");
            }
        }
    }

    @Override
    public List<OSSObjectSummary> listFiles(String fileName, String lowerLimitMarker, int limit) {
        OSSClient ossClient = null;
        try {
            ListObjectsRequest listObjectsRequest = new ListObjectsRequest(bucket, fileName,
                    lowerLimitMarker, null, limit);
            ossClient = getOssClient();
            return ossClient.listObjects(listObjectsRequest).getObjectSummaries();
        } catch (Throwable thr) {
            throw thr;
        } finally {
            release(ossClient);
        }
    }

    @Override
    public void upload(String localPath, String targetFileName) throws FileNotFoundException {
        OSSClient ossClient = null;
        String resultMessage = null;
        try {
            ossClient = getOssClient();
            File file = new File(localPath);
            ObjectMetadata objectMeta = new ObjectMetadata();
            objectMeta.setContentLength(file.length());
            InputStream input = new FileInputStream(file);
            PutObjectResult result = ossClient.putObject(bucket, targetFileName, input);
            resultMessage = getResultMessage(result);
        } catch (Throwable thr) {
            log.error(resultMessage, thr);
            throw thr;
        } finally {
            release(ossClient);
        }
    }

    private String getResultMessage(GenericResult result) {
        if (result == null) {
            return null;
        }
        ResponseMessage response = result.getResponse();
        if (response == null) {
            return null;
        }
        try {
            InputStream content = response.getContent();
            int c;
            StringBuilder line = new StringBuilder(256);
            while ((c = content.read()) != -1) {
                line.append((char) c);
            }
            return line.toString();
        } catch (IOException e) {
            log.warn("分析OSS返回失败");
        }
        return null;
    }

    @Override
    public void delete(List<String> fileNames) {
        if (CollectionUtils.isEmpty(fileNames)) {
            return;
        }
        OSSClient ossClient = null;
        String resultMessage = null;
        try {
            ossClient = getOssClient();
            DeleteObjectsResult result = ossClient.deleteObjects(new DeleteObjectsRequest(bucket).withKeys(fileNames));
            resultMessage = getResultMessage(result);
        } catch (Throwable thr) {
            log.error(resultMessage, thr);
            throw thr;
        } finally {
            release(ossClient);
        }
    }

    @Override
    public void deleteByPrefix(String prefix) {
        if (StringUtils.isEmpty(prefix)) {
            log.warn("危险操作,不能删除根");
            return;
        }
        // 创建OSSClient实例。
        OSSClient ossClient = null;
        // 列举所有包含指定前缀的文件并删除。
        String nextMarker = null;
        ObjectListing objectListing = null;
        try {
            do {
                ossClient = getOssClient();
                ListObjectsRequest listObjectsRequest = new ListObjectsRequest(bucket)
                        .withPrefix(prefix)
                        .withMarker(nextMarker);
                objectListing = ossClient.listObjects(listObjectsRequest);
                if (objectListing.getObjectSummaries().size() > 0) {
                    List<String> keys = new ArrayList<String>();
                    for (OSSObjectSummary s : objectListing.getObjectSummaries()) {
                        keys.add(s.getKey());
                    }
                    DeleteObjectsRequest deleteObjectsRequest = new DeleteObjectsRequest(bucket).withKeys(keys);
                    ossClient.deleteObjects(deleteObjectsRequest);
                }

                nextMarker = objectListing.getNextMarker();
            } while (objectListing.isTruncated());
        } catch (Throwable thr) {
            throw thr;
        } finally {
            // 关闭OSSClient。
            release(ossClient);
        }
    }

    @Override
    public InputStream read(OSSClient ossClient, String filePath) {
        OSSObject ossObject = ossClient.getObject(bucket, filePath);
        return ossObject.getObjectContent();
    }

    @Override
    public boolean doesObjectExist(String filename) {
        OSSClient ossClient = null;
        try {
            ossClient = getOssClient();
            return ossClient.doesObjectExist(bucket, filename);
        } catch (Throwable thr) {
            throw thr;
        } finally {
            release(ossClient);
        }
    }

    @Override
    public boolean doesObjectNotExist(String filename) {
        return !doesObjectExist(filename);
    }

}
