package com.xforceplus.bi.commons.zookeeper;

import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.framework.recipes.cache.ChildData;
import org.apache.curator.framework.recipes.cache.NodeCache;
import org.apache.curator.retry.ExponentialBackoffRetry;
import org.apache.zookeeper.data.Stat;

import java.text.MessageFormat;

/**
 * Zookeeper客户端
 */
public class ZookeeperClient {
    // ### 单例模式 BEGIN ###
    private volatile static ZookeeperClient INSTANCE;

    private ZookeeperClient(String zkServer) {
        curator = CuratorFrameworkFactory.builder()
                .connectString(zkServer)
                .sessionTimeoutMs(10000)
                .retryPolicy(new ExponentialBackoffRetry(1000, 5))
                .build();
        curator.start();
    }

    public static ZookeeperClient getInstance(String zkServer) {
        if (INSTANCE == null) {
            synchronized (ZookeeperClient.class) {
                if (INSTANCE == null) {
                    INSTANCE = new ZookeeperClient(zkServer);
                }
            }
        }
        return INSTANCE;
    }
    // ### 单例模式 END ###


    /**
     * Curator是Netflix公司开源的一套Zookeeper客户端框架
     */
    private volatile static CuratorFramework curator;

    /**
     * 当路径不存在创建节点
     *
     * @param nodePath 节点路径
     */
    public void createNodeDeeplyIfNotExists(String nodePath) {
        Stat exists = exists(nodePath);
        if (exists == null) {
            createPersistentNode(nodePath, "");
        }
    }

    /**
     * @param nodePath  节点路径（如果父节点不存在则会自动创建父节点）
     * @param nodeValue 节点数据
     * @return 返回创建成功的节点路径
     */
    private String createPersistentNode(String nodePath, String nodeValue) {
        try {
            return curator.create().creatingParentsIfNeeded()
                    .forPath(nodePath, nodeValue.getBytes());
        } catch (Exception e) {
            throw new RuntimeException("创建节点时发生异常 ", e);
        }
    }

    /**
     * 判断节点路径是否存在
     *
     * @param nodePath 节点路径
     * @return
     */
    public Stat exists(String nodePath) {
        try {
            return curator.checkExists().forPath(nodePath);
        } catch (Exception e) {
            throw new RuntimeException("检查路径是否存在时发生异常", e);
        }
    }

    /**
     * 注册节点监听器
     *
     * @param nodePath 节点路径
     * @return
     */
    public NodeCache registerNodeCacheListener(ZookeeperNodeUpdateListener listener, String nodePath) {
        try {
            //1. 创建一个NodeCache
            NodeCache nodeCache = new NodeCache(curator, nodePath);
            //2. 添加节点监听器
            nodeCache.getListenable().addListener(() -> {
                ChildData childData = nodeCache.getCurrentData();
                if (childData != null) {
                    byte[] data = childData.getData();
                    listener.receive(data == null ? "" : new String(data, "UTF-8"));
                }
            });
            //3. 启动监听器
            nodeCache.start();
            //4. 返回NodeCache
            return nodeCache;
        } catch (Exception e) {
            throw new RuntimeException(MessageFormat.format("注册节点监听器出现异常,nodePath:{0}", nodePath), e);
        }
    }

    /**
     * 更新节点数据
     *
     * @param nodePath  节点路径
     * @param nodeValue 节点值
     */
    public void updateNodeValue(String nodePath, String nodeValue) {
        try {
            curator.setData().forPath(nodePath, nodeValue.getBytes("UTF-8"));
        } catch (Exception e) {
            throw new RuntimeException("更新数据失败", e);
        }
    }

    /**
     * 查询路径下的字符串形式的值
     *
     * @param nodePath 节点路径
     * @return
     */
    public String lookupDataStr(String nodePath) {
        try {
            byte[] bytes = curator.getData().forPath(nodePath);
            return new String(bytes, "UTF-8");
        } catch (Exception e) {
            throw new RuntimeException("获取节点数据失败", e);
        }
    }
}
