package com.xforceplus.bi.ultraman.binlog;

import com.github.shyiko.mysql.binlog.BinaryLogClient;
import com.github.shyiko.mysql.binlog.GtidSet;
import com.github.shyiko.mysql.binlog.event.EventHeaderV4;
import com.github.shyiko.mysql.binlog.event.EventType;
import com.github.shyiko.mysql.binlog.event.GtidEventData;
import com.github.shyiko.mysql.binlog.event.RotateEventData;
import com.github.shyiko.mysql.binlog.event.deserialization.EventDeserializer;
import com.google.common.collect.Lists;
import com.xforceplus.bi.ultraman.binlog.handler.CDCBusinessHandler;
import com.xforceplus.bi.ultraman.binlog.handler.PositionHandler;
import com.xforceplus.bi.ultraman.binlog.pojo.BinlogPositionEntity;
import com.xforceplus.bi.ultraman.binlog.pojo.DatabaseInfo;
import lombok.Getter;
import lombok.Singular;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.RandomUtils;

import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.util.List;

/**
 * MySQL binlog CDC
 */
@Slf4j
public class MySQLBinaryLogCDC implements LifeCycleCDC {
    @Singular
    private List<CDCBusinessHandler> cdcBusinessHandlers = Lists.newArrayList();

    @Getter
    private DatabaseInfo databaseInfo;

    private PositionHandler positionHandler;

    private boolean useGTID;

    private GtidSet gtidSet;

    public MySQLBinaryLogCDC(DatabaseInfo databaseInfo,
                             PositionHandler positionHandler,
                             boolean useGTID) {
        this.databaseInfo = databaseInfo;
        this.positionHandler = positionHandler;
        this.useGTID = useGTID;
    }

    private BinaryLogClient client;

    private BinaryLogClient.EventListener eventListener;

    private BinaryLogClient.LifecycleListener lifecycleListener;

    public void addCDCEventHandler(CDCBusinessHandler cdcBusinessHandler) {
        if (this.client != null && this.client.isConnected()) {
            log.error("{} has started, cannot add new CDCEventHandler into it", this.getClass().getSimpleName());
            return;
        }
        cdcBusinessHandlers.add(cdcBusinessHandler);
    }

    @Override
    public void start() throws Exception {
        if (isStarted()) {
            log.warn("{} has already started, do not do it again", this.getClass().getSimpleName());
            return;
        }

        // BinaryLogClient
        this.client = createBinlogClient();
        this.client.setEventDeserializer(new EventDeserializer());
        this.client.setServerId(getRandomServerId());
//        this.client.setServerId(81778446);

        Object position = positionHandler.getPosition(this.databaseInfo);

        if (useGTID) {
            // GTID
            this.client.setGtidSet(position == null ? "" : (String) position);
            this.client.setGtidSetFallbackToPurged(true);
            gtidSet = new GtidSet(this.client.getGtidSet());
            log.info("Now gtid is {}", this.client.getGtidSet());
        } else {
            // binlog position
            BinlogPositionEntity binlogPositionEntity = (BinlogPositionEntity) position;
            if (binlogPositionEntity != null &&
                    binlogPositionEntity.getBinlogName() != null &&
                    binlogPositionEntity.getPosition() != null) {
                client.setBinlogFilename(binlogPositionEntity.getBinlogName());
                client.setBinlogPosition(binlogPositionEntity.getPosition());
                log.info("Set binlog position = {}", binlogPositionEntity.getPosition());
            }
            log.info("Now position is {}, {}", client.getBinlogFilename(), client.getBinlogPosition());
        }

        // register event listener
        this.eventListener = createBinlogEventListener();
        this.client.registerEventListener(this.eventListener);

        // register lifecycle listener
        this.lifecycleListener = createBinlogLifecycleListener();
        this.client.registerLifecycleListener(this.lifecycleListener);

        // connect...
        this.client.connect();
    }

    private long getRandomServerId() {
        try {
            return SecureRandom.getInstanceStrong().nextLong();
        } catch (NoSuchAlgorithmException e) {
            return RandomUtils.nextLong();
        }
    }

    /**
     * create binlog client
     *
     * @return
     */
    private BinaryLogClient createBinlogClient() {
        BinaryLogClient client = new BinaryLogClient(
                databaseInfo.getHostname(),
                databaseInfo.getPort(),
                databaseInfo.getUsername(),
                databaseInfo.getPassword());
        return client;
    }

    /**
     * create binlog event listener
     *
     * @return
     */
    private BinaryLogClient.EventListener createBinlogEventListener() {
        return event -> {
            EventHeaderV4 header = event.getHeader();
            /*
             * 不计入position更新的事件类型
             * FORMAT_DESCRIPTION类型为binlog起始时间
             * HEARTBEAT为心跳检测事件，不会写入master的binlog，记录该事件的position会导致重启时报错
             */
            if (header.getEventType() == EventType.FORMAT_DESCRIPTION ||
                    header.getEventType() == EventType.HEARTBEAT) {
                return;
            }

            // 保存消费位置(GTID或者位点)
            if (useGTID) {
                // GTID
                if (header.getEventType() == EventType.GTID) {
                    GtidEventData gtidEventData = event.getData();
                    this.gtidSet.add(gtidEventData.getGtid());
                    positionHandler.savePosition(databaseInfo, this.gtidSet.toString());
                }
            } else {
                // binlog position
                BinlogPositionEntity binlogPositionEntity = new BinlogPositionEntity();
                //处理rotate事件,这里会替换调binlog fileName
                if (header.getEventType().equals(EventType.ROTATE)) {
                    RotateEventData rotateEventData = event.getData();
                    binlogPositionEntity.setBinlogName(rotateEventData.getBinlogFilename());
                    binlogPositionEntity.setPosition(rotateEventData.getBinlogPosition());
                } else { //统一处理事件对应的binlog position
                    //在Redis中获取获取binlog的position配置
                    binlogPositionEntity = (BinlogPositionEntity) positionHandler.getPosition(databaseInfo);
                    binlogPositionEntity.setPosition(header.getPosition());
                }
                binlogPositionEntity.setServerId(header.getServerId());
                //将最新的配置保存到Redis中
                positionHandler.savePosition(databaseInfo, binlogPositionEntity);
            }

            // business operations
            cdcBusinessHandlers.forEach(handler -> handler.handle(event));
        };
    }

    /**
     * create binlog lifecycle Listener
     *
     * @return
     */
    private BinaryLogClient.LifecycleListener createBinlogLifecycleListener() {
        return new BinaryLogClient.LifecycleListener() {
            @Override
            public void onConnect(BinaryLogClient client) {
//                client.setBinlogPosition(-4);
                log.info("Successfully connect to the mysql server");
            }

            @Override
            public void onCommunicationFailure(BinaryLogClient client, Exception ex) {
                log.error("Communication Failure", ex);
                positionHandler.clear(databaseInfo);
                try {
                    stop();
                } catch (Exception e) {
                    log.error("Fatal to stop binlog listener", e);
                }
                try {
                    start();
                } catch (Exception e) {
                    log.error("Fatal to start binlog listener", e);
                }
                log.info("position is clean and binlog listener is restarted");
            }

            @Override
            public void onEventDeserializationFailure(BinaryLogClient client, Exception ex) {
                log.error("Event Deserialization Failure", ex);
            }

            @Override
            public void onDisconnect(BinaryLogClient client) {
                log.info("Disconnect from the mysql server");
            }
        };
    }

    @Override
    public void stop() throws Exception {
        try {
//            if (client != null) {
//                client.disconnect();
//            }
            if (eventListener != null && client != null) {
                client.unregisterEventListener(eventListener);
            }
            if (lifecycleListener != null && client != null) {
                client.unregisterLifecycleListener(lifecycleListener);
            }
            client = null;
            eventListener = null;
            lifecycleListener = null;
            log.info("MySQLBinaryLogCDC stopped...");
        } catch (Throwable throwable) {
            throw new CDCException("Error closing MySQL Binlog CDC connection");
        }
    }

    @Override
    public boolean isStarted() {
        return this.client != null && this.client.isConnected();
    }
}
