package com.xforceplus.bi.ultraman.dictionaries.service.impl;

import com.xforceplus.bi.ultraman.dictionaries.bean.DictMsgBean;
import com.xforceplus.bi.ultraman.dictionaries.service.DictServiceInterface;
import com.xforceplus.bi.ultraman.utils.EntityNameUtil;
import com.xforceplus.bi.ultraman.utils.bean.EntityType;
import lombok.extern.slf4j.Slf4j;
import org.jooq.DSLContext;
import org.jooq.Record;
import org.jooq.Table;
import org.jooq.impl.DSL;
import org.jooq.impl.SQLDataType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

import static org.jooq.impl.DSL.field;


/**
 * @author ligang
 */
@Service
@Slf4j
public class DictServiceInterfaceImpl implements DictServiceInterface {

    @Autowired
    private DSLContext dslContext;

    /**
     * 更新字典
     *
     * @param dictMsgBean
     */
    @Override
    @Transactional(value = "greenplumnTransactionManager", rollbackFor = Exception.class)
    public void modifyDictDatas(DictMsgBean dictMsgBean) {
        Table<Record> table = getRecordTable(dictMsgBean);
        if (existaTable(table)) {
            deleteDictTable(dictMsgBean);
            log.info("删除字典表数据({})", table.getName());
        } else {
            createDictTable(dictMsgBean.getName(), table);
            log.info("创建字典表({})", table.getName());
        }
        log.info("插入字典表数据({}) start", table.getName());
        insertDictValues(table, dictMsgBean.getOptions());
        log.info("插入字典表数据({}) end", table.getName());
    }

    /**
     * 判断表是否存在
     *
     * @param table
     * @return
     */
    public boolean existaTable(Table<Record> table) {
        try {
            dslContext.fetchExists(table);
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 删除原有Dict表
     *
     * @param dictMsgBean
     */
    private void deleteDictTable(DictMsgBean dictMsgBean) {
        dslContext.deleteFrom(getRecordTable(dictMsgBean)).execute();
    }

    /**
     * 创建Dict表
     *
     * @param tableComment
     * @param table
     */
    private void createDictTable(String tableComment, Table<Record> table) {
        dslContext
                .createTable(table)
                .column("id", SQLDataType.VARCHAR(100).nullable(false))
                .column("value", SQLDataType.VARCHAR(256).nullable(false))
                .column("text", SQLDataType.VARCHAR(256).nullable(false))
                .execute();
        String tableName = table.getName();
        dslContext.commentOnTable(table).is(tableComment).execute();
        dslContext.commentOnColumn(field(tableName + "." + "id")).is("字典ID").execute();
        dslContext.commentOnColumn(field(tableName + "." + "value")).is("字典值").execute();
        dslContext.commentOnColumn(field(tableName + "." + "text")).is("字典值描述").execute();
    }

    /**
     * 插入字典表数据
     *
     * @param table
     * @param
     */
    private void insertDictValues(Table<Record> table, List<DictMsgBean.OptionsBean> options) {
        for (DictMsgBean.OptionsBean option : options) {
            dslContext
                    .insertInto(table)
                    .columns(field("id"), field("value"), field("text"))
                    .values(option.getId(), option.getValue(), option.getText())
                    .execute();
            log.info("插入数据(id={},value={},text={})", option.getId(), option.getValue(), option.getText());
        }
    }

    /**
     * 获取表实例
     *
     * @param dictMsgBean
     * @return
     */
    private Table<Record> getRecordTable(DictMsgBean dictMsgBean) {
        Table<Record> table =
                DSL.table(
                        EntityNameUtil.tableNameConvert(
                                dictMsgBean.getId(), EntityType.DICT, dictMsgBean.getCode()));
        return table;
    }
}
