package com.xforceplus.delivery.cloud.tax.api.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.xforceplus.delivery.cloud.common.api.GlobalResult;
import com.xforceplus.delivery.cloud.common.api.PageResult;
import com.xforceplus.delivery.cloud.common.api.ViewResult;
import com.xforceplus.delivery.cloud.mybatis.ExampleWrapper;
import com.xforceplus.delivery.cloud.mybatis.PageQueryStringHandler;
import com.xforceplus.delivery.cloud.mybatis.ViewPage;
import com.xforceplus.delivery.cloud.tax.api.domain.DispatchQueueVO;
import com.xforceplus.delivery.cloud.tax.api.entity.DispatchRetryQueue;
import com.xforceplus.delivery.cloud.tax.api.service.IDispatchRetryQueueService;
import com.xforceplus.delivery.cloud.tax.api.support.DataDispatchBeanPostProcessor;
import com.xforceplus.delivery.cloud.tax.api.support.DataDispatchJobHandler;
import com.xforceplus.delivery.cloud.tax.api.support.DataDispatchJobParam;
import com.xforceplus.delivery.cloud.webmvc.support.QueryString;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.web.bind.annotation.*;

import java.util.Collection;
import java.util.Map;

/**
 * <p>
 * 分发重试队列 前端控制器
 * </p>
 *
 * @author Hanyongjie
 * @since 2020-07-26
 */
@Slf4j
@RestController
@RequestMapping("/dispatch/")
@ApiOperation(value = "分发队列")
public class DispatchRetryQueueController {

    @Autowired
    private DataDispatchJobHandler dataDispatchJobHandler;

    @Autowired
    private DataDispatchBeanPostProcessor dataDispatchBeanPostProcessor;

    @Autowired
    private IDispatchRetryQueueService iDispatchRetryQueueService;

    @PostMapping("/pageList")
    @ApiOperation("分页查询分发队列")
    @PreAuthorize("hasAuthority('dispatch:retry:list')")
    public PageResult<DispatchRetryQueue> pageList(@RequestBody Map<String, String> params,
                                                   @QueryString(handler = {PageQueryStringHandler.class}) Page<DispatchRetryQueue> page) {
        return ViewPage.of(iDispatchRetryQueueService.page(page, new ExampleWrapper<>(DispatchRetryQueue.class, params)));
    }

    /**
     * 手工执行数据分发
     *
     * @param evtName
     * @param dispatchArgs
     * @return
     */
    @PostMapping("/manual")
    @ApiOperation("手工执行数据分发")
    @PreAuthorize("hasAuthority('dispatch:manual:retry')")
    public GlobalResult doManualDispatch(@RequestParam("evtName") String evtName, @RequestBody String dispatchArgs) {
        return this.dataDispatchBeanPostProcessor.invokeForJson(evtName, dispatchArgs);
    }

    /**
     * 重新执行队列中的数据分发任务
     *
     * @param dispatchRetryQueue
     * @return
     */
    @PostMapping("/queue/retry")
    @ApiOperation("重新执行队列中的数据分发任务")
    @PreAuthorize("hasAuthority('dispatch:queue:retry')")
    public GlobalResult doQueueRetry(@RequestBody DispatchRetryQueue dispatchRetryQueue) {
        DataDispatchJobParam jobParam = new DataDispatchJobParam();
        jobParam.setMaxCounter(Integer.MAX_VALUE);
        this.dataDispatchJobHandler.doDispatch(dispatchRetryQueue, jobParam);
        return ViewResult.success();
    }

    /***
     * 通过ID集合删除执行队列中的数据分发任务
     *
     * @param idList
     * @return
     */
    @PostMapping("/queue/removeByIds")
    @ApiOperation("通过ID集合删除执行队列中的数据分发任务")
    @PreAuthorize("hasAuthority('dispatch:queue:remove')")
    public GlobalResult removeByIds(@RequestBody Collection<String> idList) {
        return ViewResult.of(this.iDispatchRetryQueueService.removeByIds(idList));
    }

    /**
     * 通过ID/服务/事件/关键字的组合删除(ID优先)
     *
     * @param dispatchRetryQueue
     * @return
     */
    @PostMapping("/queue/removeByExample")
    @ApiOperation("通过ID/服务/事件/关键字的组合删除(ID优先)")
    @PreAuthorize("hasAuthority('dispatch:queue:remove')")
    public GlobalResult removeByExample(@RequestBody DispatchQueueVO dispatchRetryQueue) {
        return this.iDispatchRetryQueueService.removeByExample(dispatchRetryQueue);
    }

}