package com.xforceplus.delivery.cloud.tax.api.janus;

import com.google.common.collect.Maps;
import com.xforceplus.delivery.cloud.common.util.BeanUtils;
import com.xforceplus.delivery.cloud.common.util.StringUtils;
import com.xforceplus.delivery.cloud.tax.api.constants.AopOperationEnum.InvokeIdentifier;
import com.xforceplus.delivery.cloud.tax.api.constants.AopOperationEnum.OperateType;
import com.xforceplus.delivery.cloud.tax.api.logging.AopOperation;
import com.xforceplus.delivery.cloud.tax.api.logging.AopOperationAspect;
import com.xforceplus.delivery.cloud.tax.api.service.ISealedMessageCallback;
import org.apache.commons.lang3.ObjectUtils;
import org.slf4j.MDC;
import org.slf4j.spi.MDCAdapter;
import org.springframework.core.ResolvableType;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import static com.xforceplus.delivery.cloud.tax.api.service.ISealedMessageCallback.ACK_PREFIX;

/**
 * 高于生活，源于生活
 * SealedRecMessageDispatcher
 *
 * @Author: Hanyongjie
 * @CreateDate: 2020-10-28 17:02
 * @Version: 1.0
 */
@Component
public class SealedRecMessageDispatcher {

    private Map<String, SealedRecMessageHandler<?>> handlerMap;
    private Map<String, Class<?>> payloadClassMap;

    public SealedRecMessageDispatcher(List<SealedRecMessageHandler<?>> handlers) {
        this.handlerMap = Maps.newHashMap();
        this.payloadClassMap = Maps.newHashMap();
        //ResolvableType.forInstance(handlers.get(0)).getGeneric(0).resolve()
        handlers.forEach(handler -> {
            final ResolvableType handlerType = ResolvableType.forInstance(handler).getSuperType();
            SealedRecMessage sealedRecMessage = AnnotationUtils.findAnnotation(handler.getClass(), SealedRecMessage.class);
            final String dispatchIdentifier = StringUtils.toUpperCase(sealedRecMessage.value());
            this.handlerMap.put(dispatchIdentifier, handler);
            final ResolvableType[] handlerTypeInterfaces = handlerType.getInterfaces();
            for (ResolvableType handlerTypeInterface : handlerTypeInterfaces) {
                if (Objects.equals(handlerTypeInterface.getRawClass(), SealedRecMessageHandler.class)) {
                    Class<?> payloadClass = handlerTypeInterface.getGeneric(0).getRawClass();
                    this.payloadClassMap.put(dispatchIdentifier, payloadClass);
                    break;
                }
            }
        });
    }

    /**
     * 分发消息
     *
     * @param sealedRecMessage
     */
    @AopOperation(invokeIdentifier = InvokeIdentifier.GLOBAL_FEEDBACK, operateType = OperateType.FEEDBACK, keyword = "#{#p0.header.msgId}")
    public void dispatchSealedRecMessage(com.xforceplus.core.common.domain.SealedRecMessage sealedRecMessage) {
        final com.xforceplus.core.common.domain.SealedRecMessage.Header header = sealedRecMessage.getHeader();
        try {
            this.beforeDispatch(header).ifPresent(identifier -> this.executeDispatch(identifier, sealedRecMessage));
        } finally {
            this.afterDispatch(header);
        }
    }

    /**
     * 分发前
     *
     * @param header
     */
    protected Optional<String> beforeDispatch(com.xforceplus.core.common.domain.SealedRecMessage.Header header) {
        final String requestName = header.getRequestName();
        AopOperationAspect.getOp().ifPresent(op -> op.setBusinessTypeCode(requestName));
        if (StringUtils.isBlank(requestName)) {
            AopOperationAspect.getOp().ifPresent(op -> op.setOperateRemark("requestName is null"));
            return Optional.empty();
        }
        final String identifier = StringUtils.toUpperCase(requestName);
        if (!this.payloadClassMap.containsKey(identifier)) {
            AopOperationAspect.getOp().ifPresent(op -> op.setOperateRemark("payloadClass is null"));
            return Optional.empty();
        }
        if (!this.handlerMap.containsKey(identifier)) {
            AopOperationAspect.getOp().ifPresent(op -> op.setOperateRemark("handlerClass is null"));
            return Optional.empty();
        }
        //MDC.put();
        final Map<String, Object> sourceBean = BeanUtils.beanToMap(header);
        ISealedMessageCallback.ACK_MAPS.forEach((k, p) ->
                Optional.ofNullable(sourceBean.get(p)).filter(ObjectUtils::isNotEmpty).map(Objects::toString).ifPresent(val -> MDC.put(ACK_PREFIX + k, val))
        );
        return Optional.of(identifier);
    }

    /**
     * 分发后
     *
     * @param header
     */
    protected void afterDispatch(com.xforceplus.core.common.domain.SealedRecMessage.Header header) {
        Optional.ofNullable(MDC.getMDCAdapter()).map(MDCAdapter::getCopyOfContextMap).ifPresent(ctx -> ctx.keySet().removeIf(key -> StringUtils.startWith(key, ACK_PREFIX)));
    }

    /**
     * 分发
     *
     * @param dispatchIdentifier
     * @param sealedRecMessage
     */
    protected void executeDispatch(String dispatchIdentifier, com.xforceplus.core.common.domain.SealedRecMessage sealedRecMessage) {
        final Class payloadClass = this.payloadClassMap.get(dispatchIdentifier);
        final SealedRecMessageHandler<?> handler = this.handlerMap.get(dispatchIdentifier);
        handler.process(JanusCoreReceiveMsg.of(sealedRecMessage, payloadClass));
    }

}
