package com.xforceplus.delivery.cloud.tax.api.logging;

import com.xforceplus.delivery.cloud.common.aop.SpELContext;
import com.xforceplus.delivery.cloud.common.api.GlobalResult;
import com.xforceplus.delivery.cloud.common.api.ICode;
import com.xforceplus.delivery.cloud.common.api.ResultCode;
import com.xforceplus.delivery.cloud.common.component.DataDictCache;
import com.xforceplus.delivery.cloud.common.util.*;
import com.xforceplus.delivery.cloud.tax.api.constants.AopOperationEnum.InvokeIdentifier;
import com.xforceplus.delivery.cloud.tax.api.constants.AopOperationEnum.OperateState;
import com.xforceplus.delivery.cloud.tax.api.constants.AopOperationEnum.OperateType;
import com.xforceplus.delivery.cloud.tax.api.entity.BusinessOperate;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.slf4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;
import sun.reflect.annotation.AnnotationParser;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * @vlog: 高于生活，源于生活
 * @Desc: TODO
 * @Author: Hanyongjie
 * @CreateDate: 2020-09-03 22:05
 * @Version: 1.0
 */
@Slf4j
@Aspect
@Component
@Order(AopOperationAspect.ORDERED_PRECEDENCE)
public class AopOperationAspect extends Observable implements InitializingBean {

    private static final ThreadLocal<Stack<BusinessOperate>> THREAD_LOCAL = ThreadLocal.withInitial(Stack::new);

    public static final int ORDERED_PRECEDENCE = Ordered.LOWEST_PRECEDENCE - 10;

    private final ConcurrentMap<String, AopOperation> janusOpAnnoMaps;

    private static AopOperationAspect SELF;

    public AopOperationAspect(AopOperationConsumer aopOperationConsumer) {
        this.addObserver(aopOperationConsumer);
        this.janusOpAnnoMaps = new ConcurrentHashMap<>();
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        SELF = this;
    }

    /**
     * 拦截往集成平台发送HTTP请求的方法
     *
     * @param joinPoint
     * @return
     * @throws Throwable
     */
    @Around("execution(public com.xforceplus.core.common.domain.JsonResult com.xforceplus.core.common.utils.ApolloClientUtils.sendMsg(com.xforceplus.core.common.domain.JanusRequest))")
    public Object aroundHttpMsg(ProceedingJoinPoint joinPoint) throws Throwable {
        return this.aroundOp(joinPoint, InvokeIdentifier.JANUS_HTTP, OperateType.JANUS_HTTP);
    }

    /**
     * 拦截往集成平台发送POST请求的方法
     *
     * @param joinPoint
     * @return
     * @throws Throwable
     */
    @Around("execution(public com.xforceplus.core.common.domain.JsonResult com.xforceplus.core.common.utils.ApolloClientUtils.sendHttpMsg(com.xforceplus.core.common.domain.JanusRequest))")
    public Object aroundHttpPostMsg(ProceedingJoinPoint joinPoint) throws Throwable {
        return this.aroundOp(joinPoint, InvokeIdentifier.JANUS_HTTP_POST, OperateType.JANUS_HTTP_POST);
    }

    /**
     * 拦截往集成平台发送GET请求的方法
     *
     * @param joinPoint
     * @return
     * @throws Throwable
     */
    @Around("execution(public com.xforceplus.core.common.domain.JsonResult com.xforceplus.core.common.utils.ApolloClientUtils.sendHttpGetMsg(com.xforceplus.core.common.domain.JanusRequest))")
    public Object aroundHttpGetMsg(ProceedingJoinPoint joinPoint) throws Throwable {
        return this.aroundOp(joinPoint, InvokeIdentifier.JANUS_HTTP_GET, OperateType.JANUS_HTTP_GET);
    }

    private Object aroundOp(ProceedingJoinPoint joinPoint, String invokeIdentifier, OperateType operateType) throws Throwable {
        final String signatureLongString = joinPoint.getSignature().toLongString();
        log.debug("send http msg proceeding join point for - {}", signatureLongString);
        AopOperation aopOperation = this.janusOpAnnoMaps.computeIfAbsent(signatureLongString, key -> {
            Map<String, Object> memberValues = new HashMap<>();
            memberValues.put("invokeIdentifier", invokeIdentifier);
            memberValues.put("operateType", operateType);
            memberValues.put("keyword", "#{#p0.action}");
            memberValues.put("operateState", "#{#r.code=='0'?'200':'500'}");
            memberValues.put("operateRemark", "#{#r?.message}");
            memberValues.put("returnValue", "#{#r?.data}");
            memberValues.put("businessTypeCode", "");
            memberValues.put("businessKey", "");
            return (AopOperation) AnnotationParser.annotationForMap(AopOperation.class, memberValues);
        });
        return this.around(joinPoint, aopOperation);
    }

    /**
     * 拦截@AopOperation修饰的方法
     *
     * @param joinPoint
     * @param aopOperation
     * @return
     * @throws Throwable
     */
    @Around("@annotation(aopOperation)")
    public Object around(ProceedingJoinPoint joinPoint, AopOperation aopOperation) throws Throwable {
        Object returnValue = null;
        Throwable throwable = null;
        Instant now = Instant.now();
        final Optional<String> traceId = TraceUtils.getTraceId();
        Logger logger = AspectUtils.getLogger(joinPoint).orElse(log);
        SpELContext spELContext = this.proceedBefore(joinPoint, aopOperation, traceId);
        logger.debug("AopOperation starting - {}", getOp());
        try {
            if (log.isDebugEnabled()) {
                logger.debug("AopOperation arguments - {}: {}", getOp(), JsonUtils.toJson(joinPoint.getArgs()));
            }
            returnValue = joinPoint.proceed();
            if (log.isDebugEnabled()) {
                logger.debug("AopOperation finished - {}: {}", getOp(), JsonUtils.toJson(returnValue));
            }
            spELContext.setResult(returnValue);
        } catch (Throwable e) {
            logger.debug("AopOperation throwable - {}", getOp(), e);
            throwable = e;
            throw e;
        } finally {
            long millis = Duration.between(now, Instant.now()).toMillis();
            logger.debug("AopOperation completed elapsed {} ms", millis);
            this.proceedAfter(spELContext, aopOperation, returnValue, throwable);
            if (!traceId.isPresent()) {
                TraceUtils.clsMdcTraceId();
            }
        }
        return returnValue;
    }

    private SpELContext proceedBefore(ProceedingJoinPoint joinPoint, AopOperation aopOperation, Optional<String> traceId) {
        if (!traceId.isPresent()) {
            TraceUtils.setMdcTraceId(TraceUtils.genMdcTraceId());
        }
        final Stack<BusinessOperate> operateStack = THREAD_LOCAL.get();
        Optional<BusinessOperate> prevBusinessOperate = Optional.ofNullable(operateStack.isEmpty() ? null : operateStack.peek());
        BusinessOperate currBusinessOperate = operateStack.push(new BusinessOperate());
        // setTraceId
        TraceUtils.getTraceId().ifPresent(currBusinessOperate::setTraceId);
        currBusinessOperate.setOperateType(aopOperation.operateType().getType());
        currBusinessOperate.setInvokeIdentifier(aopOperation.invokeIdentifier());
        //
        SpELContext spELContext = new SpELContext(joinPoint);
        // setBusinessKey
        String spelBusinessKey = aopOperation.businessKey();
        if (StringUtils.isNotBlank(spelBusinessKey)) {
            currBusinessOperate.setBusinessKey(spELContext.getValue(spelBusinessKey, String.class));
        }
        // setBusinessTypeCode
        currBusinessOperate.setBusinessTypeCode(aopOperation.businessTypeCode());
        // 处理Business属性
        this.handleBusinessAttr(currBusinessOperate, prevBusinessOperate);
        // setKeyword
        final String spelKeyword = aopOperation.keyword();
        if (StringUtils.isNotBlank(spelKeyword)) {
            currBusinessOperate.setKeyword(spELContext.getValue(spelKeyword, String.class));
        }
        currBusinessOperate.setArguments(JsonUtils.toJson(joinPoint.getArgs()));
        currBusinessOperate.setOperateTime(LocalDateTime.now());

        return spELContext;
    }

    /**
     * AopOperation
     *
     * @param aopOperation
     * @param spELContext
     */
    private void proceedAfter(SpELContext spELContext, AopOperation aopOperation, Object returnValue, Throwable throwable) {
        final Stack<BusinessOperate> operateStack = THREAD_LOCAL.get();
        final BusinessOperate currBusinessOperate = operateStack.pop();
        Optional<BusinessOperate> prevBusinessOperate = Optional.ofNullable(operateStack.isEmpty() ? null : operateStack.peek());
        // 处理Business属性
        this.handleBusinessAttr(currBusinessOperate, prevBusinessOperate);
        if (throwable == null) {
            // 处理Operation属性
            this.handleOperationAttr(spELContext, aopOperation, returnValue, currBusinessOperate, prevBusinessOperate);
        } else {
            if (StringUtils.isBlank(currBusinessOperate.getOperateRemark())) {
                currBusinessOperate.setOperateRemark("系统异常," + throwable.getLocalizedMessage());
            }
            currBusinessOperate.setOperateState(OperateState.FAILURE.getState());
            currBusinessOperate.setStackTrace(ExceptionUtils.toStringRootCauseStackTrace(throwable));
        }
        // 尝试从重试线程填充数据
        this.wrapFromRetry(currBusinessOperate);
        // 保存操作记录
        putOp(currBusinessOperate);
    }

    /**
     * 尝试从重试线程填充数据
     *
     * @param currBusinessOperate
     */
    private void wrapFromRetry(BusinessOperate currBusinessOperate) {
        final BusinessOperate retryOperate = AopOperationActuator.OPERATE.get();
        if (retryOperate == null) {
            return;// 非重试线程则直接返回
        }
        // 从重试线程填充业务类型
        if (StringUtils.isBlank(currBusinessOperate.getBusinessTypeCode())) {
            currBusinessOperate.setBusinessType(retryOperate.getBusinessType());
            currBusinessOperate.setBusinessTypeCode(retryOperate.getBusinessTypeCode());
        }
        // 从重试线程填充业务KEY
        if (StringUtils.isBlank(currBusinessOperate.getBusinessKey())) {
            currBusinessOperate.setBusinessKey(retryOperate.getBusinessKey());
        }
    }

    /**
     * 处理Business属性
     *
     * @param currBusinessOperate
     * @param prevBusinessOperate
     */
    private void handleBusinessAttr(BusinessOperate currBusinessOperate, Optional<BusinessOperate> prevBusinessOperate) {
        String businessTypeCode = currBusinessOperate.getBusinessTypeCode();
        if (StringUtils.isBlank(businessTypeCode)) {
            // 从上一步获取业务类型编码
            prevBusinessOperate.ifPresent(prev -> currBusinessOperate.setBusinessTypeCode(prev.getBusinessTypeCode()));
            businessTypeCode = currBusinessOperate.getBusinessTypeCode();
        }
        if (StringUtils.isNotEmpty(businessTypeCode)) {
            // 从数据字典转换为业务类型
            DataDictCache.getInstance().getDictItem("BUSINESS_OPERATION_TYPE", businessTypeCode)
                    .ifPresent(dataDict -> currBusinessOperate.setBusinessType(dataDict.getValue()));
        }
        prevBusinessOperate.ifPresent(prev -> {
            final String currBusinessTypeCode = currBusinessOperate.getBusinessTypeCode();
            if (StringUtils.isNotBlank(currBusinessTypeCode)) {
                prev.setBusinessTypeCode(currBusinessTypeCode);
                prev.setBusinessType(currBusinessOperate.getBusinessType());
            }
            final String businessKey = currBusinessOperate.getBusinessKey();
            if (StringUtils.isBlank(businessKey)) {
                currBusinessOperate.setBusinessKey(prev.getBusinessKey());// 从上一步获取业务类型编码
            } else {
                prev.setBusinessKey(businessKey);
            }
        });
        // 如果执行唯一标识为空，则赋值为业务操作类型编码$操作类型
        if (StringUtils.isBlank(currBusinessOperate.getInvokeIdentifier()) && StringUtils.isNotBlank(currBusinessOperate.getBusinessTypeCode())) {
            currBusinessOperate.setInvokeIdentifier(currBusinessOperate.getBusinessTypeCode() + "$" + currBusinessOperate.getOperateType());
        }
    }

    /**
     * 处理Operation属性
     *
     * @param spELContext
     * @param aopOperation
     * @param returnValue
     * @param currBusinessOperate
     * @param prevBusinessOperate
     */
    private void handleOperationAttr(SpELContext spELContext, AopOperation aopOperation, Object returnValue, BusinessOperate currBusinessOperate, Optional<BusinessOperate> prevBusinessOperate) {
        // setOperateState
        if (currBusinessOperate.getOperateState() == null) {
            final String spelOperateState = aopOperation.operateState();
            if (StringUtils.isBlank(spelOperateState)) {
                if (returnValue instanceof GlobalResult) {
                    currBusinessOperate.setOperateState(((GlobalResult) returnValue).getCode());
                }
            } else {
                currBusinessOperate.setOperateState(spELContext.getValue(spelOperateState, Integer.class));
            }
        }
        // setOperateRemark
        if (currBusinessOperate.getOperateRemark() == null) {
            final String spelOperateRemark = aopOperation.operateRemark();
            if (StringUtils.isBlank(spelOperateRemark)) {
                if (returnValue instanceof GlobalResult) {
                    currBusinessOperate.setOperateRemark(((GlobalResult) returnValue).getMessage());
                }
            } else {
                currBusinessOperate.setOperateRemark(spELContext.getValue(spelOperateRemark, String.class));
            }
        }
        // setReturnValue
        if (currBusinessOperate.getReturnValue() == null) {
            if (returnValue != null) {
                String spelReturnValue = aopOperation.returnValue();
                if (StringUtils.isBlank(spelReturnValue)) {
                    if (returnValue instanceof GlobalResult) {
                        spelReturnValue = "#{#r.data}";
                    }
                    if (returnValue instanceof String) {
                        currBusinessOperate.setReturnValue((String) returnValue);
                    }
                }
                if (StringUtils.isNotBlank(spelReturnValue)) {
                    returnValue = spELContext.getValue(spelReturnValue, Object.class);
                }
                currBusinessOperate.setReturnValue(JsonUtils.toJson(returnValue));
            }
        }
    }

    /**
     * 获取当前的操作对象
     *
     * @return
     */
    public static Optional<BusinessOperate> getOp() {
        if (THREAD_LOCAL.get().isEmpty()) {
            return Optional.empty();
        }
        return Optional.of(THREAD_LOCAL.get().peek());
    }

    public static void wrapOp(GlobalResult globalResult) {
        getOp().ifPresent(op -> wrapOp(op, globalResult));
    }

    public static void wrapOp(BusinessOperate businessOperate, GlobalResult globalResult) {
        // setOperateState
        if (businessOperate.getOperateState() == null) {
            businessOperate.setOperateState(globalResult.getCode());
        }
        // setOperateRemark
        if (businessOperate.getOperateRemark() == null) {
            businessOperate.setOperateRemark(globalResult.getMessage());
        }
        // setReturnValue
        if (businessOperate.getReturnValue() == null) {
            ReflectUtils.fieldValue(globalResult, "data", null)
                    .ifPresent(d -> businessOperate.setReturnValue(JsonUtils.toJson(d)));
        }
    }

    /**
     * 按ICode持久化操作对象
     *
     * @param iCode
     */
    public static void putOp(ICode iCode) {
        putOp(iCode.getCode(), iCode.getMessage());
    }

    /**
     * 按操作备注持久化操作对象
     *
     * @param operateRemark
     */
    public static void putOp(String operateRemark) {
        putOp(ResultCode.SUCCESS.getCode(), operateRemark);
    }

    /**
     * 按操作码+操作备注持久化操作对象
     *
     * @param operateState
     * @param operateRemark
     */
    public static void putOp(int operateState, String operateRemark) {
        final BusinessOperate businessOperate = getOp()
                .map(op -> BeanUtils.copy(op, BusinessOperate.class)).orElseGet(BusinessOperate::new);
        businessOperate.setOperateRemark(operateRemark);
        businessOperate.setOperateState(operateState);
        businessOperate.setOperateTime(LocalDateTime.now());
        businessOperate.setOperateType(OperateType.PERSIST.getType());
        putOp(businessOperate);
    }

    /**
     * 添加操作对象到存储队列
     *
     * @param businessOperate
     */
    public static void putOp(BusinessOperate businessOperate) {
        businessOperate.setKeyword(StringUtils.maxLength(businessOperate.getKeyword(), 1000 - 3));
        businessOperate.setBusinessKey(StringUtils.maxLength(businessOperate.getBusinessKey(), 255 - 3));
        businessOperate.setOperateRemark(StringUtils.maxLength(businessOperate.getOperateRemark(), 1000 - 3));
        SELF.setChanged();
        SELF.notifyObservers(businessOperate);
    }

}
