package com.xforceplus.delivery.cloud.tax.api.logging;

import com.xforceplus.delivery.cloud.common.util.AspectUtils;
import com.xforceplus.delivery.cloud.common.util.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.slf4j.Logger;
import org.slf4j.Marker;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.expression.spel.standard.SpelExpressionParser;
import org.springframework.stereotype.Component;

import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * @vlog: 高于生活，源于生活
 * @Desc: TODO
 * @Author: Hanyongjie
 * @CreateDate: 2020-08-17 11:39
 * @Version: 1.0
 */
@Slf4j
@Aspect
@Component
@Order(LogMarkerAspect.ORDERED_PRECEDENCE)
public class LogMarkerAspect {

    public static final int ORDERED_PRECEDENCE = Ordered.LOWEST_PRECEDENCE - 30;

    private ConcurrentMap<String, Marker> markerMaps = new ConcurrentHashMap<>();

    @Around("@annotation(logMarker)")
    public Object around(ProceedingJoinPoint joinPoint, LogMarker logMarker) throws Throwable {
        Object returnValue = null;
        Instant now = Instant.now();
        Logger logger = AspectUtils.getLogger(joinPoint).orElse(log);
        final String format = logMarker.format();
        final Object[] args = joinPoint.getArgs();
        try {
            logger.debug("LogMarker starting - {}", format);
            if (log.isDebugEnabled()) {
                logger.debug("LogMarker argument - {}", JsonUtils.toJson(args));
            }
            returnValue = joinPoint.proceed();
            if (log.isDebugEnabled()) {
                logger.debug("LogMarker finished - {}", JsonUtils.toJson(returnValue));
            }
        } catch (Throwable e) {
            logger.debug("LogMarker throwable - {}", format, e);
            throw e;
        } finally {
            long millis = Duration.between(now, Instant.now()).toMillis();
            logger.debug("LogMarker completed elapsed {} ms", millis);
            this.handleLogMarker(logger, logMarker, args, returnValue);
        }
        return returnValue;
    }

    /**
     * 处理LogMarker
     *
     * @param logger
     * @param logMarker
     * @param args
     * @param returnValue
     */
    private void handleLogMarker(Logger logger, LogMarker logMarker, Object[] args, Object returnValue) {
        final String markerSpEL = logMarker.marker();
        final Marker marker = this.markerMaps.computeIfAbsent(markerSpEL, this::executeExpr);
        Object[] dest = new Object[args.length + 1];
        for (int i = 0; i < dest.length; i++) {
            if (i < args.length) {
                dest[i] = args[i];
            } else {
                dest[i] = returnValue;
            }
        }
        logger.info(marker, logMarker.format(), dest);
    }

    private Marker executeExpr(String markerSpEL) {
        return new SpelExpressionParser().parseExpression(markerSpEL).getValue(Marker.class);
    }

}
