package com.xforceplus.delivery.cloud.tax.api.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.xforceplus.delivery.cloud.common.api.GlobalResult;
import com.xforceplus.delivery.cloud.common.api.ViewResult;
import com.xforceplus.delivery.cloud.common.util.DateUtils;
import com.xforceplus.delivery.cloud.common.util.StringUtils;
import com.xforceplus.delivery.cloud.tax.api.constants.BusinessConstants;
import com.xforceplus.delivery.cloud.tax.api.domain.BusinessOperateVO;
import com.xforceplus.delivery.cloud.tax.api.entity.BusinessOperate;
import com.xforceplus.delivery.cloud.tax.api.mapper.BusinessOperateMapper;
import com.xforceplus.delivery.cloud.tax.api.service.IBusinessOperateService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.Optional;

/**
 * <p>
 * 服务实现类
 * </p>
 *
 * @author Hanyongjie
 * @since 2020-09-03
 */
@Slf4j
@Service
public class BusinessOperateServiceImpl extends ServiceImpl<BusinessOperateMapper, BusinessOperate> implements IBusinessOperateService {
    /**
     * 新事务执行Wrapper
     *
     * @param lambdaQuery
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public boolean removeByWrapper(LambdaQueryWrapper<BusinessOperate> lambdaQuery) {
        return super.remove(lambdaQuery);
    }

    /**
     * 通过ID/业务类型/关键字等的组合删除(ID优先)
     *
     * @param businessOperate
     * @return
     */
    @Override
    public GlobalResult removeByExample(BusinessOperate businessOperate) {
        boolean affectedRows;
        final Long businessOperateId = businessOperate.getId();
        if (businessOperateId == null || businessOperateId == 0) {
            affectedRows = this.doBatchRemove(businessOperate);
        } else {
            affectedRows = super.removeById(businessOperateId);
        }
        return ViewResult.success(affectedRows);
    }

    /**
     * 批量删除
     *
     * @param businessOperate
     * @return
     */
    private boolean doBatchRemove(BusinessOperate businessOperate) {
        boolean affectedRows;
        String businessType = businessOperate.getBusinessType();
        String businessKey = businessOperate.getBusinessKey();
        log.debug("operate record clean starting: [{}]{}", businessType, businessKey);
        final LambdaQueryWrapper<BusinessOperate> lambdaQuery = Wrappers.lambdaQuery(BusinessOperate.class);
        lambdaQuery.eq(BusinessOperate::getBusinessType, businessType);
        // businessKey
        if (StringUtils.isNotBlank(businessKey)) {
            if (businessKey.contains("%")) {
                lambdaQuery.like(BusinessOperate::getBusinessKey, businessKey);
            } else {
                lambdaQuery.eq(BusinessOperate::getBusinessKey, businessKey);
            }
        }
        // keyword
        Optional.ofNullable(StringUtils.emptyToNull(businessOperate.getKeyword())).ifPresent(keyword -> {
            if (keyword.contains("%")) {
                lambdaQuery.like(BusinessOperate::getKeyword, keyword);
            } else {
                lambdaQuery.eq(BusinessOperate::getKeyword, keyword);
            }
        });
        //
        Optional.ofNullable(businessOperate.getOperateState()).ifPresent(s -> lambdaQuery.eq(BusinessOperate::getOperateState, s));
        Optional.ofNullable(businessOperate.getOperateType()).ifPresent(s -> lambdaQuery.eq(BusinessOperate::getOperateType, s));
        // cleanTime
        if (businessOperate instanceof BusinessOperateVO) {
            final String cleanTime = ((BusinessOperateVO) businessOperate).getCleanTime();
            if (StringUtils.isNotBlank(cleanTime)) {
                final String[] aCleanTime = cleanTime.split("~");
                if (aCleanTime.length == 2) {
                    lambdaQuery.between(BusinessOperate::getOperateTime, DateUtils.toLocalDateTime(aCleanTime[0]), DateUtils.toLocalDateTime(aCleanTime[1]));
                }
            }
        }
        // 按批删除避免长事务
        lambdaQuery.last(" limit " + BusinessConstants.REMOVE_BATCH_SIZE);
        do {
            log.debug("operate record clearing: [{}]{}", businessType, businessKey);
            affectedRows = super.remove(lambdaQuery);
            log.debug("operate record deleted affected: [{}]{} -> {}", businessType, businessKey, affectedRows);
        } while (affectedRows);
        log.debug("operate record clean finished: [{}]{}", businessType, businessKey);
        return true;
    }

}
