package com.xforceplus.delivery.cloud.tax.api.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.xforceplus.delivery.cloud.common.api.GlobalResult;
import com.xforceplus.delivery.cloud.common.api.ViewResult;
import com.xforceplus.delivery.cloud.common.util.DateUtils;
import com.xforceplus.delivery.cloud.common.util.SpringUtils;
import com.xforceplus.delivery.cloud.common.util.StringUtils;
import com.xforceplus.delivery.cloud.tax.api.constants.BusinessConstants;
import com.xforceplus.delivery.cloud.tax.api.domain.DispatchQueueVO;
import com.xforceplus.delivery.cloud.tax.api.entity.DispatchRetryQueue;
import com.xforceplus.delivery.cloud.tax.api.mapper.DispatchRetryQueueMapper;
import com.xforceplus.delivery.cloud.tax.api.service.IDispatchRetryQueueService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Isolation;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

/**
 * <p>
 * 分发重试队列 服务实现类
 * </p>
 *
 * @author Hanyongjie
 * @since 2020-07-26
 */
@Slf4j
@Service
public class DispatchRetryQueueServiceImpl extends ServiceImpl<DispatchRetryQueueMapper, DispatchRetryQueue> implements IDispatchRetryQueueService {

    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public void updateCounter(DispatchRetryQueue dispatchRetryQueue) {
        super.lambdaUpdate().set(DispatchRetryQueue::getCounter, dispatchRetryQueue.getCounter())
                .eq(DispatchRetryQueue::getId, dispatchRetryQueue.getId())
                .update();
    }

    /**
     * 新事务执行Wrapper
     *
     * @param lambdaQuery
     */
    @Override
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public boolean removeByWrapper(LambdaQueryWrapper<DispatchRetryQueue> lambdaQuery) {
        return super.remove(lambdaQuery);
    }

    /**
     * 通过ID/服务/事件/关键字的组合删除(ID优先)
     *
     * @param dispatchRetryQueue
     * @return
     */
    @Override
    @Transactional(isolation = Isolation.READ_COMMITTED)
    public GlobalResult removeByExample(DispatchRetryQueue dispatchRetryQueue) {
        boolean affectedRows;
        final Long retryQueueId = dispatchRetryQueue.getId();
        if (retryQueueId == null || retryQueueId == 0) {
            affectedRows = this.doBatchRemove(dispatchRetryQueue);
        } else {
            affectedRows = super.removeById(retryQueueId);
        }
        return ViewResult.success(affectedRows);
    }

    /**
     * 批量删除
     *
     * @param dispatchRetryQueue
     * @return
     */
    private boolean doBatchRemove(DispatchRetryQueue dispatchRetryQueue) {
        boolean affectedRows;
        String eventName = dispatchRetryQueue.getEvtname();
        String keyword = dispatchRetryQueue.getKeyword();
        log.debug("dispatch queue clean starting: [{}]{}", eventName, keyword);
        final LambdaQueryWrapper<DispatchRetryQueue> lambdaQuery = Wrappers.lambdaQuery(DispatchRetryQueue.class);
        lambdaQuery.eq(DispatchRetryQueue::getEvtname, eventName);
        lambdaQuery.eq(DispatchRetryQueue::getSvc, StringUtils.blankToDefault(dispatchRetryQueue.getSvc(), SpringUtils.getAppName()));
        if (StringUtils.isNotBlank(keyword)) {
            if (keyword.contains("%")) {
                lambdaQuery.like(DispatchRetryQueue::getKeyword, keyword);
            } else {
                lambdaQuery.eq(DispatchRetryQueue::getKeyword, keyword);
            }
        }
        if (dispatchRetryQueue instanceof DispatchQueueVO) {
            final String cleanTime = ((DispatchQueueVO) dispatchRetryQueue).getCleanTime();
            if (StringUtils.isNotBlank(cleanTime)) {
                final String[] aCleanTime = cleanTime.split("~");
                if (aCleanTime.length == 2) {
                    lambdaQuery.between(DispatchRetryQueue::getCreateTime, DateUtils.toLocalDateTime(aCleanTime[0]), DateUtils.toLocalDateTime(aCleanTime[1]));
                }
            }
        }
        // 按批删除避免长事务
        lambdaQuery.last(" limit " + BusinessConstants.REMOVE_BATCH_SIZE);
        do {
            log.debug("dispatch queue clearing: [{}]{}", eventName, keyword);
            affectedRows = super.remove(lambdaQuery);
            log.debug("dispatch queue deleted affected: [{}]{} -> {}", eventName, keyword, affectedRows);
        } while (affectedRows);
        log.debug("dispatch queue clean finished: [{}]{}", eventName, keyword);
        return true;
    }

}
