package com.xforceplus.delivery.cloud.tax.api.support;

import com.google.common.collect.Sets;
import com.xforceplus.delivery.cloud.common.api.GlobalResult;
import com.xforceplus.delivery.cloud.common.api.ResultCode;
import com.xforceplus.delivery.cloud.common.util.SpringUtils;
import com.xforceplus.delivery.cloud.tax.api.entity.DispatchRetryQueue;
import com.xforceplus.delivery.cloud.tax.api.service.IDispatchRetryQueueService;
import lombok.Getter;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.stereotype.Component;

import java.util.Set;

/**
 * @vlog: 高于生活，源于生活
 * @Desc: DataDispatchInvoker
 * @Author: Hanyongjie
 * @CreateDate: 2020-07-26 22:52
 * @Version: 1.0
 */
@Slf4j
@Component
@ConfigurationProperties(prefix = "delivery.cloud.dispatch.retry")
public class DataDispatchExecutor {

    @Setter
    @Getter
    private Set<Integer> excludes;

    @Setter
    @Getter
    private Set<Integer> includes;

    @Autowired
    private DataDispatchRetryer dataDispatchRetryer;

    @Autowired
    private IDispatchRetryQueueService iDispatchRetryQueueService;

    @Autowired
    private DataDispatchBeanPostProcessor dataDispatchBeanPostProcessor;

    public DataDispatchExecutor() {
        this.excludes = Sets.newHashSet(ResultCode.SUCCESS.getCode());
        this.includes = Sets.newHashSet(ResultCode.RETRY_WITH.getCode());
    }

    /**
     * 分发数据，返回成功或者参数检验失败则删除任务
     *
     * @param dispatchQueue
     * @param jobParam
     * @return
     */
    public GlobalResult doDispatch(DispatchRetryQueue dispatchQueue, DataDispatchJobParam jobParam) {
        final String dispatchArgs = dispatchQueue.getArgs();
        final GlobalResult globalResult = this.dataDispatchBeanPostProcessor.invokeForJson(dispatchQueue.getEvtname(), dispatchArgs);
        // 处理成功或者校验不满足的则删除任务
        if (this.isRetryWith(globalResult)) {
            final int counter = dispatchQueue.getCounter() + 1;
            boolean isReachedMaxCounter = jobParam.getMaxCounter() > 0 && counter >= jobParam.getMaxCounter();
            if (isReachedMaxCounter) {
                log.debug("dispatch queue reached the max count({}): {}", counter, dispatchQueue);
                this.iDispatchRetryQueueService.removeById(dispatchQueue.getId());
            } else {
                dispatchQueue.setCounter(counter);
                log.debug("dispatch queue increasing: {}", dispatchQueue);
                this.iDispatchRetryQueueService.updateCounter(dispatchQueue);
            }
        } else {
            log.debug("dispatch queue clearing: {}", dispatchQueue);
            this.iDispatchRetryQueueService.removeById(dispatchQueue.getId());
        }
        return globalResult;
    }

    /**
     * 分发数据
     *
     * @param eventName
     * @param args
     */
    public void doDispatch(String eventName, String keyword, Object... args) {
        final GlobalResult globalResult = this.dataDispatchBeanPostProcessor.invokeMethod(eventName, args);
        if (this.isRetryWith(globalResult)) {
            log.debug("dispatch retry add [{}]{}", eventName, keyword);
            this.dataDispatchRetryer.dispatch(eventName, keyword, args);
        } else {
            DispatchRetryQueue dispatchRetryQueue = new DispatchRetryQueue();
            dispatchRetryQueue.setEvtname(eventName);
            dispatchRetryQueue.setKeyword(keyword);
            dispatchRetryQueue.setSvc(SpringUtils.getAppName());
            log.debug("dispatch retry del [{}]{}", eventName, keyword);
            this.iDispatchRetryQueueService.removeByExample(dispatchRetryQueue);
        }
    }

    /**
     * 检验是否重试
     *
     * @param globalResult
     * @return
     */
    private boolean isRetryWith(GlobalResult globalResult) {
        Integer code = globalResult.getCode();
        log.debug("dispatch retry exclude [{}]{}", code, this.excludes);
        if (this.excludes.contains(code)) {
            return false;
        }
        log.debug("dispatch retry include [{}]{}", code, this.includes);
        return this.includes.isEmpty() || this.includes.contains(code);
    }

}