package com.xforceplus.delivery.cloud.tax.api.support;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.xforceplus.delivery.cloud.common.api.GlobalResult;
import com.xforceplus.delivery.cloud.common.util.ExceptionUtils;
import com.xforceplus.delivery.cloud.common.util.JsonUtils;
import com.xforceplus.delivery.cloud.common.util.SpringUtils;
import com.xforceplus.delivery.cloud.common.util.TraceUtils;
import com.xforceplus.delivery.cloud.tax.api.entity.DispatchRetryQueue;
import com.xforceplus.delivery.cloud.tax.api.service.IDispatchRetryQueueService;
import com.xxl.job.core.biz.model.ReturnT;
import com.xxl.job.core.handler.IJobHandler;
import com.xxl.job.core.handler.annotation.XxlJob;
import com.xxl.job.core.log.XxlJobLogger;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Comparator;
import java.util.List;

/**
 * @vlog: 高于生活，源于生活
 * @Desc: TODO
 * @Author: Hanyongjie
 * @CreateDate: 2020-07-26 22:29
 * @Version: 1.0
 */
@Slf4j
@Component
public class DataDispatchJobHandler {

    @Autowired
    private DataDispatchExecutor dataDispatchExecutor;

    @Autowired
    private IDispatchRetryQueueService iDispatchRetryQueueService;

    @XxlJob("dataDispatchJobHandler")
    public ReturnT<String> dataDispatchJobHandler(String jsonArgs) throws Exception {
        log.debug("XXL-JOB, data dispatch param - {}", jsonArgs);
        final DataDispatchJobParam jobParam = JsonUtils.fromJson(jsonArgs, DataDispatchJobParam.class);
        Long id = 0L;
        int pageNo = 1;
        final int maxPageCounter = jobParam.getPageCounter();
        do {
            id = this.doBatchHandle(jobParam, SpringUtils.getAppName(), id);
        } while (id != null && (maxPageCounter == -1 || pageNo++ < maxPageCounter));
        return IJobHandler.SUCCESS;
    }

    /**
     * 处理一批数据
     *
     * @param jobParam
     * @param appName
     * @param id
     * @return
     */
    private Long doBatchHandle(DataDispatchJobParam jobParam, String appName, Long id) {
        final int perCounter = jobParam.getPerCounter();
        final Page<DispatchRetryQueue> dispatchRetryQueues = iDispatchRetryQueueService.lambdaQuery()
                .eq(DispatchRetryQueue::getEvtname, jobParam.getEvtName())
                .eq(DispatchRetryQueue::getSvc, appName)
                .gt(DispatchRetryQueue::getId, id)
                .page(new Page<>(1, perCounter));
        final List<DispatchRetryQueue> records = dispatchRetryQueues.getRecords();
        log.debug("XXL-JOB, data dispatch batch query - {}/{}({})", records.size(), dispatchRetryQueues.getTotal(), perCounter);
        records.forEach(queue -> this.doDispatch(queue, jobParam));
        if (records.size() < perCounter) {
            return null;
        }
        return records.stream().max(Comparator.comparing(DispatchRetryQueue::getId)).map(DispatchRetryQueue::getId).orElse(null);
    }

    public void doDispatch(DispatchRetryQueue dispatchQueue, DataDispatchJobParam jobParam) {
        final String evtname = dispatchQueue.getEvtname();
        final String keyword = dispatchQueue.getKeyword();
        final String traceId = dispatchQueue.getTraceId();
        TraceUtils.setMdcTraceId(traceId);
        if (Thread.currentThread().isInterrupted()) {
            log.debug("XXL-JOB, [{}][{}]dispatch interrupted - {}", traceId, evtname, keyword);
            XxlJobLogger.log("[{}][{}]dispatch interrupted - {}", traceId, evtname, keyword);
            return;
        }
        try {
            XxlJobLogger.log("[{}][{}]dispatch starting - {}", traceId, evtname, keyword);
            final GlobalResult globalResult = this.dataDispatchExecutor.doDispatch(dispatchQueue, jobParam);
            XxlJobLogger.log("[{}][{}]dispatch finished - {}", traceId, evtname, JsonUtils.toJson(globalResult));
        } catch (Exception e) {
            log.warn("XXL-JOB, dispatch exception - {}", evtname + ":" + keyword, e);
            XxlJobLogger.log("[{}][{}]dispatch exception - {}", traceId, evtname, ExceptionUtils.toStringRootCauseStackTrace(e));
        } finally {
            TraceUtils.clsMdcTraceId();
        }
    }

}