package com.xforceplus.janus.flow.sys.entity;

import com.baomidou.mybatisplus.annotation.TableField;
import com.baomidou.mybatisplus.annotation.TableName;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.xforceplus.apollo.utils.JacksonUtil;
import com.xforceplus.janus.commons.dto.BaseEntity;
import com.xforceplus.janus.commons.util.UniqIdUtils;
import com.xforceplus.janus.flow.core.entity.Flow;
import com.xforceplus.janus.flow.core.entity.Node;
import com.xforceplus.janus.flow.core.entity.dto.*;
import com.xforceplus.janus.flow.core.enums.NodeTypeEnum;
import com.xforceplus.janus.flow.logic.model.FieldMappingDto;
import com.xforceplus.janus.flow.sys.entity.vo.BranchConditVO;
import com.xforceplus.janus.flow.sys.exception.NodeNotFoundException;

import lombok.Data;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @program: janus-flow->JanusFlow
 * @description: 数据流信息
 * @author: xuchuanhou
 * @create: 2020-06-06 15:08
 **/
@TableName("t_janus_flow")
@Data
@JsonIgnoreProperties(ignoreUnknown = true)
public class FlowEntity extends BaseEntity {

    public static final String STATUS_INIT = "1";
    public static final String STATUS_PUBLISH = "2";

    public static final String TYPE_FLOW_ACTION = "1";
    public static final String TYPE_FLOW_API = "2";
    public static final String TYPE_DATA_LINK_API = "3";
    public static final String TYPE_DATA_LINK_ACTION = "4";
    /**属地执行的数据流*/
    public static final String TYPE_FLOW_LOCAL = "5";

    @NotBlank(message = "项目编号不能为空")
    private String projectId;

    private String name;

    // 暂未放开
    @TableField(exist = false)
    private String version;
    /***
     * 数据流入口绑定接口(从模板流程拷贝来的则为模板接口ID)
     */
//    @NotBlank(message = "绑定接口不能为空")
    private String boundApi;
    //
//    @NotBlank(message = "授权绑定编排执行接口不能为空")
    private String boundAction;

    /**
     * 1:创建 2:发布共享
     */
    private String status = STATUS_INIT;

    /**流程类型*/
    private String type=TYPE_FLOW_ACTION;

    @Valid
    @NotNull(message = "流程节点不能为空")
    @TableField(exist = false)
    private List<NodeEntity> nodeDataArray;

    @NotNull(message = "节点关系不能为空")
    @TableField(exist = false)
    private List<NodeMappingEntity> linkDataArray;

    /**
     * 将前台 key 赋值
     */
    public void fromVo2Entity() {
        if (CollectionUtils.isEmpty(nodeDataArray)) {
            return;
        }

        if (StringUtils.isBlank(this.getId())) {
            this.setId(UniqIdUtils.getInstance().getUniqID());
        }

        Map<String, NodeEntity> succNodeMap = new HashMap<>();
        for (NodeEntity node : nodeDataArray) {
            if (StringUtils.isBlank(node.getId()) && StringUtils.isNotBlank(node.getKey())) {
                node.setId(UniqIdUtils.getInstance().getUniqID());
            }
            node.setFlowId(this.getId());
        }
        for (NodeEntity node : nodeDataArray) {
            for (NodeMappingEntity nodeMapping : linkDataArray) {
                if (StringUtils.isBlank(nodeMapping.getLastNodeId()) && nodeMapping.getFrom().equals(node.getKey())) {
                    nodeMapping.setLastNodeId(node.getId());
                }

                if (StringUtils.isBlank(nodeMapping.getNextNodeId()) && nodeMapping.getTo().equals(node.getKey())) {
                    nodeMapping.setNextNodeId(node.getId());
                }

                nodeMapping.setFlowId(this.getId());

                if (node.getId().equals(nodeMapping.getNextNodeId())
                        || node.getId().equals(nodeMapping.getLastNodeId())) {
                    succNodeMap.put(node.getId(), node);
                }
            }

            if (MapUtils.isNotEmpty(node.getExtendDtoPropeties())) {
                List<NodeConfigEntity> nodeConfigs = new ArrayList<>(node.getExtendDtoPropeties().size());
                node.setNodeConfigs(nodeConfigs);

                NodeTypeEnum nodeTypeEnum = NodeTypeEnum.getByCode(node.getNodeType());
                switch (nodeTypeEnum) {
                    case BRANCH:

                        List<BranchConditVO> branchConfigs = JacksonUtil.getInstance().fromJsonToList(JacksonUtil.getInstance().toJson(node.getExtendDtoPropeties().get("branchs")), BranchConditVO.class);
                        for (BranchConditVO branchConfig : branchConfigs) {
                            NodeEntity nextNode = findByNodeKey(branchConfig.getNextNodeKey());
                            if (nextNode == null) {
                                throw new NodeNotFoundException(branchConfig.getNextNodeKey() + "下个节点未找到");
                            }
                            branchConfig.setNextNodeId(nextNode.getId());
                        }
                        node.getExtendDtoPropeties().put("branchs", branchConfigs);
                        break;

                    default:
                        break;
                }
                //only use nodeConfig
                node.setNodeConfig(JacksonUtil.getInstance().toJson(node.getExtendDtoPropeties()));

            }
        }

        this.nodeDataArray = new ArrayList<>(succNodeMap.values());
    }

    private NodeEntity findByNodeKey(String key) {
        for (NodeEntity node : nodeDataArray) {
            if (node.getKey().equals(key)) {
                return node;
            }
        }
        return null;
    }

    private String getNextNodeId(NodeEntity nodeEntity) {
        if (NodeTypeEnum.BRANCH.getCode().equals(nodeEntity.getNodeType())) {
            return null;
        }

        for (NodeMappingEntity nodeMap : linkDataArray) {
            if (nodeMap.getLastNodeId().equals(nodeEntity.getId())) {
                return nodeMap.getNextNodeId();
            }
        }
        return null;
    }

    public void fromEntity2Vo() {

        for (NodeEntity node : nodeDataArray) {
            node.setKey(node.getId());

            if (StringUtils.isNotBlank(node.getNodeConfig())) {
                node.setExtendDtoPropeties(JacksonUtil.getInstance().fromJson(node.getNodeConfig(), Map.class));
            }
            node.setNodeConfig(null);
            node.setNodeConfigs(null);
        }

        for (NodeMappingEntity nodeMapping : linkDataArray) {
            nodeMapping.setFrom(nodeMapping.getLastNodeId());
            nodeMapping.setTo(nodeMapping.getNextNodeId());
        }
    }

    /**
     * 转化到flow-server executor 中的dto
     *
     * @author xuchuanhou
     */
    public Flow fromEntity2Dto() {

        List<Node> nodes = new ArrayList<>();

        for (NodeEntity nodeEntity : nodeDataArray) {
            NodeTypeEnum nodeTypeEnum = NodeTypeEnum.getByCode(nodeEntity.getNodeType());

            Map<String, Object> nodeConfigMap = getNodeProperties(nodeEntity);

            Node node = null;

            switch (nodeTypeEnum) {
                case START:
                    node = new Node<NodeConfigStartDto>(nodeEntity.getId(), this.getId(), nodeEntity.getNodeName(),
                            nodeEntity.getNodeType());
                    node.setConfig(new NodeConfigStartDto());
                    break;
                case API:
                    NodeConfigApiDto apiDto = JacksonUtil.getInstance()
                            .fromJson(JacksonUtil.getInstance().toJson(nodeConfigMap), NodeConfigApiDto.class);
                    if (apiDto.getApiRetryDto() != null && apiDto.getApiRetryDto().getRetryTimes() == null) {
                        apiDto.getApiRetryDto().setRetryTimes(1);
                        apiDto.getApiRetryDto().setInterval(1000L);
                    }
                    node = new Node<NodeConfigApiDto>(nodeEntity.getId(), this.getId(), nodeEntity.getNodeName(),
                            nodeEntity.getNodeType());
                    if (StringUtils.isNotBlank(apiDto.getAction())) {
                        if (apiDto.getHeaderKeys() == null) {
                            apiDto.setHeaderKeys(new ArrayList<>());
                        }
                        apiDto.getHeaderKeys().add(new FieldMappingDto("action", null, apiDto.getAction()));
                        apiDto.getHeaderKeys()
                                .add(new FieldMappingDto("Authentication", null, apiDto.getAuthentication()));
                        apiDto.getHeaderKeys().add(new FieldMappingDto("rpcType", null, NodeConfigApiDto.RPC_TYPE_HTTP));
                        if (StringUtils.isNotBlank(apiDto.getUiaSign())) {
                            apiDto.getHeaderKeys().add(new FieldMappingDto("uiaSign", null, apiDto.getUiaSign()));
                        }
                    }

                    node.setConfig(apiDto);
                    break;
                case CMPL:
                    NodeConfigCmplDto cmplDto = JacksonUtil.getInstance()
                            .fromJson(JacksonUtil.getInstance().toJson(nodeConfigMap), NodeConfigCmplDto.class);

                    node = new Node<NodeConfigEtlDto>(nodeEntity.getId(), this.getId(), nodeEntity.getNodeName(),
                            nodeEntity.getNodeType());

                    node.setConfig(cmplDto);
                    break;
                case TRANSFORM:
                    NodeConfigEtlDto etDto = JacksonUtil.getInstance()
                            .fromJson(JacksonUtil.getInstance().toJson(nodeConfigMap), NodeConfigEtlDto.class);
                    node = new Node<NodeConfigEtlDto>(nodeEntity.getId(), this.getId(), nodeEntity.getNodeName(),
                            nodeEntity.getNodeType());
                    node.setConfig(etDto);
                    break;
                case EXTRACT:
                    NodeConfigExtractDto extractDto = JacksonUtil.getInstance()
                            .fromJson(JacksonUtil.getInstance().toJson(nodeConfigMap), NodeConfigExtractDto.class);
                    node = new Node<NodeConfigEtlDto>(nodeEntity.getId(), this.getId(), nodeEntity.getNodeName(),
                            nodeEntity.getNodeType());
                    node.setConfig(extractDto);
                    break;
                case BRANCH:
                    if (MapUtils.isNotEmpty(nodeConfigMap)) {
                        NodeConfigBranchDto branchDto =  JacksonUtil.getInstance().fromJson(
                                JacksonUtil.getInstance().toJson(nodeConfigMap), NodeConfigBranchDto.class);

                        node = new Node<ArrayList<BranchConditVO>>(nodeEntity.getId(), this.getId(),
                                nodeEntity.getNodeName(), nodeEntity.getNodeType());
                        node.setConfig(branchDto);
                    }
                    break;
                case BUILD_PARAM:
                    NodeConfigConcatDto concatDto = JacksonUtil.getInstance()
                            .fromJson(JacksonUtil.getInstance().toJson(nodeConfigMap), NodeConfigConcatDto.class);
                    node = new Node<NodeConfigConcatDto>(nodeEntity.getId(), this.getId(), nodeEntity.getNodeName(),
                            nodeEntity.getNodeType());
                    node.setConfig(concatDto);
                    break;
                case END:
                    node = new Node<NodeConfigEndDto>(nodeEntity.getId(), this.getId(), nodeEntity.getNodeName(),
                            nodeEntity.getNodeType());
                    if (MapUtils.isNotEmpty(nodeConfigMap)) {
                        NodeConfigEndDto endDto = JacksonUtil.getInstance()
                                .fromJson(JacksonUtil.getInstance().toJson(nodeConfigMap), NodeConfigEndDto.class);

                        node.setConfig(endDto);
                    }
                    break;
                default:
                    break;
            }
            node.setNextNodeId(getNextNodeId(nodeEntity));
            nodes.add(node);
        }

        Flow flow = new Flow(this.getId(), this.getName(), nodes);
        return flow;
    }


    private Map<String, Object> getNodeProperties(NodeEntity nodeEntity) {
        Map<String, Object> nodeConfigMap = new HashMap<>();
        if (StringUtils.isNotBlank(nodeEntity.getNodeConfig())) {
            nodeConfigMap = JacksonUtil.getInstance().fromJson(nodeEntity.getNodeConfig(), Map.class);
            return nodeConfigMap;
        }
        return nodeConfigMap;
    }


    /***
     * 对当前流程节点进行排序
     * 只能针对串联流程，分支流程无法操作
     * @author xuchuanhou
     * @return
     */
    public  List<NodeEntity> sortNode() {
        List<NodeEntity> sortedNodeList = new ArrayList<>();
        Map<String, NodeEntity> nodeIdMap = this.nodeDataArray.stream().collect(Collectors.toMap(NodeEntity::getId, Function.identity(), (v1, v2) -> v1));
        NodeMappingEntity rootMap = getRootNode(this.linkDataArray);
        sortedNodeList.add(nodeIdMap.get(rootMap.getLastNodeId()));
        sortedNodeList.add(nodeIdMap.get(rootMap.getNextNodeId()));
        while ((rootMap = getNextNodeMap(rootMap, this.linkDataArray)) != null) {
            sortedNodeList.add(nodeIdMap.get(rootMap.getNextNodeId()));
        }
        return sortedNodeList;
    }

    private static NodeMappingEntity getRootNode(List<NodeMappingEntity> nmList) {
        NodeMappingEntity root = null;
        for (int i = 0; i < nmList.size(); i++) {
            if (i == 0) {
                root = nmList.get(0);
            } else if (nmList.get(i).getNextNodeId().equals(root.getLastNodeId())) {
                root = nmList.get(i);
            }
        }
        return root;
    }

    private static NodeMappingEntity getNextNodeMap(NodeMappingEntity lastNM, List<NodeMappingEntity> nmList) {
        for (NodeMappingEntity nm : nmList) {
            if (nm.getLastNodeId().equals(lastNM.getNextNodeId())) {
                return nm;
            }
        }
        return null;
    }

}
