package com.xforceplus.janus.flow.sys.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.xforceplus.janus.commons.dto.R;
import com.xforceplus.janus.commons.exception.RRException;
import com.xforceplus.janus.commons.util.DateUtils;
import com.xforceplus.janus.commons.validator.ValidatorUtils;
import com.xforceplus.janus.flow.sys.dao.FsFlowEntityDao;
import com.xforceplus.janus.flow.sys.entity.FlowEntity;
import com.xforceplus.janus.flow.sys.entity.NodeMappingEntity;
import com.xforceplus.janus.flow.sys.param.FlowQueryParam;
import com.xforceplus.janus.flow.sys.service.FlowService;
import com.xforceplus.janus.flow.sys.service.NodeMappingService;
import com.xforceplus.janus.flow.sys.service.NodeService;
import com.xforceplus.janus.flow.sys.util.Constant;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

import lombok.extern.slf4j.Slf4j;

/**
 * @program: gateway-ops->DataFlowServiceImpl
 * @description:
 * @author: xuchuanhou
 * @create: 2020-06-06 17:31
 **/
@Slf4j
@Service("flowServerService")
@CacheConfig(cacheNames = "flow")
public class FsFlowServiceImpl extends ServiceImpl<FsFlowEntityDao, FlowEntity> implements FlowService {

    @Autowired
    private NodeService nodeService;

    @Autowired
    private NodeMappingService nodeMappingService;


    @Transactional
    @Override
    public R saveDataFlow(FlowEntity dataFlow) throws Exception {
        dataFlow.fromVo2Entity();
        ValidatorUtils.validateEntity(dataFlow);

        if (!FlowEntity.TYPE_FLOW_LOCAL_TASK.equals(dataFlow.getType())
                &&!FlowEntity.TYPE_FLOW_LOCAL_MQ.equals(dataFlow.getType())
                &&StringUtils.isBlank(dataFlow.getBoundAction())
                && StringUtils.isBlank(dataFlow.getBoundApi())) {
            throw new RRException("绑定接口或者授权action不能同时为空");
        }
        FlowEntity existDataFlow = null;
        if (StringUtils.isNotBlank(dataFlow.getBoundAction())) {
            existDataFlow = findByAction(dataFlow.getBoundAction(), dataFlow.getType());
        }

        if (existDataFlow == null && StringUtils.isNotBlank(dataFlow.getBoundApi())) {
            existDataFlow = this.findOneFlow(dataFlow.getProjectId(), null, dataFlow.getBoundApi(), dataFlow.getType());
        }

        if (existDataFlow != null && !existDataFlow.getId().equals(dataFlow.getId())) {
            return R.error("action服务编排已经存在!");
        }
        this.saveOrUpdate(dataFlow);
        nodeService.saveOrUpdateBatchNode(dataFlow.getNodeDataArray());

        nodeMappingService.deleteLogicByFlowId(dataFlow.getId());
        if (CollectionUtils.isNotEmpty(dataFlow.getLinkDataArray())) {
            nodeMappingService.saveOrUpdateBatch(dataFlow.getLinkDataArray());
        }

        return R.ok();
    }

    @Override
    public FlowEntity richDataByBoundApiId(String boundApi) {

        FlowEntity dataFlow = findByBoundApiId(boundApi, null);
        if (dataFlow == null) {
            return null;
        }
        this.extendFlow(dataFlow);
        return dataFlow;
    }

    private void extendFlow(FlowEntity dataFlow) {
        dataFlow.setNodeDataArray(nodeService.listRichByFlowId(dataFlow.getId()));
        if (CollectionUtils.isNotEmpty(dataFlow.getNodeDataArray())) {
            List<NodeMappingEntity> nodeMappingEntities = nodeMappingService.ListRichByFlowId(dataFlow.getId());
            List<String> ids =
                    dataFlow.getNodeDataArray().stream().map(map -> map.getId()).collect(Collectors.toList());

            List<NodeMappingEntity> nodeMapList = nodeMappingEntities.stream().filter(nodeMap -> {
                if (StringUtils.isNotBlank(nodeMap.getNextNodeId()) && !ids.contains(nodeMap.getNextNodeId())) {
                    return false;
                }
                if (StringUtils.isNotBlank(nodeMap.getLastNodeId()) && !ids.contains(nodeMap.getLastNodeId())) {
                    return false;
                }
                return true;
            }).collect(Collectors.toList());

            dataFlow.setLinkDataArray(nodeMapList);

        }
    }


    @Override
    public FlowEntity richDataByBoundAction(String boundAction) {
        FlowEntity dataFlow = findByAction(boundAction, null);
        if (dataFlow == null) {
            return null;
        }

        this.extendFlow(dataFlow);
        return dataFlow;
    }

    @Override
    public FlowEntity richDataById(String flowId) {
        FlowEntity dataFlow = this.getById(flowId);
        if (dataFlow == null) {
            return null;
        }

        this.extendFlow(dataFlow);
        return dataFlow;
    }

    private FlowEntity findByBoundApiId(String apiId, String type) {
        return this
                .getOne(new QueryWrapper<FlowEntity>()
                        .eq("bound_api", apiId)
                        .eq(StringUtils.isNotBlank(type), "type", type)
                        .eq(Constant.IS_VALID, Constant.IS_VALIDED));
    }

    @Override
    public FlowEntity findByAction(String action, String type) {
        FlowEntity flowEntity = this
                .getOne(new QueryWrapper<FlowEntity>()
                        .eq("bound_action", action)
                        .eq(StringUtils.isNotBlank(type), "type", type)
                        .eq(Constant.IS_VALID, Constant.IS_VALIDED));
        return flowEntity;
    }

    @Override
    public FlowEntity findByApiId(String projectId, String apiId) {

        return this.getOne(new QueryWrapper<FlowEntity>().eq("bound_api", apiId)
                .eq("project_id", projectId)
                .eq(Constant.IS_VALID, Constant.IS_VALIDED));
    }


    @Override
    public List<FlowEntity> findAll() {
        List<FlowEntity> flowList =
                this.list(new QueryWrapper<FlowEntity>().eq(Constant.IS_VALID, Constant.IS_VALIDED));
        if (CollectionUtils.isEmpty(flowList)) {
            return null;
        }

        flowList.forEach(flowEntity -> {
            flowEntity.setNodeDataArray(nodeService.listRichByFlowId(flowEntity.getId()));
            flowEntity.setLinkDataArray(nodeMappingService.ListRichByFlowId(flowEntity.getId()));
        });

        return flowList;
    }

    @Override
    public List<FlowEntity> findSimpleFlow(String status) {
        List<FlowEntity> flowList =
                this.list(new QueryWrapper<FlowEntity>().eq(Constant.IS_VALID, Constant.IS_VALIDED)
                        .eq(StringUtils.isNotBlank(status), "status", FlowEntity.STATUS_PUBLISH));
        return flowList;
    }

    @Override
    public R publishFlow(String flowId) {
        FlowEntity flowEntity = new FlowEntity();
        flowEntity.setId(flowId);
        flowEntity.setStatus(FlowEntity.STATUS_PUBLISH);
        this.updateById(flowEntity);
        return R.ok();
    }

    @Override
    public FlowEntity findOneFlow(String projectId, String action, String api, String type) {
        return this.getOne(new QueryWrapper<FlowEntity>()
                .eq(StringUtils.isNotBlank(projectId), "project_id", projectId)
                .eq(StringUtils.isNotBlank(api), "bound_api", api)
                .eq(StringUtils.isNotBlank(action), "bound_action", action)
                .eq(StringUtils.isNotBlank(type), "type", type)
                .eq(Constant.IS_VALID, Constant.IS_VALIDED)
        );
    }

    @Override
    public R unPublishFlow(String flowId) {
        FlowEntity flowEntity = new FlowEntity();
        flowEntity.setId(flowId);
        flowEntity.setStatus(FlowEntity.STATUS_INIT);
        this.updateById(flowEntity);
        return R.ok();
    }

    @Override
    public List<FlowEntity> find(FlowQueryParam param) {
        return this.list(new QueryWrapper<FlowEntity>()
                .eq(StringUtils.isNotBlank(param.getProjectId()), "project_id", param.getProjectId())
                .eq(StringUtils.isNotBlank(param.getApiId()), "bound_api", param.getApiId())
                .eq(StringUtils.isNotBlank(param.getAction()), "bound_action", param.getAction())
                .eq(StringUtils.isNotBlank(param.getType()), "type", param.getType())
                .eq(StringUtils.isNotBlank(param.getStatus()), "status", param.getStatus())
                .in(CollectionUtils.isNotEmpty(param.getTypes()),"type", param.getTypes())
                .eq(Constant.IS_VALID, Constant.IS_VALIDED)
        );
    }

    @Transactional
    @Override
    public boolean deleteById(String flowId) {
        FlowEntity flow = this.getById(flowId);
        if(flow==null){
            return true;
        }
        String curTime = DateUtils.format(new Date(), DateUtils.DATE_TIME_PATTERN);
        flow.setIsValid(Constant.IS_NOT_VALIDED);
        flow.setModifiedTime(curTime);
        this.updateById(flow);
        this.nodeService.deleteByFlowId(flowId);
        this.nodeMappingService.deleteLogicByFlowId(flowId);

        return true;
    }
}
