package com.xforceplus.janus.config.core.config;

import com.xforceplus.apollo.utils.ErrorUtil;
import com.xforceplus.apollo.utils.JacksonUtil;
import com.xforceplus.janus.config.core.util.Constant;
import com.xforceplus.janus.config.core.util.JanusHttpUtil;

import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpStatus;
import org.springframework.util.CollectionUtils;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import lombok.AllArgsConstructor;
import lombok.NoArgsConstructor;
import lombok.extern.slf4j.Slf4j;

/**
 * 通过http 获取配置中心配置，获取配置的默认方式
 *
 * @Author: xuchuanhou
 * @Date:2021/12/29上午10:37
 */
@Slf4j
@NoArgsConstructor
@AllArgsConstructor
public class ConfigHttpBuilder implements ConfigBuilder {

    private HttpConfig config;

    /**
     * 项目配置中心版本
     */
    public static final String KEY_PROJECT_CONFIG_VERSION = "projectConfigVersion";

    @Override
    public String buildConfig() {

        if (config == null || StringUtils.isEmpty(config.getAuthentication()) || StringUtils.isEmpty(config.getAction().getConfig())) {
            throw new NullPointerException("系统配置 action or Authentication is null");
        }

        Map<String, String> headers = new HashMap<>();
        headers.put("Authentication", config.getAuthentication());
        headers.put("action", config.getAction().getConfig());
        headers.put(Constant.KEY_SERIALNO, "" + System.currentTimeMillis());
        headers.put("Accept-Encoding", "etf");
        headers.put("janusVersion", config.getVersion());
        headers.put("clientType", config.getClientType());
        headers.put("clientVersion", config.getClientVersion());
        headers.put("curconfigversion", HttpConfig.getConfig(KEY_PROJECT_CONFIG_VERSION));
        if (StringUtils.isNotBlank(config.getProxyHost())) {
            headers.put(JanusHttpUtil.KEY_PROXY_HOST, config.getProxyHost());
            headers.put(JanusHttpUtil.KEY_PROXY_PORT, config.getProxyPort());
        }
        int retryTimes = 5;
        while (retryTimes > 0) {
            retryTimes--;
            try {
                JanusHttpUtil.ResponseCus responseCus = JanusHttpUtil.doGetEntire(config.getUrl(), null, false, headers);
                log.info("fetchJanusConfig1 status : {}", responseCus.getStatus());
                if (responseCus.getStatus() == HttpStatus.SC_OK) {
                    log.info("fetchJanusConfig1 : {}", responseCus.getBody());
                    String finalReult = convertApiResult2TcpResult(responseCus.getBody());

                    return finalReult;
                } else {
                    throw new IOException("fetchJanusConfig 失败");
                }

            } catch (IOException ex) {

                try {
                    JanusHttpUtil.ResponseCus responseCus = JanusHttpUtil.doPostJsonEntire(config.getUrl(), "{}", headers, null);
                    log.info("fetchJanusConfig2 status : {}", responseCus.getStatus());
                    if (responseCus.getStatus() == HttpStatus.SC_OK) {
                        log.info("fetchJanusConfig2 : {}", responseCus.getBody());
                        String finalReult = convertApiResult2TcpResult(responseCus.getBody());
                        return finalReult;
                    }
                    Thread.sleep(1000 * 1);
                } catch (Exception ircp) {
                    log.error("fetchJanusConfig exception:{},{}", ErrorUtil.getStackMsg(ex));
                }
            }
        }
        log.error("通过http 获取系统配置信息失败");
        throw new RuntimeException("从集成平台获取系统配置信息失败");
    }


    /**
     * API的配置中心信息结构修改与TCP 保持一致*
     *
     * @return
     */
    public String convertApiResult2TcpResult(String result) throws IOException {
        // API的配置中心信息结构修改与TCP 保持一致
        Map<String, Object> resultMap = JacksonUtil.getInstance().fromJson(result, Map.class);
        if (resultMap == null || !String.valueOf(HttpStatus.SC_OK).equals(resultMap.get("code")) || resultMap.get("configs") == null) {
            throw new IOException("fetch remote config center error");
        }

        List<Map> configs = JacksonUtil.getInstance().fromJsonToList(JacksonUtil.getInstance().toJson(resultMap.get("configs")), Map.class);
        if (CollectionUtils.isEmpty(configs)) {
            return null;
        }
        Map<String, Object> configsMap = new HashMap<>();
        configs.forEach(config -> {
            configsMap.put((String) config.get("configKey"), config.get("configValue"));
        });

        return JacksonUtil.getInstance().toJson(configsMap);
    }
}
