package com.xforceplus.janus.config.core.init;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.xforceplus.apollo.utils.JacksonUtil;
import com.xforceplus.janus.config.core.config.ConfigBuilder;
import com.xforceplus.janus.config.core.config.ConfigHandler;
import com.xforceplus.janus.config.core.config.HttpConfig;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.CommandLineRunner;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;

/**
 * 系统初始化配置信息(从云端拉取配置信息)
 *
 * @Author: xuchuanhou
 * @Date:2021/12/29上午10:21
 */
@Slf4j
@AllArgsConstructor
//@Component
//public class JanusConfigInitRunnner implements InitializingBean {
public class JanusConfigInitRunnner implements CommandLineRunner {

    private List<ConfigHandler> configHandlerList;

    private ConfigBuilder configBuilder;

    private HttpConfig httpConfig;


    @Override
    public void run(String... args) throws Exception {
        this.jsonOrderInit();
        this.fetchSysConfig();
        String config = configBuilder.buildConfig();
        if (StringUtils.isNotBlank(config)) {
            dispatchConfig(config);
        }
    }

    private void jsonOrderInit() {
        if (!httpConfig.isJsonOrder()) {
            //特殊处理项,去除JSON序列化排序
            try {
                Field mappField = JacksonUtil.class.getDeclaredField("mapper");
                mappField.setAccessible(true);
                ObjectMapper mapper = (ObjectMapper) mappField.get(JacksonUtil.getInstance());
                mapper.configure(SerializationFeature.ORDER_MAP_ENTRIES_BY_KEYS, false);
            } catch (Exception e) {
                log.warn("去除JSON序列化排序 失败");
            }
        }
    }

    protected void fetchSysConfig() {
        new Thread(() -> {
            while (true) {
                try {
                    Thread.sleep(300000);
                    String config = configBuilder.buildConfig();
                    if (StringUtils.isNotBlank(config)) {
                        dispatchConfig(config);
                    }
                } catch (Exception ex) {
                    log.error("拉取配置中心配置异常", ex);
                }
            }
        }).start();

    }


    /**
     * 分发到不同处理器
     */
    private void dispatchConfig(String body) {
        if (CollectionUtils.isEmpty(configHandlerList)) {
            return;
        }

        Map<String, Object> allConfigs = JacksonUtil.getInstance().fromJson(body, HashMap.class);

        //无处理器的配置全部放到缓存中
        Set<String> configKeys = configHandlerList.stream().map(cfh -> cfh.getConfigKey()).collect(Collectors.toSet());
        for (Map.Entry<String, Object> entry : allConfigs.entrySet()) {
            if (entry.getKey() != null || entry.getValue() != null) {
                if (configKeys.contains(entry.getKey())) {
                    continue;
                }
                httpConfig.setConfig(entry.getKey(), entry.getValue());
            }
        }


        for (ConfigHandler configHandler : configHandlerList) {
            log.info("system config key:{} init>>>>>>>>>>", configHandler.getConfigKey());
            Object configValue = allConfigs.get(configHandler.getConfigKey());
            if (configValue == null || StringUtils.isEmpty(configValue.toString())) {
                configHandler.nullValueHandler();
                continue;
            }
            String configJson = null;
            if (configValue instanceof String) {
                configJson = (String) configValue;
            } else {
                configJson = JacksonUtil.getInstance().toJson(configValue);
            }

            log.debug("client init config:{} ,value:{}", configHandler.getConfigKey(), configJson);
            configHandler.doHandler(configJson);
        }


    }
}
