package com.xforceplus.janus.config.core.util;

import com.xforceplus.apollo.utils.HttpUtil;
import com.xforceplus.apollo.utils.IOUtil;
import com.xforceplus.apollo.utils.StringUtil;
import lombok.Data;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.NameValuePair;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.*;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.entity.InputStreamEntity;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.message.BasicNameValuePair;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.misc.BASE64Decoder;

import java.io.*;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLEncoder;
import java.util.*;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

/**
 * @Author: xuchuanhou
 * @Date:2022/7/21下午1:47
 */
public class JanusHttpUtil {
    private static final Logger logger = LoggerFactory.getLogger(HttpUtil.class);
    private static final String DEFAULT_CHARSET = "UTF-8";
    private static PoolingHttpClientConnectionManager connMgr;
    private static RequestConfig requestConfig;
    private static HttpClient httpClient;
    private static final int MAX_TIMEOUT = 3000;
    private static final int MAX_SOCKET_TIMEOUT = 60000;
    public static final String APPLICATION_JSON = "application/json";
    public static final String CONTENT_TYPE_TEXT_JSON = "text/json";
    public static final String CONTENT_TYPE_TEXT_STREAM = "application/octet-stream";

    public static final String HEADER_KEY_CONTENT_TYPE = "Content-Type";
    public static final String HEADER_KEY_CONTENT_ENCODING = "Content-Encoding";
    public static final String CONTENT_ENCODING_GZIP = "gzip";

    public static final String KEY_PROXY_HOST = "proxyHost";
    public static final String KEY_PROXY_PORT = "proxyPort";

    static {
        Registry<ConnectionSocketFactory> registry = RegistryBuilder.<ConnectionSocketFactory>create().register("http", PlainConnectionSocketFactory.INSTANCE)
                .register("https", SSLConnectionSocketFactory.getSystemSocketFactory()).build();

        connMgr = new PoolingHttpClientConnectionManager(registry);
        connMgr.setMaxTotal(100);
        connMgr.setDefaultMaxPerRoute(connMgr.getMaxTotal());
        //default 2000
//        connMgr.setValidateAfterInactivity(1000);
        RequestConfig.Builder configBuilder = RequestConfig.custom();
        configBuilder.setConnectTimeout(MAX_TIMEOUT);
        /**读取数据超时时间*/
        configBuilder.setSocketTimeout(MAX_SOCKET_TIMEOUT);
        /**连接池获取连接超时时间*/
        configBuilder.setConnectionRequestTimeout(MAX_TIMEOUT);
        configBuilder.setStaleConnectionCheckEnabled(true);
        requestConfig = configBuilder.build();
//        System.getProperties().setProperty("http.keepAlive", "true");
        httpClient = HttpClients.custom().setConnectionManager(connMgr).build();
    }

    public JanusHttpUtil() {
    }

    public static String doPost(String url) throws IOException {
        return doPost(url, (Map) null);
    }

    public static String doGet(String url) throws IOException {
        return doGet(url, (Map) null);
    }

    public static JanusHttpUtil.ResponseCus doDeleteEntire(String url, Map<String, String> headers, Map<String, Object> params) throws IOException {
        JanusHttpUtil.ResponseCus responseCus = new JanusHttpUtil.ResponseCus();

        HttpDelete delete = new HttpDelete(buildQueryUrlWithParams(url, params));
        delete.addHeader(HEADER_KEY_CONTENT_TYPE, "application/json");
        if (MapUtils.isNotEmpty(headers)) {
            Iterator var4 = headers.entrySet().iterator();

            while (var4.hasNext()) {
                Map.Entry<String, String> entry = (Map.Entry) var4.next();
                delete.addHeader((String) entry.getKey(), (String) entry.getValue());
            }
        }

        InputStream instream = null;
        try {
            delete.setConfig(getProxyRequestConfig(headers));
            HttpResponse response = httpClient.execute(delete);
            responseCus.setStatus(response.getStatusLine().getStatusCode());
            HttpEntity entity = response.getEntity();
            if (entity != null) {
                instream = entity.getContent();
                String result = getStreamAsString(instream, DEFAULT_CHARSET);
                EntityUtils.consumeQuietly(entity);
                responseCus.setBody(result);
            }
        } finally {
            IOUtil.closeQuietly(new Closeable[]{instream});
        }

        return responseCus;
    }

    public static JanusHttpUtil.ResponseCus doPutPatchEntire(String url, String httpMethod, String json, Map<String, String> headers, Map<String, Object> params) throws IOException {
        JanusHttpUtil.ResponseCus responseCus = new JanusHttpUtil.ResponseCus();
        HttpEntityEnclosingRequestBase requestBase = null;
        if ("PUT".equalsIgnoreCase(httpMethod)) {
            requestBase = new HttpPut(buildQueryUrlWithParams(url, params));
        } else if ("PATCH".equalsIgnoreCase(httpMethod)) {
            requestBase = new HttpPatch(buildQueryUrlWithParams(url, params));
        }

        InputStream instream = null;

        try {
            ((HttpEntityEnclosingRequestBase) requestBase).addHeader(HEADER_KEY_CONTENT_TYPE, "application/json");
            if (MapUtils.isNotEmpty(headers)) {
                Iterator var9 = headers.entrySet().iterator();

                while (var9.hasNext()) {
                    Map.Entry<String, String> entry = (Map.Entry) var9.next();
                    ((HttpEntityEnclosingRequestBase) requestBase).addHeader((String) entry.getKey(), (String) entry.getValue());
                }
            }

            if (StringUtils.isNotEmpty(json)) {
                StringEntity se = new StringEntity(json, DEFAULT_CHARSET);
                se.setContentType(CONTENT_TYPE_TEXT_JSON);
                ((HttpEntityEnclosingRequestBase) requestBase).setEntity(se);
            }

            ((HttpEntityEnclosingRequestBase) requestBase).setConfig(getProxyRequestConfig(headers));
            HttpResponse response = httpClient.execute((HttpUriRequest) requestBase);
            HttpEntity entity = response.getEntity();
            responseCus.setStatus(response.getStatusLine().getStatusCode());
            if (entity != null) {
                instream = entity.getContent();
                String result = getStreamAsString(instream, DEFAULT_CHARSET);
                EntityUtils.consumeQuietly(entity);
                responseCus.setBody(result);
            }
        } finally {
            IOUtil.closeQuietly(new Closeable[]{instream});
        }

        return responseCus;
    }

    public static String doPost(String url, Map<String, Object> params) throws IOException {
        JanusHttpUtil.ResponseCus responseCus = doPostJsonEntire(url, (String) null, (Map) null, params);
        return responseCus.getBody();
    }

    public static String doGet(String url, Map<String, Object> params) throws IOException {
        String result = null;
        JanusHttpUtil.ResponseCus responseCus = doGetEntire(url, params, true, (Map) null);
        return responseCus.getBody();
    }

    public static String buildQuery(Map<String, Object> params, String charset) throws IOException {
        if (params != null && !params.isEmpty()) {
            StringBuilder query = new StringBuilder();
            Set<Map.Entry<String, Object>> entries = params.entrySet();
            boolean hasParam = false;
            Iterator var5 = entries.iterator();

            while (var5.hasNext()) {
                Map.Entry<String, Object> entry = (Map.Entry) var5.next();
                String name = (String) entry.getKey();
                String value = String.valueOf(entry.getValue());
                if (StringUtil.areNotEmpty(new String[]{name, value})) {
                    if (hasParam) {
                        query.append("&");
                    } else {
                        hasParam = true;
                    }

                    query.append(name).append("=").append(URLEncoder.encode(value, charset));
                }
            }

            return query.toString();
        } else {
            return null;
        }
    }

    private static String getStreamAsString(InputStream stream, String charset) throws IOException {
        BufferedReader in = null;

        try {
            in = new BufferedReader(new InputStreamReader(stream, charset));
            StringBuffer buffer = new StringBuffer();

            String line;
            while ((line = in.readLine()) != null) {
                buffer.append(line);
            }

            String result = buffer.toString();
            return result;
        } finally {
            if (null != stream) {
                stream.close();
            }

            if (null != in) {
                in.close();
            }

        }
    }

    public static String http(String url, Map<String, String> params) {
        HttpURLConnection con = null;
        StringBuffer sb = new StringBuffer();
        if (params != null) {
            Iterator var5 = params.entrySet().iterator();

            while (var5.hasNext()) {
                Map.Entry<String, String> e = (Map.Entry) var5.next();
                sb.append((String) e.getKey());
                sb.append("=");
                sb.append((String) e.getValue());
                sb.append("&");
            }

            sb.substring(0, sb.length() - 1);
        }

        try {
            URL u = new URL(url);
            con = (HttpURLConnection) u.openConnection();
            con.setRequestMethod("POST");
            con.setDoOutput(true);
            con.setDoInput(true);
            con.setUseCaches(false);
            con.setRequestProperty(HEADER_KEY_CONTENT_TYPE, "application/x-www-form-urlencoded");
            OutputStreamWriter osw = new OutputStreamWriter(con.getOutputStream(), DEFAULT_CHARSET);
            osw.write(sb.toString());
            osw.flush();
            osw.close();
        } catch (Exception var11) {
            var11.printStackTrace();
        } finally {
            if (con != null) {
                con.disconnect();
            }

        }

        StringBuffer buffer = new StringBuffer();

        try {
            BufferedReader br = new BufferedReader(new InputStreamReader(con.getInputStream(), DEFAULT_CHARSET));

            String temp;
            while ((temp = br.readLine()) != null) {
                buffer.append(temp);
                buffer.append("\n");
            }
        } catch (Exception var13) {
            var13.printStackTrace();
        }

        return buffer.toString();
    }

    public static String doPostJson(String url, String json) {
        String result;
        try {
            JanusHttpUtil.ResponseCus responseCus = doPostJsonEntire(url, json, (Map) null, (Map) null);
            result = responseCus.getBody();
        } catch (IOException var4) {
            var4.printStackTrace();
            result = var4.getMessage();
        }

        return result;
    }

//    public static String doPostSSL(String apiUrl, Map<String, Object> params) {
//        HttpPost httpPost = new HttpPost(apiUrl);
//        HttpResponse response = null;
//        String httpStr = null;
//
//        Map.Entry entry;
//        try {
//            List<NameValuePair> pairList = new ArrayList(params.size());
//            Iterator var6 = params.entrySet().iterator();
//
//            BasicNameValuePair pair;
//            while(var6.hasNext()) {
//                entry = (Map.Entry)var6.next();
//                pair = new BasicNameValuePair((String)entry.getKey(), entry.getValue().toString());
//                pairList.add(pair);
//            }
//
//            httpPost.setEntity(new UrlEncodedFormEntity(pairList, Charset.forName("UTF-8")));
//            response = httpClient.execute(httpPost);
//            int statusCode = response.getStatusLine().getStatusCode();
//            if (statusCode == 200) {
//                HttpEntity entity = response.getEntity();
//                if (entity == null) {
//                    pair = null;
//                    return pair;
//                }
//
//                httpStr = EntityUtils.toString(entity, "UTF-8");
//                return httpStr;
//            }
//
//            entry = null;
//        } catch (Exception var20) {
//            logger.error(ErrorUtil.getStackMsg(var20));
//            return httpStr;
//        } finally {
//            if (response != null) {
//                try {
//                    EntityUtils.consume(response.getEntity());
//                } catch (IOException var19) {
//                    logger.error(ErrorUtil.getStackMsg(var19));
//                }
//            }
//
//        }
//
//        return entry;
//    }

//    public static String doPostSSL(String apiUrl, Object json) {
//        HttpPost httpPost = new HttpPost(apiUrl);
//        HttpResponse response = null;
//        String httpStr = null;
//
//        try {
//            httpPost.setConfig(requestConfig);
//            StringEntity stringEntity = new StringEntity(json.toString(), "UTF-8");
//            stringEntity.setContentEncoding("UTF-8");
//            stringEntity.setContentType("application/json");
//            httpPost.setEntity(stringEntity);
//            response = httpClient.execute(httpPost);
//            int statusCode = response.getStatusLine().getStatusCode();
//            HttpEntity entity;
//            if (statusCode != 200) {
//                entity = null;
//                return entity;
//            }
//
//            entity = response.getEntity();
//            if (entity == null) {
//                Object var8 = null;
//                return (String)var8;
//            }
//
//            httpStr = EntityUtils.toString(entity, "UTF-8");
//        } catch (Exception var20) {
//            var20.printStackTrace();
//        } finally {
//            if (response != null) {
//                try {
//                    EntityUtils.consume(response.getEntity());
//                } catch (IOException var19) {
//                    var19.printStackTrace();
//                }
//            }
//
//        }
//
//        return httpStr;
//    }

    public static String doPostForm(String url, Map<String, String> param) {
        HttpResponse response = null;
        String resultString = "";

        try {
            HttpPost httpPost = new HttpPost(url);
            if (param != null) {
                List<NameValuePair> paramList = new ArrayList();
                Iterator var6 = param.keySet().iterator();

                while (var6.hasNext()) {
                    String key = (String) var6.next();
                    paramList.add(new BasicNameValuePair(key, (String) param.get(key)));
                }

                UrlEncodedFormEntity entity = new UrlEncodedFormEntity(paramList, DEFAULT_CHARSET);
                httpPost.setEntity(entity);
            }

            response = httpClient.execute(httpPost);
            resultString = EntityUtils.toString(response.getEntity(), DEFAULT_CHARSET);
        } catch (Exception var8) {
            var8.printStackTrace();
        }

        return resultString;
    }

    public static String buildQueryUrlWithParams(String url, Map<String, Object> params) throws IOException {
        if (MapUtils.isEmpty(params)) {
            return url;
        } else {
            StringBuilder urlSb = new StringBuilder(url);
            String urlParam = buildQuery(params, DEFAULT_CHARSET);
            if (url.contains("?") && url.contains("=")) {
                urlSb.append("&");
            } else {
                urlSb.append("?");
            }

            urlSb.append(urlParam);
            return urlSb.toString();
        }
    }

    public static JanusHttpUtil.ResponseCus doGetEntire(String url, Map<String, Object> params, boolean json, Map<String, String> headers) throws IOException {
        JanusHttpUtil.ResponseCus responseCus = new JanusHttpUtil.ResponseCus();
        InputStream instream = null;

        try {
            HttpGet httpget = new HttpGet(buildQueryUrlWithParams(url, params));

            httpget.setConfig(getProxyRequestConfig(headers));

            if (json) {
                httpget.addHeader(HEADER_KEY_CONTENT_TYPE, APPLICATION_JSON);
            }

            if (MapUtils.isNotEmpty(headers)) {
                Iterator var7 = headers.entrySet().iterator();

                while (var7.hasNext()) {
                    Map.Entry<String, String> entry = (Map.Entry) var7.next();
                    httpget.addHeader((String) entry.getKey(), (String) entry.getValue());
                }
            }

            HttpResponse response = httpClient.execute(httpget);
            responseCus.setStatus(response.getStatusLine().getStatusCode());
            HttpEntity entity = response.getEntity();
            if (entity != null) {
                instream = entity.getContent();
                responseCus.setBody(getStreamAsString(instream, DEFAULT_CHARSET));
                EntityUtils.consumeQuietly(entity);
            }
        } finally {
            IOUtil.closeQuietly(new Closeable[]{instream});
        }

        return responseCus;
    }

    /**
     * 如果有代理则设置代理地址
     */
    public static RequestConfig getProxyRequestConfig(Map<String, String> headers) {
        if (headers == null || !headers.containsKey(KEY_PROXY_HOST)) {
            return JanusHttpUtil.requestConfig;
        }
        String proxyHost = headers.get(KEY_PROXY_HOST);
        String proxyPort = headers.get(KEY_PROXY_PORT);

        RequestConfig.Builder configBuilder = RequestConfig.custom();
        configBuilder.setConnectTimeout(MAX_TIMEOUT);
        /**读取数据超时时间*/
        configBuilder.setSocketTimeout(MAX_SOCKET_TIMEOUT);
        /**连接池获取连接超时时间*/
        configBuilder.setConnectionRequestTimeout(MAX_TIMEOUT);

        if (StringUtils.isNotBlank(proxyPort)) {
            configBuilder.setProxy(new HttpHost(proxyHost, Integer.valueOf(proxyPort)));
        } else {
            configBuilder.setProxy(new HttpHost(proxyHost));
        }
        return configBuilder.build();
    }


    /**
     * <tr>content-type 包含form 按照表单来处理  </tr>
     * <tr>content-encoding 为gzip 则对报文进行压缩处理调用 </tr>
     *
     * @param
     * @return
     * @author xucuanhou
     * @date 2023/6/27
     */
    public static JanusHttpUtil.ResponseCus doPostJsonEntire(String url, String json, Map<String, String> headers, Map<String, Object> params) throws IOException {
        JanusHttpUtil.ResponseCus responseCus = new JanusHttpUtil.ResponseCus();
        InputStream instream = null;
        String contentType = headers.get(HEADER_KEY_CONTENT_TYPE);
        if (contentType != null && contentType.contains("form")) {
            return doPostForm(url, headers, params);
        }

        try {
            HttpPost httpPost = new HttpPost(buildQueryUrlWithParams(url, params));
            if (contentType == null) {
                httpPost.addHeader(HEADER_KEY_CONTENT_TYPE, APPLICATION_JSON);
            }

            if (MapUtils.isNotEmpty(headers)) {
                Iterator var7 = headers.entrySet().iterator();

                while (var7.hasNext()) {
                    Map.Entry<String, String> entry = (Map.Entry) var7.next();
                    httpPost.addHeader((String) entry.getKey(), (String) entry.getValue());
                }
            }

            if (StringUtils.isNotEmpty(json)) {

                if (headers.containsKey(HEADER_KEY_CONTENT_ENCODING) && headers.get(HEADER_KEY_CONTENT_ENCODING).equals(CONTENT_ENCODING_GZIP)) {
                    if (StringUtils.isBlank(json)) {
                        return null;
                    }
                    ByteArrayOutputStream out = new ByteArrayOutputStream();
                    try (GZIPOutputStream gzip = new GZIPOutputStream(out);) {
                        gzip.write(json.getBytes("UTF-8"));
                        gzip.close();
                        byte[] gzipEncrypt = out.toByteArray();
                        httpPost.setEntity(new InputStreamEntity(new ByteArrayInputStream(gzipEncrypt), gzipEncrypt.length));
                    } catch (IOException e) {
                        throw e;
                    }
                } else {
                    StringEntity se = new StringEntity(json, DEFAULT_CHARSET);
                    if (contentType != null && contentType.contains("xml")) {
                    } else {
                        se.setContentType(CONTENT_TYPE_TEXT_JSON);
                    }
                    httpPost.setEntity(se);
                }
            }

            httpPost.setConfig(getProxyRequestConfig(headers));
            HttpResponse response = httpClient.execute(httpPost);
            responseCus.setStatus(response.getStatusLine().getStatusCode());
            HttpEntity entity = response.getEntity();
            if (entity != null) {
                instream = entity.getContent();
                responseCus.setBody(getStreamAsString(instream, DEFAULT_CHARSET));
            }

        } finally {
            IOUtil.closeQuietly(new Closeable[]{instream});
        }

        return responseCus;
    }

    public static JanusHttpUtil.ResponseCus doPostForm(String url, Map<String, String> headers, Map<String, Object> params) throws IOException {
        JanusHttpUtil.ResponseCus responseCus = new JanusHttpUtil.ResponseCus();
        InputStream instream = null;

        try {
            HttpPost httpPost = new HttpPost(url);
            if (MapUtils.isNotEmpty(headers)) {
                for (Map.Entry<String, String> entry : headers.entrySet()) {
                    httpPost.addHeader(entry.getKey(), entry.getValue());
                }
            }

            if (httpPost.getFirstHeader(HEADER_KEY_CONTENT_TYPE) == null) {
                httpPost.addHeader(HEADER_KEY_CONTENT_TYPE, "application/x-www-form-urlencoded;charset=utf-8");
            }

            httpPost.setHeader("Accept", "application/json");
            List<NameValuePair> pairs = new ArrayList();

//            if (MapUtils.isNotEmpty(params) && params.size() == 1) {
//                StringEntity stringEntity = null;
//                for (Map.Entry<String, Object> entry : params.entrySet()) {
//                    stringEntity = new StringEntity(entry.getKey() + "=" + (String) entry.getValue(), "utf-8");
//                }
//                httpPost.setEntity(stringEntity);
//            } else {
            for (Map.Entry<String, Object> entry : params.entrySet()) {
                pairs.add(new BasicNameValuePair(entry.getKey(), (String) entry.getValue()));
            }
            httpPost.setEntity(new UrlEncodedFormEntity(pairs));
//            }


            httpPost.setConfig(getProxyRequestConfig(headers));
            HttpResponse response = httpClient.execute(httpPost);
            responseCus.setStatus(response.getStatusLine().getStatusCode());
            HttpEntity entity = response.getEntity();
            if (entity != null) {
                instream = entity.getContent();
                responseCus.setBody(getStreamAsString(instream, DEFAULT_CHARSET));
            }
        } finally {
            IOUtil.closeQuietly(new Closeable[]{instream});
        }

        return responseCus;
    }

    public static JanusHttpUtil.ResponseCus readFromGzipStream(String url, String json, Map<String, String> headers, Map<String, Object> params) throws IOException {
        JanusHttpUtil.ResponseCus responseCus = new JanusHttpUtil.ResponseCus();
        InputStream instream = null;
        try {
            HttpPost httpPost = new HttpPost(buildQueryUrlWithParams(url, params));
            String contentType = headers.get(HEADER_KEY_CONTENT_TYPE);
            if (contentType == null) {
                httpPost.addHeader(HEADER_KEY_CONTENT_TYPE, APPLICATION_JSON);
            }

            if (MapUtils.isNotEmpty(headers)) {
                Iterator var7 = headers.entrySet().iterator();

                while (var7.hasNext()) {
                    Map.Entry<String, String> entry = (Map.Entry) var7.next();
                    httpPost.addHeader((String) entry.getKey(), (String) entry.getValue());
                }
            }

            StringEntity se = new StringEntity(json, DEFAULT_CHARSET);
            if (contentType != null && contentType.contains("xml")) {
            } else {
                se.setContentType(CONTENT_TYPE_TEXT_JSON);
            }
            httpPost.setEntity(se);

            httpPost.setConfig(getProxyRequestConfig(headers));

            HttpResponse response = httpClient.execute(httpPost);
            responseCus.setStatus(response.getStatusLine().getStatusCode());
            HttpEntity entity = response.getEntity();
            if (entity != null) {
                ByteArrayOutputStream out = new ByteArrayOutputStream();

                try (GZIPInputStream ginzip = new GZIPInputStream(entity.getContent())) {

                    byte[] buffer = new byte[1024];
                    boolean var7 = true;

                    int offset;
                    while ((offset = ginzip.read(buffer)) != -1) {
                        out.write(buffer, 0, offset);
                    }
                    out.flush();
                    responseCus.setBody(out.toString());
                } catch (IOException var24) {
                    var24.printStackTrace();
                } finally {
                    EntityUtils.consumeQuietly(entity);
                    IOUtils.closeQuietly(out);

                }
            }
        } finally {
            IOUtil.closeQuietly(new Closeable[]{instream});
        }

        return responseCus;
    }


    @Data
    public static class ResponseCus implements Serializable {
        int status = 500;
        String body;


        public ResponseCus(int status, String body) {
            this.status = status;
            this.body = body;
        }

        public ResponseCus() {
        }
    }
}
