package com.xforceplus.janus.framework.cmd;

import com.xforceplus.apollo.core.utils.UniqIdUtils;
import com.xforceplus.apollo.msg.SealedMessage;
import com.xforceplus.apollo.utils.ErrorUtil;
import com.xforceplus.apollo.utils.IOUtil;
import com.xforceplus.apollo.utils.JacksonUtil;
import com.xforceplus.janus.config.core.config.HttpConfig;
import com.xforceplus.janus.config.core.monitor.JanusUploader;
import com.xforceplus.janus.framework.record.cache.SysCmdCache;
import com.xforceplus.janus.framework.record.domain.LocalClusterLogQuery;
import com.xforceplus.janus.framework.util.DateUtils;
import com.xforceplus.janus.framework.util.FileCmdCache;
import com.xforceplus.janus.framework.util.FileUtils;
import com.xforceplus.janus.framework.util.IPUtils;
import com.xforceplus.janus.framework.util.ThreadPoolCmdUtils;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Appender;
import org.apache.log4j.FileAppender;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.system.ApplicationHome;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.FutureTask;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * 执行操作系统指令，并将结果反馈至云端
 *
 * @Author: xuchuanhou
 * @Date:2022/7/6下午1:46
 */
@Slf4j
@JanusCmdAnnotation(sourceType = "executeSysCmd")
public class SysCmdHandler implements IJanusCmdHandler, InitializingBean {

    @Setter
    private JanusUploader janusUploader;
    private static final int cmd_response_max = 500000;
    private static final int cmd_response_max_line = 100;
    private static final boolean printDetail = false;
    private static final int size = 5;
    private static final BlockingQueue<SealedMessage> tasks = new ArrayBlockingQueue(size);

    @Setter
    private HttpConfig httpConfig;

    /**
     * 先支持指定日期、单个关键字的查询； 查询当日日志，先查询未压缩文件，再查询压缩文件； 查询非当日日志，只查询压缩文件； 如果上次查询结果来源于压缩文件，可以支持分页；
     * 查询非压缩文件，尽可能多查一点数据
     *
     * @param sealedMessage
     */
    @Override
    public void doHandler(SealedMessage sealedMessage) {
        if (!httpConfig.isRemoteLog()) {
            return;
        }
        log.debug(JacksonUtil.getInstance().toJson(sealedMessage));
        String os_name = loadOsName();
        if (os_name == null || os_name.equals("") || os_name.equals(noSupport)) {
            log.info("您当前使用的系统不支持日志查询");
            String errorMsg = "您当前使用的系统不支持日志查询";
            sendError(sealedMessage, errorMsg);
            return;
        }
        if (!tasks.offer(sealedMessage)) {
            String errorMsg = "待执行的查询日志指令已超过" + size + "条";
            sendError(sealedMessage, errorMsg);
        }
    }

    private void doHandlerSealedMessage(SealedMessage sealedMessage) {
        long start = System.currentTimeMillis();
        try {
            int index = 0;
            int limit = 0;
            String startTime = sealedMessage.getHeader().getOthers().get("startTime");
            String endTime = sealedMessage.getHeader().getOthers().get("endTime");
            String indexStr = sealedMessage.getHeader().getOthers().get("index");
            String logFileName = sealedMessage.getHeader().getOthers().get("logFileName");
            String limitStr = sealedMessage.getHeader().getOthers().get("limit");
            String keyWords = sealedMessage.getHeader().getOthers().get("keyWords");
            String logContent = sealedMessage.getHeader().getOthers().get("logContent");
            String logLocation = sealedMessage.getHeader().getOthers().get("logLocation");
            if (StringUtils.isBlank(logLocation)) {
                logLocation = "self";
            }
            boolean onlyQueryRecord = false;
            if (StringUtils.isNotBlank(logContent) && "true".equals(logContent)) {
                onlyQueryRecord = true;
            }
            if (StringUtils.isBlank(keyWords)) {
                throw new IllegalArgumentException("查询关键字不能为空");
            }
            if (StringUtils.isBlank(startTime) && StringUtils.isBlank(endTime)) {
                throw new IllegalArgumentException("查询时间不能为空");
            }
            if (StringUtils.isBlank(startTime) && StringUtils.isBlank(endTime)) {
                startTime = endTime;
            }
            if (startTime.length() != 19 && startTime.length() != 16 && startTime.length() != 13 && startTime.length() != 10) {
                throw new IllegalArgumentException("查询时间格式：yyyy-MM-dd HH:mm:ss,yyyy-MM-dd HH:mm,yyyy-MM-dd HH,yyyy-MM-dd");
            }
            if (StringUtils.isNotBlank(indexStr)) {
                index = Integer.valueOf(indexStr);
            }
            if (index < 0) {
                index = 0;
            }
            if (StringUtils.isNotBlank(limitStr)) {
                limit = Integer.valueOf(limitStr);
            }

            if (limit <= 0 || limit > cmd_response_max) {
                limit = cmd_response_max;
            }
            String ipAddress = address == null || address.size() == 0 ? "" : String.join(",", address);
            //属地集群子项目处理
            if (!logLocation.equals("self")) {
                String uuid = UniqIdUtils.getInstance().getUniqID();
                LocalClusterLogQuery localClusterLogQuery = new LocalClusterLogQuery();
                localClusterLogQuery.setStartTime(startTime);
                localClusterLogQuery.setEndTime(endTime);
                localClusterLogQuery.setIndex(index + "");
                localClusterLogQuery.setLimit(limit + "");
                localClusterLogQuery.setLogFileName(logFileName);
                localClusterLogQuery.setKeyWords(keyWords);
                localClusterLogQuery.setLogLocation(logLocation);
                localClusterLogQuery.setLogContent(logContent);
                localClusterLogQuery.setTimeStamp(System.currentTimeMillis() + "");
                localClusterLogQuery.setIpAddress(ipAddress);
                localClusterLogQuery.setUuid(uuid);
                localClusterLogQuery.setSourceType(sealedMessage.getHeader().getOthers().get("sourceType"));
                localClusterLogQuery.setCmdSerialKey(sealedMessage.getHeader().getOthers().get("cmdSerialKey"));
                SysCmdCache.put(localClusterLogQuery);
                Map resultMap = new HashMap();
                resultMap.put("logFileName", "");
                resultMap.put("readStatus", "false");
                resultMap.put("readResult", "等待属地子项目查询日志");
                resultMap.put("uuid", uuid);
                resultMap.put("ipAddress", ipAddress);
                resultMap.put("readInx", "0");
                resultMap.put("totalLength", "");
                sendSucc(resultMap, sealedMessage);
                return;
            }
            String startDay = startTime.substring(0, 10);
            String today = DateUtils.format(new Date(), DateUtils.DATE_PATTERN);
            //是否查询当天日志
            boolean isToday = today.equals(startDay);
            List<String> files1 = fileMap.get(startDay);
            List<String> files2 = fileMap.get("nodateTime");
            //日志文件列表未加载
            boolean fileListNotLoaded = CollectionUtils.isEmpty(files1) && CollectionUtils.isEmpty(files2);
            if (printDetail) {
                log.info("fileListNotLoaded: {} ", fileListNotLoaded);
                log.info("isToday: {} ", isToday);
            }
            //如果还未读取到日志文件列表，读取默认的
            if (fileListNotLoaded) {
                String fileName = getLogsDir() + File.separator + "apollo";
                String cmd = getCmd(startTime, keyWords, fileName, onlyQueryRecord);
                Map resultMap = doCmd(cmd, fileName, false, 0, cmd_response_max);
                if (resultMap != null && "true".equals(resultMap.get("readStatus"))) {
                    log.info("send1Succ {}", fileName);
                    sendSucc(resultMap, sealedMessage);
                    return;
                }
            } else {
                //查询当天日志
                if (isToday) {
                    //第一次查询或者上次读取读到就是未压缩的文件,先查非压缩文件中的日志，再查压缩文件中的日志
                    if (StringUtils.isBlank(logFileName) || !logFileName.endsWith(".gz")) {
                        if (noUseIndex(sealedMessage, index, limit, startTime, keyWords, files1, files2, onlyQueryRecord)) {
                            return;
                        }
                    } else {
                        if (userIndex(sealedMessage, index, limit, startTime, logFileName, keyWords, files1, files2, onlyQueryRecord)) {
                            return;
                        }
                    }
                } else {
                    //查询非当天日志， 先查压缩文件中的日志，再查非压缩文件中的日志
                    if (userIndex(sealedMessage, index, limit, startTime, logFileName, keyWords, files1, files2, onlyQueryRecord)) {
                        return;
                    }
                }

            }
            String errorMsg = "没有查询到日志";
            sendError(sealedMessage, errorMsg);
        } catch (Exception e) {
            log.error(ErrorUtil.getStackMsg(e));
            String errorMsg = "查询日志出现异常";
            sendError(sealedMessage, errorMsg);
        } finally {
            log.info("doHandlerSealedMessage costTime : {}", (System.currentTimeMillis() - start));
        }
    }

    private boolean noUseIndex(SealedMessage sealedMessage,
                               int index,
                               int limit, String startTime,
                               String keyWords,
                               List<String> files1,
                               List<String> files2,
                               boolean onlyQueryRecord) {
        if (CollectionUtils.isNotEmpty(files2)) {
            String fileName = files2.get(0);
            String cmd = getCmd(startTime, keyWords, fileName, onlyQueryRecord);
            Map resultMap = doCmd(cmd, fileName, false, 0, cmd_response_max);
            if (resultMap != null && "true".equals(resultMap.get("readStatus"))) {
                log.info("send1Succ {}", fileName);
                sendSucc(resultMap, sealedMessage);
                return true;
            }
        }
        List<FutureTask<Map<String, String>>> tasks = new ArrayList<>();
        //先找有日期 .gz结尾的文件
        if (CollectionUtils.isNotEmpty(files1)) {
            for (String fileName : files1) {
                String cmd = getCmd(startTime, keyWords, fileName, onlyQueryRecord);
                Integer size = FileCmdCache.get(cmd);
                if (size == null) {
                    int finalLimit = limit;
                    Callable<Map<String, String>> callable = () -> doCmd(cmd, fileName, true, 0, finalLimit);
                    FutureTask<Map<String, String>> task = new FutureTask<>(callable);
                    tasks.add(task);
                } else if (size == 0 || size <= index) {
                    continue;
                } else {
                    Map resultMap = doCmd(cmd, fileName, true, index, limit);
                    if (resultMap != null && "true".equals(resultMap.get("readStatus"))) {
                        log.info("send1Succ {}", fileName);
                        sendSucc(resultMap, sealedMessage);
                        return true;
                    }
                }
            }
        }
        Map<String, Map<String, String>> resultMap = new HashMap<>();
        if (CollectionUtils.isNotEmpty(tasks)) {
            for (FutureTask<Map<String, String>> task : tasks) {
                ThreadPoolCmdUtils.execute(task);
            }
            for (FutureTask<Map<String, String>> task : tasks) {
                try {
                    Map<String, String> cmdResultMap = task.get();
                    if ("true".equals(cmdResultMap.get("readStatus"))) {
                        resultMap.put(cmdResultMap.get("logFileName"), cmdResultMap);
                    }
                } catch (InterruptedException e) {
                    log.error("InterruptedException[{}]", e);
                } catch (ExecutionException e) {
                    log.error("ExecutionException[{}]", e);
                } catch (Exception e) {
                    log.error("exception[{}]", e);
                }
            }

            if (CollectionUtils.isNotEmpty(files1)) {
                for (String fileName : files1) {
                    Map<String, String> cmdResultMap = resultMap.get(fileName);
                    if (cmdResultMap != null && "true".equals(cmdResultMap.get("readStatus"))) {
                        log.info("send1Succ {}", fileName);
                        sendSucc(cmdResultMap, sealedMessage);
                        return true;
                    }

                }
            }
        }
        return false;
    }

    private boolean userIndex(SealedMessage sealedMessage,
                              int index,
                              int limit,
                              String startTime,
                              String logFileName,
                              String keyWords,
                              List<String> files1,
                              List<String> files2,
                              boolean onlyQueryRecord) {
        List<FutureTask<Map<String, String>>> tasks = new ArrayList<>();
        //先找有日期 .gz结尾的文件
        if (CollectionUtils.isNotEmpty(files1)) {
            for (String fileName : files1) {
                log.info("fileName.compareTo(logFileName) {} :{} ,{}", fileName.compareTo(logFileName), fileName, logFileName);
                if (fileName.compareTo(logFileName) < 0) {
                    continue;
                }
                String cmd = getCmd(startTime, keyWords, fileName, onlyQueryRecord);
                Integer size = FileCmdCache.get(cmd);
                log.info("fileName.compareTo(logFileName) {} :{} ,{}", fileName.compareTo(logFileName), fileName, logFileName);
                if (size == null) {
                    int finalLimit = limit;
                    Callable<Map<String, String>> callable = () -> doCmd(cmd, fileName, true, 0, finalLimit);
                    FutureTask<Map<String, String>> task = new FutureTask<>(callable);
                    tasks.add(task);
                } else if (fileName.compareTo(logFileName) > 0) {
                    Map resultMap = doCmd(cmd, fileName, true, 0, limit);
                    if (resultMap != null && "true".equals(resultMap.get("readStatus"))) {
                        log.info("send1Succ {}", fileName);
                        sendSucc(resultMap, sealedMessage);
                        return true;
                    }
                } else if (size > index) {
                    Map resultMap = doCmd(cmd, fileName, true, index, limit);
                    if (resultMap != null && "true".equals(resultMap.get("readStatus"))) {
                        log.info("send1Succ {}", fileName);
                        sendSucc(resultMap, sealedMessage);
                        return true;
                    }
                }
            }
        }
        //正常情况应该只有一个没有日期 非.gz结尾的文件
        if (CollectionUtils.isNotEmpty(files2)) {
            String fileName = files2.get(0);
            String cmd = getCmd(startTime, keyWords, fileName, onlyQueryRecord);
            int finalLimit = limit;
            Callable<Map<String, String>> callable = () -> doCmd(cmd, fileName, false, 0, finalLimit);
            FutureTask<Map<String, String>> task = new FutureTask<>(callable);
            tasks.add(task);
        }
        if (CollectionUtils.isNotEmpty(tasks)) {
            for (FutureTask<Map<String, String>> task : tasks) {
                ThreadPoolCmdUtils.execute(task);
            }
        }
        Map<String, Map<String, String>> resultMap = new HashMap<>();
        for (FutureTask<Map<String, String>> task : tasks) {
            try {
                Map<String, String> cmdResultMap = task.get();
                if ("true".equals(cmdResultMap.get("readStatus"))) {
                    resultMap.put(cmdResultMap.get("logFileName"), cmdResultMap);
                }
            } catch (InterruptedException e) {
                log.error("InterruptedException[{}]", e);
            } catch (ExecutionException e) {
                log.error("ExecutionException[{}]", e);
            } catch (Exception e) {
                log.error("exception[{}]", e);
            }
        }
        if (CollectionUtils.isNotEmpty(files1)) {
            for (String fileName : files1) {
                Map<String, String> cmdResultMap = resultMap.get(fileName);
                log.info("fileName: {} ,{} ", fileName, cmdResultMap != null && "true".equals(cmdResultMap.get("readStatus")));
                if (cmdResultMap != null && "true".equals(cmdResultMap.get("readStatus"))) {
                    log.info("send1Succ {}", fileName);
                    sendSucc(cmdResultMap, sealedMessage);
                    return true;
                }
            }
        }
        if (CollectionUtils.isNotEmpty(files2)) {
            for (String fileName : files2) {
                Map<String, String> cmdResultMap = resultMap.get(fileName);
                if (cmdResultMap != null && "true".equals(cmdResultMap.get("readStatus"))) {
                    log.info("send1Succ {}", fileName);
                    sendSucc(cmdResultMap, sealedMessage);
                    return true;
                }
            }
        }
        return false;
    }


    private void sendError(SealedMessage sealedMessage, String errorMsg) {
        Map<String, String> result = new HashMap<>();
        result.put("readStatus", "false");
        result.put("readResult", errorMsg);
        result.put("readInx", "0");
        result.put("totalLength", "0");
        result.put("logFileName", "");
        Map<String, Object> resultMap = new HashMap<>();
        resultMap.put("code", "0");
        resultMap.put("message", "FAILED");
        resultMap.put("executeSysCmd", "executeSysCmd");
        resultMap.put("result", result);
        sendMsg(resultMap, sealedMessage);
    }

    private void sendSucc(Map<String, String> result, SealedMessage sealedMessage) {
        Map<String, Object> resultMap = new HashMap<>();
        resultMap.put("code", "1");
        resultMap.put("message", "FINISHED");
        resultMap.put("executeSysCmd", "executeSysCmd");
        resultMap.put("result", result);
        sendMsg(resultMap, sealedMessage);
    }

    private void sendMsg(Map<String, Object> resultMap, SealedMessage sealedMessage) {
        String resultstr = JacksonUtil.getInstance().toJson(resultMap);
        janusUploader.sendJanusCmdResult(sealedMessage.getHeader().getOthers().get("sourceType"), sealedMessage.getHeader().getOthers().get("cmdSerialKey"), resultstr);
    }


    //    cat /logs/apollo | grep  '2022-07-21 10:05'  | grep 'org.springframework.scheduling.concurrent'
    //    zcat /logs/apollo.2022-07-20-下午.gz | grep  '2022-07-20 13:34'  | grep 'com.xforceplus.apollo.janus.standalone.utils.userCenter.JwtEncoder-90'

    private String getCmd(String startTime, String keyWords, String fileName, boolean onlyQueryRecord) {
        String os_name = loadOsName();
        if (os_name == null || os_name.equals("") || os_name.equals(noSupport)) {
            log.info("您当前使用的系统不支持日志查询");
            return "";
        }
        StringBuilder sb = new StringBuilder();
        if (os_name.equals(Linux)) {
            if (fileName.endsWith(".gz")) {
                sb.append("zcat ").append(fileName).append(" | grep  '").append(startTime).append("' ");
                buildLinuxKeyWords(keyWords, sb);
            } else {
                sb.append("cat ").append(fileName).append(" | grep  '").append(startTime).append("' ");
                buildLinuxKeyWords(keyWords, sb);
            }
            sb.append("  | grep  -v 'com.xforceplus.janus.framework.cmd.SysCmdHandler' ")
                    .append("  | grep  -v 'executeSysCmd' ");
            if (onlyQueryRecord) {
                sb.append("  | grep  'com.xforceplus.janus.framework.record.portal.LogAccessRecordConsumer' ")
                        .append("  | grep  -v 'com.xforceplus.janus.config.core.monitor.JanusUploader'");
            }
        } else if (os_name.equals(Windows)) {
            sb.append("cmd /c   ").append(" find  \"").append(startTime).append("\"  ").append(fileName);
            buildWindowsKeyWords(keyWords, sb);
            sb.append("  | find  /v \"com.xforceplus.janus.framework.cmd.SysCmdHandler\" ")
                    .append("  | find  /v \"executeSysCmd\" ");
            if (onlyQueryRecord) {
                sb.append("  | find  \"com.xforceplus.janus.framework.record.portal.LogAccessRecordConsumer\"  ")
                        .append("  | find  /v \" com.xforceplus.janus.config.core.monitor.JanusUploader\" ");
            }
        }
        return sb.toString();
    }

    private void buildLinuxKeyWords(String keyWords, StringBuilder sb) {
        if (keyWords.contains(",")) {
            String[] splits = keyWords.split(",");
            for (String split : splits) {
                if (StringUtils.isNotBlank(split)) {
                    sb.append(" | grep  '").append(split.trim()).append("' -A2 ");
                }
            }
        } else {
            sb.append(" | grep  '").append(keyWords.trim()).append("' -A2 ");
        }
    }

    private void buildWindowsKeyWords(String keyWords, StringBuilder sb) {
        if (keyWords.contains(",")) {
            String[] splits = keyWords.split(",");
            for (String split : splits) {
                if (StringUtils.isNotBlank(split)) {
                    sb.append(" | find  \"").append(split.trim()).append("\" ");
                }
            }
        } else {
            sb.append(" | find  \"").append(keyWords.trim()).append("\"  ");
        }
    }


    /**
     * @param cmd      执行的命令
     * @param fileName 查询的文件
     * @param cache    是否缓存 查询结果的大小
     * @param index    报文偏移量
     * @param limit    返回的报文大小限制
     * @return
     */
    private Map<String, String> doCmd(String cmd, String fileName, boolean cache, int index, int limit) {
        log.info("receive fileName {},cmd:{},cache:{}", fileName, cmd, cache);
        Map<String, String> resultMap = new HashMap<>();
        resultMap.put("logFileName", fileName);
        StringBuilder cmdResponse = new StringBuilder();
        Process process = null;
        BufferedReader bufferedReader = null;
        BufferedReader errorReader = null;
        int LastLength = 0;
        int length = 0;
        try {
            String os_name = loadOsName();
            if (os_name == null || os_name.equals("") || os_name.equals(noSupport)) {
                log.info("您当前使用的系统不支持日志查询");
                throw new IllegalArgumentException("您当前使用的系统不支持日志查询");
            }
            if (os_name.equals(Linux)) {
                process = Runtime.getRuntime().exec(new String[]{"/bin/sh", "-c", cmd});
                bufferedReader = new BufferedReader(new InputStreamReader(process.getInputStream()));
                errorReader = new BufferedReader(new InputStreamReader(process.getErrorStream()));
                boolean proresult = process.waitFor(4, TimeUnit.SECONDS);
            } else if (os_name.equals(Windows)) {
                process = Runtime.getRuntime().exec(cmd);
                bufferedReader = new BufferedReader(new InputStreamReader(process.getInputStream()));
                errorReader = new BufferedReader(new InputStreamReader(process.getErrorStream()));
                boolean proresult = process.waitFor(4, TimeUnit.SECONDS);
            }
            String cont = null;
            int line = 0;
            boolean full = false;
            int readInx = index;
            while ((cont = bufferedReader.readLine()) != null) {
                LastLength = length;
                length += cont.length();
                if (printDetail) {
                    log.info("LastLength :{},length: {},full:{},cont.length: {},", LastLength, length, full, cont.length());
                }
                if (!full && index < length) {
                    if (index >= LastLength) {
                        String subString = cont.substring(index - LastLength);
                        if (StringUtils.isNotBlank(subString)) {
                            if (subString.length() < (limit - cmdResponse.length())) {
                                cmdResponse.append(subString).append("\n\r");
                            } else {
                                cmdResponse.append(subString.substring(0, limit - cmdResponse.length())).append("\n\r");
                            }
                        }
                    } else {
                        if (StringUtils.isNotBlank(cont)) {
                            if (cont.length() < (limit - cmdResponse.length())) {
                                cmdResponse.append(cont).append("\n\r");
                            } else {
                                cmdResponse.append(cont.substring(0, limit - cmdResponse.length())).append("\n\r");
                            }
                        }

                    }
                    readInx = length;
                }
                if (cmdResponse.length() >= limit) {
                    if (printDetail) {
                        log.info("cmdResponse.length() :{},limit: {},", cmdResponse.length(), limit);
                    }
                    full = true;
                }
                if (line > cmd_response_max_line) {
                    full = true;
                }
                line++;
            }
            resultMap.put("readInx", "" + readInx);
            String errorLine = null;
            if (cmdResponse.length() <= 0) {
                while ((errorLine = errorReader.readLine()) != null) {
                    cmdResponse.append(errorLine).append("\n\r");
                }
                if (cmdResponse.length() > 0) {
                    resultMap.put("readStatus", "false");
                    resultMap.put("readResult", "");
                    log.error("cmd:{}  run error,{}", cmd, cmdResponse);
                }
            } else {
                resultMap.put("readResult", cmdResponse.toString());
                resultMap.put("readStatus", "true");
            }

        } catch (IOException exception) {
            log.error("cmd:{}  run error,{},{}", cmd, exception.getMessage(), ErrorUtil.getStackMsg(exception));
            resultMap.put("readStatus", "false");
            resultMap.put("readResult", "");
        } catch (InterruptedException inrptedException) {
            log.error("cmd:{}  run error,{},{}", cmd, inrptedException.getMessage(), ErrorUtil.getStackMsg(inrptedException));
            resultMap.put("readStatus", "false");
            resultMap.put("readResult", "");
        } catch (Exception exception) {
            log.error("cmd:{}  run error,{},{}", cmd, exception.getMessage(), ErrorUtil.getStackMsg(exception));
            resultMap.put("readStatus", "false");
            resultMap.put("readResult", "");
        } finally {
            if (errorReader != null) {
                IOUtil.closeQuietly(errorReader);
            }

            if (bufferedReader != null) {
                IOUtil.closeQuietly(bufferedReader);
            }
            if (process != null) {
                process.destroy();
            }
        }
        if ("false".equals(resultMap.get("readStatus"))) {
            resultMap.put("readInx", "" + index);
            length = 0;
        }
        if (cache) {
            //缓存一天
            FileCmdCache.set(cmd, length, 1000 * 60 * 60 * 24);
        }
        if (printDetail) {
            log.info("cmdResponse.length() :{},limit: {},readStatus:{}", cmdResponse.length(), limit, resultMap.get("readStatus"));
        }
        resultMap.put("totalLength", "" + length);
        return resultMap;
    }

    private Set<String> forbidCmds = new HashSet<String>() {{
        add("rm ");
        add("rmdir ");
        add("-delete");
        add("awk ");
        add("rmdir ");
    }};

    private String allowCmdPattern = "^(ls|cd|grep|cat|zcat|pwd).*";

    Pattern pattern = Pattern.compile(allowCmdPattern);

    private boolean validateCmd(String cmd) {
        if (StringUtils.isBlank(cmd)) {
            return false;
        }

        for (String forbidCmd : forbidCmds) {
            if (cmd.contains(forbidCmd)) {
                return false;
            }
        }
        Matcher matcher = pattern.matcher(cmd);
        return matcher.find();
    }


    private String deployLibDir() {
        String workDir = "";
        try {
            //第五种
            ApplicationHome h = new ApplicationHome(getClass());
            File jarF = h.getSource();
            workDir = jarF.getParentFile().toString();
            log.info("checklogFiles, path:{} ", workDir);
        } catch (Exception e) {
            log.info("checklogFiles, path  error:{} ", ErrorUtil.getStackMsg(e));
            workDir = "";
        }
        return workDir;
    }

    private static final String log_folder = "logs";
    private static final String no_zip_log_folder = "unZipLogs";
    private volatile Map<String, List<String>> fileMap = new HashMap();
    private volatile List<String> address = new LinkedList<>();

    private String deployJarDir() {
        String workDir = "";
        try {
            workDir = System.getProperty("user.dir");
            return workDir;
        } catch (Exception e) {
            log.info("checklogFiles, path  error:{} ", ErrorUtil.getStackMsg(e));
            workDir = "";
        }
        return workDir;
    }

    public String getLogsDir() {
        try {
            Logger rootLogger = Logger.getRootLogger();
            Enumeration allAppenders = rootLogger.getAllAppenders();
            while (allAppenders.hasMoreElements()) {
                Appender appender = (Appender) allAppenders.nextElement();
                if (appender instanceof FileAppender) {
                    FileAppender fileAppender = (FileAppender) appender;
                    String fileName = fileAppender.getFile();
                    File file = new File(fileName);
                    String canonicalPath = file.getCanonicalPath();
                    canonicalPath = canonicalPath.substring(0, canonicalPath.lastIndexOf(File.separator));
                    log.info("canonicalPath= {}", canonicalPath);
                    return canonicalPath;
                }
            }
        } catch (Exception e) {
            log.error(ErrorUtil.getStackMsg(e));
        } catch (Error e) {
            log.error(ErrorUtil.getStackMsg(e));
        }
        if (StringUtils.isNotBlank(httpConfig.getCusLogDir())) {
            return httpConfig.getCusLogDir();
        }
        return deployJarDir() + File.separator + log_folder;
    }

    private String osName = "";
    private final String Windows = "Windows";
    private final String Linux = "Linux";
    private final String noSupport = "noSupport";

    private String loadOsName() {
        if ("".equals(osName)) {
            String name = System.getProperty("os.name");
            log.info("已检测到您当前使用的系统为：" + name);
            if (osName != null) {
                if (name.startsWith("Windows")) {
                    osName = Windows;
                } else if (name.startsWith("Linux") || name.startsWith("Mac")) {
                    osName = Linux;
                } else {
                    osName = "noSupport";
                    log.info("您当前使用的系统不支持日志查询");
                }
            } else {
                osName = "noSupport";
                log.info("您当前使用的系统不支持日志查询");
            }

        }
        return osName;

    }

    /**
     * 每两分钟查一下日志目录下的文件
     */

    public void checklogFiles() {
        if (!httpConfig.isRemoteLog()) {
            return;
        }
        new Thread(() -> {
            while (true) {
                try {
                    TimeUnit.MINUTES.sleep(10);

                    String os_name = loadOsName();
                    if (os_name == null || os_name.equals("") || os_name.equals(noSupport)) {
                        log.info("您当前使用的系统不支持日志查询");
                        return;
                    }
                    if (os_name.equals(Windows)) {
                        checklogFilesWindows();
                    } else if (os_name.equals(Linux)) {
                        checklogFilesLinux();
                    }
                } catch (InterruptedException interruptedException) {
                    interruptedException.printStackTrace();
                }
            }
        }).start();
    }

    public void checklogFilesLinux() {
        address = IPUtils.getIpAddress();
        String taskId = UniqIdUtils.getInstance().getUniqID();
        String listFilesSmd = "ls " + getLogsDir();
        String result = doCmd(listFilesSmd);
        Map<String, List<String>> fileMapTemp = new HashMap();
        if (StringUtils.isNotBlank(result)) {
            String[] lines = result.split("\\r?\\n");
            for (String fileName : lines) {
                if (StringUtils.isNotBlank(fileName)) {
                    fileName = fileName.replace("\r", "").trim();
                    Map<String, String> map = findDateTime(fileName);
                    String dateTime = map.get("dateTime");
                    putfile(fileMapTemp, fileName, dateTime);
                }
            }
        }
        if (fileMapTemp.size() > 0) {
            for (Map.Entry<String, List<String>> entry : fileMapTemp.entrySet()) {
                List<String> files = entry.getValue();
                String date = entry.getKey();
                Collections.sort(files, String::compareTo);
                fileMapTemp.put(date, files);
            }
        }
        fileMap = fileMapTemp;
        log.info("checklogFiles , taskId:{} ,CMD: {}end fileMap:{}", taskId, listFilesSmd, JacksonUtil.getInstance().toJson(fileMap));
    }

    public void checklogFilesWindows() {
        address = IPUtils.getIpAddress();
        String taskId = UniqIdUtils.getInstance().getUniqID();
        String unzipSearchFolderPath = deployJarDir() + File.separator + no_zip_log_folder;
        List<String> unzipFileNames = ListFileNames(unzipSearchFolderPath, true);
        File unzipSearchFolder = new File(unzipSearchFolderPath);
        if (!unzipSearchFolder.exists()) {
            unzipSearchFolder.mkdirs();
        }
        List<String> needUnzipFileNames = new ArrayList<>();
        String searchFolderPath = getLogsDir();
        List<String> fileNames = ListFileNames(searchFolderPath, false);
        Map<String, List<String>> fileMapTemp = new HashMap();
        if (CollectionUtils.isNotEmpty(fileNames)) {
            for (String fileName : fileNames) {
                if (StringUtils.isNotBlank(fileName)) {
                    fileName = fileName.replace("\r", "").trim();
                    Map<String, String> map = findDateTime(fileName);
                    String dateTime = map.get("dateTime");
                    List<String> files = fileMapTemp.get(dateTime);
                    if (files == null) {
                        files = new ArrayList<>();
                    }
                    if (dateTime == null || "nodateTime".equals(dateTime) || fileName.indexOf(".gz") < 0) {
                        fileName = searchFolderPath + File.separator + fileName;
                        files.add(fileName);
                    } else {
                        String sourceFileName = searchFolderPath + File.separator + fileName;
                        String unzipFileName = unzipSearchFolderPath + File.separator + fileName.replace(".gz", "");
                        if (!unzipFileNames.contains(unzipFileName)) {
                            FileUtils.writeUnGzip(sourceFileName, unzipFileName);
                        }
                        files.add(unzipFileName);
                        needUnzipFileNames.add(unzipFileName);
                    }
                    fileMapTemp.put(dateTime, files);
                }
            }
            if (fileMapTemp.size() > 0) {
                for (Map.Entry<String, List<String>> entry : fileMapTemp.entrySet()) {
                    List<String> files = entry.getValue();
                    String date = entry.getKey();
                    Collections.sort(files, String::compareTo);
                    fileMapTemp.put(date, files);
                }
            }
        }
        if (CollectionUtils.isNotEmpty(unzipFileNames)) {
            for (String unzipFileName : unzipFileNames) {
                if (StringUtils.isNotBlank(unzipFileName)) {
                    if (!needUnzipFileNames.contains(unzipFileName)) {
                        File file = new File(unzipFileName);
                        if (file.exists()) {
                            file.delete();
                        }
                    }
                }
            }
        }
        fileMap = fileMapTemp;
        log.info("checklogFiles , taskId:{} ,CMD: {}end fileMap:{}", taskId, JacksonUtil.getInstance().toJson(fileMap));
    }

    private String doCmd(String cmd) {
        StringBuilder cmdResponse = new StringBuilder();
        Process process = null;
        BufferedReader bufferedReader = null;
        BufferedReader errorReader = null;
        String result = "";
        try {
            process = Runtime.getRuntime().exec(new String[]{"/bin/sh", "-c", cmd});
            bufferedReader = new BufferedReader(new InputStreamReader(process.getInputStream()));
            errorReader = new BufferedReader(new InputStreamReader(process.getErrorStream()));
            boolean proresult = process.waitFor(4, TimeUnit.SECONDS);
            String cont = null;
            int i = 0;
            while ((cont = bufferedReader.readLine()) != null) {
                cmdResponse.append(cont).append("\n\r");
                if (cmdResponse.length() >= cmd_response_max) {
                    break;
                }
                if (i > cmd_response_max_line) {
                    break;
                }
                i++;
            }
            String errorLine = null;
            if (cmdResponse.length() <= 0) {
                while ((errorLine = errorReader.readLine()) != null) {
                    cmdResponse.append(errorLine).append("\n\r");
                }
                if (cmdResponse.length() > 0) {
                    log.error("errorLine {}", cmd, cmdResponse.toString());
                }
            } else {
                result = cmdResponse.toString();
            }

        } catch (IOException exception) {
            result = "";
            log.error("cmd:{}  run error,{},{}", cmd, exception.getMessage(), ErrorUtil.getStackMsg(exception));
        } catch (InterruptedException inrptedException) {
            result = "";
            log.error("cmd:{}  run error,{},{}", cmd, inrptedException.getMessage(), ErrorUtil.getStackMsg(inrptedException));
        } catch (Exception exception) {
            result = "";
            log.error("cmd:{}  run error,{},{}", cmd, exception.getMessage(), ErrorUtil.getStackMsg(exception));
        } catch (Error error) {
            result = "";
            log.error("cmd:{}  run error,{},{}", cmd, error.getMessage(), ErrorUtil.getStackMsg(error));
        } finally {
            if (errorReader != null) {
                IOUtil.closeQuietly(errorReader);
            }

            if (bufferedReader != null) {
                IOUtil.closeQuietly(bufferedReader);
            }
            if (process != null) {
                process.destroy();
            }
        }
        return result;
    }

    private String doCmdWindows(String cmd, String charsetName) {
        log.info(cmd);
        StringBuilder cmdResponse = new StringBuilder();
        Process process = null;
        BufferedReader bufferedReader = null;
        BufferedReader errorReader = null;
        String result = "";
        try {
            process = Runtime.getRuntime().exec(cmd);
            bufferedReader = new BufferedReader(new InputStreamReader(process.getInputStream(), charsetName));
            errorReader = new BufferedReader(new InputStreamReader(process.getErrorStream(), charsetName));
            boolean proresult = process.waitFor(4, TimeUnit.SECONDS);
            String cont = null;
            int i = 0;
            while ((cont = bufferedReader.readLine()) != null) {
                cmdResponse.append(cont).append("\n\r");
                if (cmdResponse.length() >= cmd_response_max) {
                    break;
                }
                if (i > cmd_response_max_line) {
                    break;
                }
                i++;
            }
            String errorLine = null;
            if (cmdResponse.length() <= 0) {
                while ((errorLine = errorReader.readLine()) != null) {
                    cmdResponse.append(errorLine).append("\n\r");
                }
                if (cmdResponse.length() > 0) {
                    log.error("errorLine {}", cmd, cmdResponse.toString());
                }
            } else {
                result = cmdResponse.toString();
            }

        } catch (IOException exception) {
            result = "";
            log.error("cmd:{}  run error,{},{}", cmd, exception.getMessage(), ErrorUtil.getStackMsg(exception));
        } catch (InterruptedException inrptedException) {
            result = "";
            log.error("cmd:{}  run error,{},{}", cmd, inrptedException.getMessage(), ErrorUtil.getStackMsg(inrptedException));
        } catch (Exception exception) {
            result = "";
            log.error("cmd:{}  run error,{},{}", cmd, exception.getMessage(), ErrorUtil.getStackMsg(exception));
        } catch (Error error) {
            result = "";
            log.error("cmd:{}  run error,{},{}", cmd, error.getMessage(), ErrorUtil.getStackMsg(error));
        } finally {
            if (errorReader != null) {
                IOUtil.closeQuietly(errorReader);
            }

            if (bufferedReader != null) {
                IOUtil.closeQuietly(bufferedReader);
            }
            if (process != null) {
                process.destroy();
            }
        }
        return result;
    }

    private void putfile(Map<String, List<String>> fileMapTemp, String fileName, String dateTime) {
        List<String> files = fileMapTemp.get(dateTime);
        if (files == null) {
            files = new ArrayList<>();
        }
        fileName = getLogsDir() + File.separator + fileName;
        files.add(fileName);
        fileMapTemp.put(dateTime, files);
    }

    private static Map<String, String> findDateTime(String fileName) {
        Map<String, String> map = new HashMap<>();
        map.put("orginFileName", fileName);
        map.put("hasDateTime", "false");
        map.put("dateTime", "nodateTime");
        try {
            Pattern p = Pattern.compile("(\\d{4})-(\\d{1,2})-(\\d{1,2})");
            Matcher m = p.matcher(fileName);
            while (m.find()) {
                String dateTime = m.group(0);
                map.put("dateTime", dateTime);
                map.put("hasDateTime", "true");
                break;
            }
        } catch (Exception e) {
            log.info(ErrorUtil.getStackMsg(e));
        }
        return map;
    }


//    public static void main(String[] args) {
//        String envPath = "C:\\Users\\zebra\\Desktop\\localhost\\local-front-redirect";
//        SysCmdHandler sysCmdHandler = new SysCmdHandler();
////
////        System.out.println(fileMapTemp.size());
////        fileMap = fileMapTemp;
//
//
//        String cmd = " cmd /c   find  \"projectInfo\"  D:\\pyt\\gateway\\janus-standalone\\src\\main\\resources\\env-config\\local-front-redirect\\logs\\apollo11-22222  ";
//        String cmd1 = " |   find  \"2022-10-18 13\"   ";
//        String result = sysCmdHandler.doCmdWindows(cmd+cmd1,"UTF-8");
//        System.out.println("===========================");
//        System.out.println(result);
//    }

    private List<String> ListFileNames(String searchFolderPath, boolean containPath) {
        List<String> fileNames = new ArrayList<>();
        try {
            String cmd = " cmd /c dir /s " + searchFolderPath;
            String result = doCmdWindows(cmd, "GBK");
            if (StringUtils.isNotBlank(result)) {
                String[] lines = result.split("\\n\\r");
                for (String line : lines) {
                    if (line == null || line.trim().equals("")) {
                        continue;
                    }
                    String[] fields = line.split("\\s+");
                    if (fields == null || fields.length < 4 || fields[0] == null || fields[0].trim().equals("") || fields[3] == null || fields[0].trim().equals("")) {
                        continue;
                    }
                    String createTime = fields[0];
                    if (createTime == null || createTime.equals("")) {
                        continue;
                    }
                    String fileType = fields[2];
                    if (fileType == null || fileType.equals("") || fileType.equals("<DIR>")) {
                        continue;
                    }
                    String fileName = fields[3];
                    if (fileName == null || fileName.equals("")) {
                        continue;
                    }
                    if (containPath) {
                        fileName = searchFolderPath + File.separator + fileName;
                    }
                    fileNames.add(fileName);
                }
            }
        } catch (Exception e) {
            log.error(ErrorUtil.getStackMsg(e));
        }
        return fileNames;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        String os_name = loadOsName();
        this.checklogFiles();
        if (os_name == null || os_name.equals("") || os_name.equals(noSupport)) {
            log.info("您当前使用的系统不支持日志查询");
        } else {
            new Thread(() -> {
                while (true) {
                    try {
                        boolean hasMessage = false;
                        SealedMessage sealedMessage = tasks.take();
                        if (sealedMessage != null) {
                            hasMessage = true;
                            doHandlerSealedMessage(sealedMessage);
                        }
                        if (!hasMessage) {
                            TimeUnit.SECONDS.sleep(1);
                        }
                    } catch (Exception e) {
                        log.info(ErrorUtil.getStackMsg(e));
                    }
                }
            }, "SysCmdTaskExecutor").start();
        }
    }
}
