package com.xforceplus.janus.framework.record.interceptor;

import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.PathNotFoundException;
import com.xforceplus.apollo.jx.exchange.Json2Json;
import com.xforceplus.apollo.utils.JacksonUtil;
import com.xforceplus.janus.config.core.config.HttpConfig;
import com.xforceplus.janus.config.core.dto.ApiDto;
import com.xforceplus.janus.config.core.dto.FieldMappingDto;
import com.xforceplus.janus.config.core.util.Constant;
import com.xforceplus.janus.config.core.util.JanusHttpUtil;
import com.xforceplus.janus.framework.dto.JanusRequst;
import com.xforceplus.janus.framework.dto.Result;
import com.xforceplus.janus.framework.record.cache.AccessRecordCache;
import com.xforceplus.janus.framework.record.domain.AccessContentDto;
import com.xforceplus.janus.framework.record.domain.AccessRecord;
import com.xforceplus.janus.framework.record.init.ProjectCacheHandler;
import com.xforceplus.janus.framework.util.IPUtils;
import com.xforceplus.janus.framework.util.RequestUtil;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPatch;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpPut;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpStatus;
import org.springframework.web.servlet.HandlerInterceptor;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * 请求记录拦击记录履历
 *
 * @Author: xuchuanhou
 * @Date:2022/2/19下午3:46
 */
@Slf4j
@Order(1)
public class RequestInterceptor implements HandlerInterceptor {

    private static final String CONTENT_TYPE_MULTIPART = "multipart/form-data";

    @Setter
    private ProjectCacheHandler projectCacheHandler;

    @Setter
    private HttpConfig httpConfig;
    @Setter
    private IHttpDiyFlowHandler diyFlowHandler;


    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        request.setAttribute("startTime", System.currentTimeMillis());
        String uri = request.getRequestURI();
        uri = uri.split("\\?")[0];

        String urlId = uri + request.getMethod();
        request.setAttribute("urlId", urlId);
        //无权限，则直接返回
        ApiDto apiDto = projectCacheHandler.getApiCache().get(urlId);
        if (apiDto == null && StringUtils.isNotBlank(request.getHeader("action"))) {
            apiDto = new ApiDto();
            apiDto.setBindAction(request.getHeader("action"));
            apiDto.setRequestMethod(request.getMethod());
        }

        if (apiDto == null) {
            response.setStatus(HttpStatus.OK.value());
            response.setContentType("application/json");
            Result result = new Result();
            result.setCode("403");
            result.setMessage("该接口尚未注册，请联系票易通集成管理员注册接口");
            response.getOutputStream().write(JacksonUtil.getInstance().toJson(result).getBytes());
            return false;
        } else if (StringUtils.isNotBlank(apiDto.getBindAction())) {
            //代理接口
            this.httpRequest(apiDto, request, response);
            this.afterCompletion(request, response, null, null);
            return false;
        } else if (StringUtils.isNotBlank(apiDto.getId()) && projectCacheHandler.getHttpDiyFlowCache().containsKey(apiDto.getId())) {
            //自定义接口编排
            if (this.diyFlowHandler == null) {
                response.setStatus(HttpStatus.OK.value());
                response.setContentType("application/json");
                Result result = new Result();
                result.setCode("403");
                result.setMessage("该接口定义异常，请联系票易通集成管理员配置接口");
                response.getOutputStream().write(JacksonUtil.getInstance().toJson(result).getBytes());
                return false;
            } else {
                this.diyFlowHandler.flowDiyHandle(apiDto, request, response);
            }

            this.afterCompletion(request, response, null, null);
            return false;
        }
        return true;
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {
        if (request.getHeader("Content-Encoding") != null && request.getHeader("Content-Encoding").indexOf("gzip") >= 0) {
            //gzip 不做拦截，不记录请求履历
            return;
        }

        long startTime = (long) request.getAttribute("startTime");
        String urlId = (String) request.getAttribute("urlId");
        ApiDto apiDto = projectCacheHandler.getApiCache().get(urlId);
        long cost = System.currentTimeMillis() - startTime;
        AccessRecord.AccessRecordBuilder builder = AccessRecord.builder()
                .costTime(cost)
                .requestTime(DateFormatUtils.format(startTime, "yyyyMMddHHmmssSSS"))
                .action((apiDto != null ? apiDto.getBindAction() : request.getRequestURI()))
                .apiId((apiDto != null ? apiDto.getId() : ""))
                .serverProjectId((apiDto != null ? apiDto.getProjectId() : ""))
                .requestMethod(request.getMethod()).status(response.getStatus())
                .serialNo(request.getHeader("serialNo"))
                .sourceIp(IPUtils.getIpAddr(request));

        AccessContentDto accessContentDto = new AccessContentDto();

        if (response instanceof CustomHttpServletResponseWrapper) {
            String responseBody = new String(((CustomHttpServletResponseWrapper) response).getBytes());
            accessContentDto.setResponseBody(responseBody);
            builder.reqDataLen(responseBody.getBytes(StandardCharsets.UTF_8).length);
        }

        Map<String, String> headerMap = new HashMap<>();
        Enumeration<String> headerEnu = request.getHeaderNames();
        while (headerEnu.hasMoreElements()) {
            String key = headerEnu.nextElement();
            if (EXCLUDES.contains(key.toUpperCase())) {
                continue;
            }
            headerMap.put(key, request.getHeader(key));
        }
        headerMap.put("RequestURI", request.getRequestURI());
        Map<String, String> paramMap = new HashMap<>();

        if (request.getContentType() != null && request.getContentType().contains(CONTENT_TYPE_MULTIPART)) {
            Enumeration<String> paramEnu = request.getParameterNames();
            while (paramEnu.hasMoreElements()) {
                String key = paramEnu.nextElement();
                paramMap.put(key, request.getParameter(key));
            }
        }

        accessContentDto.setRequestHeader(headerMap);
        accessContentDto.setRequestParam(paramMap);

        if (request.getInputStream() != null && !request.getInputStream().isFinished()) {
            StringBuilder reqBody = RequestUtil.getRequestBody(request);
            if (reqBody != null && reqBody.length() > 0) {
                String reqBodyStr = reqBody.toString();
                accessContentDto.setRequestBody(reqBodyStr);
                builder.reqDataLen(reqBodyStr.getBytes(StandardCharsets.UTF_8).length);
            }
        }

        AccessRecord accessRecord = builder.build();
        accessRecord.setAccessContent(accessContentDto);

        try {
//            log.info("{} cost:{},method:{} headers:{}", accessRecord.getAction(), accessRecord.getCostTime(), accessRecord.getRequestMethod());
            AccessRecordCache.pushRecord(accessRecord);
        } catch (Exception exception) {
            log.error("record error:{}", accessRecord.toString());
        }
    }


    Set<String> EXCLUDES = new HashSet<String>() {{
        add("CONTENT-LENGTH");
        add("USER-AGENT");
        add("HOST");
        add("X-FORWARDED-CLUSTER");
        add("WEB-SERVER-TYPE");
        add("ACCEPT-ENCODING");
        add("AUTHENTICATION");

    }};


    /**
     * 调用云端接口并回写
     *
     * @param
     * @return
     * @author xucuanhou
     * @date 2022/7/13
     */
    public void httpRequest(ApiDto apiDto, HttpServletRequest request, HttpServletResponse response) {
        Map<String, String> headers = new HashMap<>();
        headers.put("Authentication", httpConfig.getAuthentication());
        headers.put("action", apiDto.getBindAction());
        headers.put("rpcType", "http");
        headers.put("Accept-Encoding", "deft");

        Map<String, Object> params = new HashMap<>();

        JanusRequst janusRequst = RequestUtil.readFromRequest(request);

        String janusRequestStr = JacksonUtil.getInstance().toJson(janusRequst);
        //拓展映射
        if (CollectionUtils.isNotEmpty(apiDto.getHeaders())) {
            try {
                for (FieldMappingDto fieldMapping : apiDto.getHeaders()) {
                    if (StringUtils.isNotBlank(fieldMapping.getFieldValueName())) {
                        headers.put(fieldMapping.getFieldName(), JsonPath.read(janusRequestStr, "$." + fieldMapping.getFieldValueName()));
                    } else if (StringUtils.isNotBlank(fieldMapping.getDefaultValue())) {
                        headers.put(fieldMapping.getFieldName(), fieldMapping.getDefaultValue());
                    }
                }
            } catch (PathNotFoundException pnfe) {
                log.error("header json path not found", pnfe);
            }
        }
        if (MapUtils.isNotEmpty(janusRequst.getHeaders())) {
            for (Map.Entry<String, String> entry : janusRequst.getHeaders().entrySet()) {
                if (EXCLUDES.contains(entry.getKey().toUpperCase())) {
                    continue;
                }
                if (headers.containsKey(entry.getKey().toUpperCase())) {
                    continue;
                } else {
                    headers.put(entry.getKey(), entry.getValue());
                }
            }
        }

        if (MapUtils.isNotEmpty(janusRequst.getParams())) {
            for (Map.Entry<String, String> entry : janusRequst.getParams().entrySet()) {
                if (params.containsKey(entry.getKey())) {
                    continue;
                } else {
                    params.put(entry.getKey(), entry.getValue());
                }
            }
        }
        if (CollectionUtils.isNotEmpty(apiDto.getParams())) {
            try {
                for (FieldMappingDto paramMapping : apiDto.getParams()) {
                    if (StringUtils.isNotBlank(paramMapping.getFieldValueName())) {
                        headers.put(paramMapping.getFieldName(), JsonPath.read(janusRequestStr, "$." + paramMapping.getFieldValueName()));
                    } else if (StringUtils.isNotBlank(paramMapping.getDefaultValue())) {
                        headers.put(paramMapping.getFieldName(), paramMapping.getDefaultValue());
                    }
                }
            } catch (PathNotFoundException pnfe) {
                log.error("param json path not found", pnfe);
            }
        }

        String serailNo = StringUtils.isBlank(headers.get(Constant.KEY_SERIALNO)) ? headers.get(Constant.KEY_SERIALNO.toLowerCase()) : headers.get(Constant.KEY_SERIALNO);
        if (StringUtils.isBlank(serailNo)) {
            serailNo = "" + System.currentTimeMillis();
            headers.put(Constant.KEY_SERIALNO, serailNo);
        }


        if (request instanceof CustomHttpServletRequestWrapper) {
            ((CustomHttpServletRequestWrapper) request).addHeader(Constant.KEY_SERIALNO, serailNo);
        }

        try {
            JanusHttpUtil.ResponseCus remoteResult = null;
            String body = null;
            if (janusRequst.getBody() != null) {
                body = janusRequst.getBody() instanceof String ? (String) janusRequst.getBody() : JacksonUtil.getInstance().toJson(janusRequst.getBody());
            }

            if (StringUtils.isNotBlank(apiDto.getBodyExpression()) && StringUtils.isNotBlank(body)) {
                Json2Json json2Json = new Json2Json();
                body = json2Json.exchange(body, apiDto.getBodyExpression());
                log.info("转换后body:{}", body);
            }
            if (StringUtils.isBlank(body)) {
                body = "{}";
            }

            String httpMethod = StringUtils.isBlank(apiDto.getBindHttpMethod()) ? request.getMethod() : apiDto.getBindHttpMethod();

            if (httpMethod.equalsIgnoreCase(HttpGet.METHOD_NAME)) {
                remoteResult = JanusHttpUtil.doGetEntire(httpConfig.getUrl(), params, true, headers);
            } else if (httpMethod.equalsIgnoreCase(HttpPost.METHOD_NAME)) {
                remoteResult = JanusHttpUtil.doPostJsonEntire(httpConfig.getUrl(), body, headers, params);
            } else if (httpMethod.equalsIgnoreCase(HttpPut.METHOD_NAME) || httpMethod.equalsIgnoreCase(HttpPatch.METHOD_NAME)) {
                remoteResult = JanusHttpUtil.doPutPatchEntire(httpConfig.getUrl(), httpMethod, body, headers, params);
            } else if (httpMethod.equalsIgnoreCase(HttpDelete.METHOD_NAME)) {
                remoteResult = JanusHttpUtil.doDeleteEntire(httpConfig.getUrl(), headers, params);
            }
            response.setStatus(remoteResult.getStatus());
            response.setContentType("application/json");
            response.getOutputStream().write(remoteResult.getBody().getBytes());
        } catch (IOException ex) {
            log.error(httpConfig.getUrl() + "异常", ex);
        }

    }


}
