package com.xforceplus.janus.framework.record.portal;

import com.xforceplus.apollo.core.utils.UniqIdUtils;
import com.xforceplus.apollo.utils.HttpUtil;
import com.xforceplus.apollo.utils.JacksonUtil;
import com.xforceplus.janus.config.core.config.AccessConfig;
import com.xforceplus.janus.config.core.config.HttpConfig;
import com.xforceplus.janus.framework.dto.RecordPageParam;
import com.xforceplus.janus.framework.record.domain.AccessContent;
import com.xforceplus.janus.framework.record.domain.AccessContentDto;
import com.xforceplus.janus.framework.record.domain.AccessRecord;
import com.xforceplus.janus.framework.record.service.AccessRecordService;
import com.xforceplus.janus.framework.util.PageUtils;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.autoconfigure.web.ServerProperties;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import lombok.Setter;
import lombok.extern.slf4j.Slf4j;


/**
 * @Author: xuchuanhou
 * @Date:2022/3/17上午9:33
 */
@Slf4j
public class AccessRecordServiceImpl implements AccessRecordService {

    @Setter
    private ServerProperties serverProperties;

    @Setter
    private AccessRecordRepository accessRecordRepository;

    @Setter
    private AccessContentRepository accessContentRepository;

    @Setter
    private HttpConfig coreConfig;

    @Override
    public PageUtils queryForPage(RecordPageParam param) {
        return accessRecordRepository.queryForPage(param);
    }

    @Override
    public boolean saveBatch(List<AccessRecord> records) {
        if (CollectionUtils.isEmpty(records)) {
            return false;
        }

        List<AccessContent> contents = new ArrayList<>();
        records.forEach(r -> {
            if (StringUtils.isBlank(r.getId())) {
                r.setId(UniqIdUtils.getInstance().getUniqID());
            }

            if (StringUtils.isBlank(r.getContentId())) {
                r.setContentId(r.getId());
            }
            if (r.getAccessContent() != null) {
                contents.add(new AccessContent(r.getContentId(), JacksonUtil.getInstance().toJson(r.getAccessContent())));
            }

        });

        this.accessRecordRepository.saveBatch(records);

        if (CollectionUtils.isNotEmpty(contents)) {
            accessContentRepository.saveBatch(contents);
        }
        return true;
    }

    /***
     *
     * @param contentId
     * @param requestTime yyyy:MM:dd HH:mm:ss
     * @return
     */
    @Override
    public AccessContent queryByConentId(String contentId, String requestTime) {
        String yyyyMMd =getTabSuffix(requestTime);

        AccessContent content = accessContentRepository.getById(contentId, yyyyMMd);
        return content;
    }

    private String getTabSuffix(String requestTime){
        String yyyyMMd = requestTime.replaceAll("-", "").substring(0, 8);
        if(AccessConfig.SHARDING_YEAR.equals(coreConfig.getAccess().getShardType())){
            yyyyMMd=yyyyMMd.substring(0,4);
        }else if(AccessConfig.SHARDING_MONTH.equals(coreConfig.getAccess().getShardType())){
            yyyyMMd=yyyyMMd.substring(0,6);
        }
        return yyyyMMd;
    }

    /***
     *
     * @param id
     * @param requestTime yyyy:MM:dd HH:mm:ss
     * @return
     */
    @Override
    public AccessRecord getById(String id, String requestTime) {
        String yyyyMMd =getTabSuffix(requestTime);
        AccessRecord record = accessRecordRepository.getById(id, yyyyMMd);
        return record;
    }

    @Override
    public AccessRecord getRichRecord(String recordId, String requestTime) {
        String yyyyMMd =getTabSuffix(requestTime);
        AccessRecord record = accessRecordRepository.getById(recordId, yyyyMMd);
        if (record == null) {
            return record;
        }

        AccessContent content = queryByConentId(record.getContentId(), requestTime);
        if (content != null) {
            AccessContentDto contentDto = JacksonUtil.getInstance().fromJson(content.getContent(), AccessContentDto.class);
            record.setAccessContent(contentDto);
        }
        return record;

    }


    static Set<String> excludeHeaderSet = new HashSet<String>() {{
        add("content-length");
        add("user-agent");
        add("host");
        add("x-forwarded-cluster");
        add("web-server-type");
        add("RequestURI");
    }};


    /***
     * 履历重推
     *  @param recordId
     * @param requestTime yyyy:MM:dd HH:mm:ss
     * @return
     */
    @Override
    public boolean retryAccessRecord(String recordId, String requestTime) {
        AccessRecord record = getRichRecord(recordId, requestTime);
        return retryAccessRecord(record);
    }

    @Override
    public boolean retryAccessRecord(AccessRecord record) {
        HttpUtil.ResponseCus resp = null;
        Map<String, Object> params = null;
        Map<String, String> headers = null;
        String  url = "http://localhost:" + serverProperties.getPort() + record.getAccessContent().getRequestHeader().get("RequestURI");

        if (MapUtils.isNotEmpty(record.getAccessContent().getRequestHeader())) {
            headers = new HashMap<>();
            for (Map.Entry<String, String> entry : record.getAccessContent().getRequestHeader().entrySet()) {
                if (excludeHeaderSet.contains(entry.getKey().toLowerCase())) {
                    continue;
                }
                headers.put(entry.getKey(), entry.getValue());
            }

        }

        if (MapUtils.isNotEmpty(record.getAccessContent().getRequestParam())) {
            params = new HashMap<>();
            params.putAll(record.getAccessContent().getRequestParam());
        }


        try {
            if (HttpMethod.GET.name().equalsIgnoreCase(record.getRequestMethod())) {
                resp = HttpUtil.doGetEntire(url, params, false, headers);
            } else if (HttpMethod.POST.name().equalsIgnoreCase(record.getRequestMethod())) {
                resp = HttpUtil.doPostJsonEntire(url, record.getAccessContent().getRequestBody(), headers, params);
            } else if (HttpMethod.DELETE.name().equalsIgnoreCase(record.getRequestMethod())) {
                String result = HttpUtil.doDelete(url, headers, params);
            } else if (HttpMethod.PUT.name().equalsIgnoreCase(record.getRequestMethod())) {
                resp = HttpUtil.doPutPatchEntire(url, HttpMethod.PUT.name(), record.getAccessContent().getRequestBody(), headers, params);
            } else if (HttpMethod.PATCH.name().equalsIgnoreCase(record.getRequestMethod())) {
                resp = HttpUtil.doPutPatchEntire(url, HttpMethod.PATCH.name(), record.getAccessContent().getRequestBody(), headers, params);
            }
        } catch (Exception ex) {
            return false;
        }
        return resp.getStatus() == HttpStatus.OK.value();
    }

}
