package com.xforceplus.janus.framework.cmd;

import com.xforceplus.apollo.core.utils.UniqIdUtils;
import com.xforceplus.apollo.msg.SealedMessage;
import com.xforceplus.apollo.utils.ErrorUtil;
import com.xforceplus.apollo.utils.IOUtil;
import com.xforceplus.apollo.utils.JacksonUtil;
import com.xforceplus.janus.config.core.config.HttpConfig;
import com.xforceplus.janus.config.core.monitor.JanusUploader;
import com.xforceplus.janus.framework.dto.LogFileDescribeCacheDto;
import com.xforceplus.janus.framework.dto.LogFileDescribeDto;
import com.xforceplus.janus.framework.record.cache.SysCmdCache;
import com.xforceplus.janus.framework.record.domain.LocalClusterLogQuery;
import com.xforceplus.janus.framework.util.*;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Appender;
import org.apache.log4j.FileAppender;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.system.ApplicationHome;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.attribute.BasicFileAttributes;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

/**
 * 执行操作系统指令，并将结果反馈至云端
 *
 * @Author: xuchuanhou
 * @Date:2022/7/6下午1:46
 */
@Slf4j
@JanusCmdAnnotation(sourceType = "executeSysCmd")
public class SysCmdHandler implements IJanusCmdHandler, InitializingBean {

    @Setter
    private JanusUploader janusUploader;
    private static final int cmd_response_max = 500000;
    private static final int cmd_response_max_line = 200;
    private static final boolean testEnable = false;
    private static final int size = 5;
    private static final BlockingQueue<SealedMessage> tasks = new ArrayBlockingQueue(size);

    @Setter
    private HttpConfig httpConfig;

    /**
     * 先支持指定日期、单个关键字的查询； 查询当日日志，先查询未压缩文件，再查询压缩文件； 查询非当日日志，只查询压缩文件； 如果上次查询结果来源于压缩文件，可以支持分页；
     * 查询非压缩文件，尽可能多查一点数据
     *
     * @param sealedMessage
     */
    @Override
    public void doHandler(SealedMessage sealedMessage) {
        if (!httpConfig.isRemoteLog()) {
            return;
        }
        log.debug(JacksonUtil.getInstance().toJson(sealedMessage));
        String os_name = loadOsName();
        if (os_name == null || os_name.equals("") || os_name.equals(noSupport)) {
            log.info("您当前使用的系统不支持日志查询");
            String errorMsg = "您当前使用的系统不支持日志查询";
            sendError(sealedMessage, errorMsg);
            return;
        }
        if (!tasks.offer(sealedMessage)) {
            String errorMsg = "待执行的查询日志指令已超过" + size + "条";
            sendError(sealedMessage, errorMsg);
        }
    }

    private void doHandlerSealedMessage(SealedMessage sealedMessage) {
        long start = System.currentTimeMillis();
        try {
            int index = 0;
            int limit = 0;
            String startTime = sealedMessage.getHeader().getOthers().get("startTime");
            String endTime = sealedMessage.getHeader().getOthers().get("endTime");
            String indexStr = sealedMessage.getHeader().getOthers().get("index");
            String logFileName = sealedMessage.getHeader().getOthers().get("logFileName");
            String limitStr = sealedMessage.getHeader().getOthers().get("limit");
            String keyWords = sealedMessage.getHeader().getOthers().get("keyWords");
            String logContent = sealedMessage.getHeader().getOthers().get("logContent");
            String logLocation = sealedMessage.getHeader().getOthers().get("logLocation");
            String getLogsDir = sealedMessage.getHeader().getOthers().get("getLogsDir");
            if (StringUtils.isNotBlank(getLogsDir) && getLogsDir.equalsIgnoreCase("yes")) {
                sendError(sealedMessage, JacksonUtil.getInstance().toJson(logFileDescribeCache));
                return;
            }
            if (StringUtils.isBlank(logLocation)) {
                logLocation = "self";
            }
            boolean onlyQueryRecord = false;
            //只查消息和履历
            if (StringUtils.isNotBlank(logContent) && "true".equals(logContent)) {
                onlyQueryRecord = true;
            }
            if (StringUtils.isBlank(keyWords)) {
                throw new IllegalArgumentException("查询关键字不能为空");
            }
            if (StringUtils.isBlank(startTime) && StringUtils.isBlank(endTime)) {
                throw new IllegalArgumentException("查询时间不能为空");
            }
            if (StringUtils.isBlank(startTime) && StringUtils.isBlank(endTime)) {
                startTime = endTime;
            }
            if (startTime.length() != 19 && startTime.length() != 16 && startTime.length() != 13 && startTime.length() != 10) {
                throw new IllegalArgumentException("查询时间格式：yyyy-MM-dd HH:mm:ss,yyyy-MM-dd HH:mm,yyyy-MM-dd HH,yyyy-MM-dd");
            }
            if (StringUtils.isNotBlank(indexStr)) {
                index = Integer.valueOf(indexStr);
            }
            if (index < 0) {
                index = 0;
            }
            if (StringUtils.isNotBlank(limitStr)) {
                limit = Integer.valueOf(limitStr);
            }

            if (limit <= 0 || limit > cmd_response_max) {
                limit = cmd_response_max;
            }
            String ipAddress = address == null || address.size() == 0 ? "" : String.join(",", address);
            //属地集群子项目处理
            if (!logLocation.equals("self")) {
                String uuid = UniqIdUtils.getInstance().getUniqID();
                LocalClusterLogQuery localClusterLogQuery = new LocalClusterLogQuery();
                localClusterLogQuery.setStartTime(startTime);
                localClusterLogQuery.setEndTime(endTime);
                localClusterLogQuery.setIndex(index + "");
                localClusterLogQuery.setLimit(limit + "");
                localClusterLogQuery.setLogFileName(logFileName);
                localClusterLogQuery.setKeyWords(keyWords);
                localClusterLogQuery.setLogLocation(logLocation);
                localClusterLogQuery.setLogContent(logContent);
                localClusterLogQuery.setTimeStamp(System.currentTimeMillis() + "");
                localClusterLogQuery.setIpAddress(ipAddress);
                localClusterLogQuery.setUuid(uuid);
                localClusterLogQuery.setSourceType(sealedMessage.getHeader().getOthers().get("sourceType"));
                localClusterLogQuery.setCmdSerialKey(sealedMessage.getHeader().getOthers().get("cmdSerialKey"));
                SysCmdCache.put(localClusterLogQuery);
                Map resultMap = new HashMap();
                resultMap.put("logFileName", "");
                resultMap.put("readStatus", "false");
                resultMap.put("readResult", "等待属地子项目查询日志");
                resultMap.put("uuid", uuid);
                resultMap.put("ipAddress", ipAddress);
                resultMap.put("readInx", "0");
                resultMap.put("totalLength", "");
                sendSucc(resultMap, sealedMessage);
                return;
            }
            String startDay = startTime.substring(0, 10);
            List<LogFileDescribeDto> files1 = logFileDescribeCache.listLogFileByDateTime(startDay);
            //日志文件列表未加载
            int fileListNotLoaded = CollectionUtils.isEmpty(files1) ? 0 : files1.size();
            log.info("fileListNotLoaded: {} ", startDay, fileListNotLoaded);
            //如果还未读取到日志文件列表，读取默认的
            if (fileListNotLoaded <= 0) {
                String errorMsg = "还未读取到日志文件列表," + getLogsDir();
                sendError(sealedMessage, errorMsg);
            } else {
                if (searchFile(sealedMessage, index, limit, startTime, logFileName, keyWords, files1, onlyQueryRecord)) {
                    return;
                }
            }
            String errorMsg = "没有查询到日志";
            sendError(sealedMessage, errorMsg);
        } catch (Exception e) {
            log.error(ErrorUtil.getStackMsg(e));
            String errorMsg = "查询日志出现异常";
            sendError(sealedMessage, errorMsg);
        } finally {
            log.info("doHandlerSealedMessage costTime : {}", (System.currentTimeMillis() - start));
        }
    }

    private boolean searchFile(SealedMessage sealedMessage,
                               int index,
                               int limit,
                               String startTime,
                               String logFileName,
                               String keyWords,
                               List<LogFileDescribeDto> files1,
                               boolean onlyQueryRecord) {
        try {
            if (CollectionUtils.isEmpty(files1)) {
                return false;
            }
            List<LogFileDescribeDto> searchFiles = new ArrayList<>();
            if (StringUtils.isNotBlank(logFileName)) {
                String src = logFileDescribeCache.getDec2SrcMap().get(logFileName);
                if (StringUtils.isNotBlank(src)) {
                    LogFileDescribeDto logFileDescribeDto = logFileDescribeCache.getlogFileMap().get(src);
                    if (logFileDescribeDto != null) {
                        searchFiles = files1.stream().filter(e -> {
                            if (e.getLogFileFullPath().compareTo(logFileDescribeDto.getLogFileFullPath()) == 0) {
                                return true;
                            } else if (e.getLastModifiedTime() - logFileDescribeDto.getLastModifiedTime() <= 0) {
                                return true;
                            } else if (e.getCreationTime() - logFileDescribeDto.getCreationTime() <= 0) {
                                return true;
                            } else {
                                return false;
                            }
                        }).collect(Collectors.toList());
                    }
                }
            }
            if (CollectionUtils.isEmpty(searchFiles)) {
                searchFiles = files1;
            }
            for (int i = 0; i < searchFiles.size(); i++) {
                if (i == searchFiles.size() - 1 || i % 3 == 2) {
                    if (searchFileOnePage(sealedMessage, index, limit, startTime, logFileName, keyWords, onlyQueryRecord, searchFiles))
                        return true;
                }
            }
        } catch (Exception e) {
            log.error("userIndex error {}", ErrorUtil.getStackMsg(e));
        }
        return false;
    }

    private boolean searchFileOnePage(SealedMessage sealedMessage, int index, int limit, String startTime, String logFileName, String keyWords, boolean onlyQueryRecord, List<LogFileDescribeDto> searchFiles) {
        List<FutureTask<Map<String, String>>> tasks = new ArrayList<>();
        if (CollectionUtils.isNotEmpty(searchFiles)) {
            for (LogFileDescribeDto logFileDescribeDto : searchFiles) {
                String fileName = logFileDescribeDto.getLogFileFullPath();

                String cmd = getCmd(startTime, keyWords, fileName, onlyQueryRecord);
                Integer size = FileCmdCache.get(cmd);
                boolean cache = (logFileDescribeDto.getChangeFlag() == 2) && (size == null);
                int finalLimit = limit;
                if (size == null || size > 0) {
                    if (testEnable) {
                        log.info("testEnable {},{}", tasks.size(), cmd);
                    }
                    Callable<Map<String, String>> callable = () -> doCmd(cmd, fileName, cache, 0, finalLimit);
                    FutureTask<Map<String, String>> task = new FutureTask<>(callable);
                    tasks.add(task);
                }
            }
        }
        if (CollectionUtils.isNotEmpty(tasks)) {
            for (FutureTask<Map<String, String>> task : tasks) {
                ThreadPoolCmdUtils.execute(task);
            }
        }
        Map<String, Map<String, String>> resultMap = new HashMap<>();
        for (FutureTask<Map<String, String>> task : tasks) {
            try {
                Map<String, String> cmdResultMap = task.get();
                if ("true".equals(cmdResultMap.get("readStatus"))) {
                    resultMap.put(cmdResultMap.get("logFileName"), cmdResultMap);
                }
            } catch (InterruptedException e) {
                log.error("InterruptedException[{}]", e);
            } catch (ExecutionException e) {
                log.error("ExecutionException[{}]", e);
            } catch (Exception e) {
                log.error("exception[{}]", e);
            } catch (Error e) {
                log.error("exception[{}]", e);
            }
        }
        if (CollectionUtils.isNotEmpty(searchFiles)) {
            for (LogFileDescribeDto logFileDescribeDto : searchFiles) {
                String fileName = logFileDescribeDto.getLogFileFullPath();
                Map<String, String> cmdResultMap = resultMap.get(fileName);
                log.info("fileName: {} ,{} ", fileName, cmdResultMap != null && "true".equals(cmdResultMap.get("readStatus")));
                if (cmdResultMap != null && "true".equals(cmdResultMap.get("readStatus"))) {
                    log.info("send1Succ {}", fileName);
                    sendSucc(cmdResultMap, sealedMessage);
                    return true;
                }
            }
        }
        return false;
    }


    private void sendError(SealedMessage sealedMessage, String errorMsg) {
        Map<String, String> result = new HashMap<>();
        result.put("readStatus", "false");
        result.put("readResult", errorMsg);
        result.put("readInx", "0");
        result.put("totalLength", "0");
        result.put("logFileName", "");
        Map<String, Object> resultMap = new HashMap<>();
        resultMap.put("code", "0");
        resultMap.put("message", "FAILED");
        resultMap.put("executeSysCmd", "executeSysCmd");
        resultMap.put("result", result);
        sendMsg(resultMap, sealedMessage);
    }

    private void sendSucc(Map<String, String> result, SealedMessage sealedMessage) {
        Map<String, Object> resultMap = new HashMap<>();
        resultMap.put("code", "1");
        resultMap.put("message", "FINISHED");
        resultMap.put("executeSysCmd", "executeSysCmd");
        resultMap.put("result", result);
        sendMsg(resultMap, sealedMessage);
    }

    private void sendMsg(Map<String, Object> resultMap, SealedMessage sealedMessage) {
        String resultstr = JacksonUtil.getInstance().toJson(resultMap);
        janusUploader.sendJanusCmdResult(sealedMessage.getHeader().getOthers().get("sourceType"), sealedMessage.getHeader().getOthers().get("cmdSerialKey"), resultstr);
    }


    //    cat /logs/apollo | grep  '2022-07-21 10:05'  | grep 'org.springframework.scheduling.concurrent'
    //    zcat /logs/apollo.2022-07-20-下午.gz | grep  '2022-07-20 13:34'  | grep 'com.xforceplus.apollo.janus.standalone.utils.userCenter.JwtEncoder-90'

    private String getCmd(String startTime, String keyWords, String fileName, boolean onlyQueryRecord) {
        String os_name = loadOsName();
        if (os_name == null || os_name.equals("") || os_name.equals(noSupport)) {
            log.info("您当前使用的系统不支持日志查询");
            return "";
        }
        StringBuilder sb = new StringBuilder();
        if (os_name.equals(Linux)) {
            if (fileName.endsWith(".gz")) {
                sb.append("zcat ").append(fileName).append(" | grep  '").append(startTime).append("' ");
                buildLinuxKeyWords(keyWords, sb);
            } else {
                sb.append("cat ").append(fileName).append(" | grep  '").append(startTime).append("' ");
                buildLinuxKeyWords(keyWords, sb);
            }
            sb.append("  | grep  -v 'com.xforceplus.janus.framework.cmd.SysCmdHandler' ")
                    .append("  | grep  -v 'executeSysCmd' ");
            if (onlyQueryRecord) {
                sb.append("  | grep  'com.xforceplus.janus.framework.record.portal.LogAccessRecordConsumer' ")
                        .append("  | grep  -v 'com.xforceplus.janus.config.core.monitor.JanusUploader'");
            }
        } else if (os_name.equals(Windows)) {
            sb.append("cmd /c   ").append(" find  \"").append(startTime).append("\"  ").append(fileName);
            buildWindowsKeyWords(keyWords, sb);
            sb.append("  | find  /v \"com.xforceplus.janus.framework.cmd.SysCmdHandler\" ")
                    .append("  | find  /v \"executeSysCmd\" ");
            if (onlyQueryRecord) {
                sb.append("  | find  \"com.xforceplus.janus.framework.record.portal.LogAccessRecordConsumer\"  ")
                        .append("  | find  /v \" com.xforceplus.janus.config.core.monitor.JanusUploader\" ");
            }
        }
        return sb.toString();
    }

    private void buildLinuxKeyWords(String keyWords, StringBuilder sb) {
        if (keyWords.contains(",")) {
            String[] splits = keyWords.split(",");
            for (String split : splits) {
                if (StringUtils.isNotBlank(split)) {
                    sb.append(" | grep  '").append(split.trim()).append("' -A2 ");
                }
            }
        } else {
            sb.append(" | grep  '").append(keyWords.trim()).append("' -A2 ");
        }
    }

    private void buildWindowsKeyWords(String keyWords, StringBuilder sb) {
        if (keyWords.contains(",")) {
            String[] splits = keyWords.split(",");
            for (String split : splits) {
                if (StringUtils.isNotBlank(split)) {
                    sb.append(" | find  \"").append(split.trim()).append("\" ");
                }
            }
        } else {
            sb.append(" | find  \"").append(keyWords.trim()).append("\"  ");
        }
    }


    /**
     * @param cmd      执行的命令
     * @param fileName 查询的文件
     * @param cache    是否缓存 查询结果的大小
     * @param index    报文偏移量
     * @param limit    返回的报文大小限制
     * @return
     */
    private Map<String, String> doCmd(String cmd, String fileName, boolean cache, int index, int limit) {
        Map<String, String> resultMap = new HashMap<>();
        try {
            log.info("CmdPrint10 {},cmd:{},cache:{}", fileName, cmd, cache);
            resultMap.put("logFileName", fileName);
            StringBuilder cmdResponse = new StringBuilder();
            Process process = null;
            BufferedReader bufferedReader = null;
            BufferedReader errorReader = null;
            int LastLength = 0;
            int length = 0;
            try {
                String os_name = loadOsName();
                if (os_name == null || os_name.equals("") || os_name.equals(noSupport)) {
                    log.info("您当前使用的系统不支持日志查询");
                    throw new IllegalArgumentException("您当前使用的系统不支持日志查询");
                }
                if (os_name.equals(Linux)) {
                    process = Runtime.getRuntime().exec(new String[]{"/bin/sh", "-c", cmd});
                    bufferedReader = new BufferedReader(new InputStreamReader(process.getInputStream()));
                    errorReader = new BufferedReader(new InputStreamReader(process.getErrorStream()));
                    boolean proresult = process.waitFor(4, TimeUnit.SECONDS);
                } else if (os_name.equals(Windows)) {
                    process = Runtime.getRuntime().exec(cmd);
                    bufferedReader = new BufferedReader(new InputStreamReader(process.getInputStream()));
                    errorReader = new BufferedReader(new InputStreamReader(process.getErrorStream()));
                    boolean proresult = process.waitFor(4, TimeUnit.SECONDS);
                } else {
                    log.info("您当前使用的系统不支持日志查询");
                    throw new IllegalArgumentException("您当前使用的系统不支持日志查询");
                }
                String cont = null;
                int line = 0;
                boolean full = false;
                int readInx = index;
                while ((cont = bufferedReader.readLine()) != null) {
                    LastLength = length;
                    length += cont.length();
                    if (!full && index < length) {
                        if (index >= LastLength) {
                            String subString = cont.substring(index - LastLength);
                            if (StringUtils.isNotBlank(subString)) {
                                if (subString.length() < (limit - cmdResponse.length())) {
                                    cmdResponse.append(subString).append("\n\r");
                                } else {
                                    cmdResponse.append(subString.substring(0, limit - cmdResponse.length())).append("\n\r");
                                }
                            }
                        } else {
                            if (StringUtils.isNotBlank(cont)) {
                                if (cont.length() < (limit - cmdResponse.length())) {
                                    cmdResponse.append(cont).append("\n\r");
                                } else {
                                    cmdResponse.append(cont.substring(0, limit - cmdResponse.length())).append("\n\r");
                                }
                            }

                        }
                        readInx = length;
                    }
                    if (cmdResponse.length() >= limit) {
                        full = true;
                        if (!cache) {
                            break;
                        }
                    }
                    if (line > cmd_response_max_line) {
                        full = true;
                        if (!cache) {
                            break;
                        }
                    }
                    line++;
                }
                resultMap.put("readInx", "" + readInx);
                String errorLine = null;
                if (cmdResponse.length() <= 0) {
                    while ((errorLine = errorReader.readLine()) != null) {
                        cmdResponse.append(errorLine).append("\n\r");
                    }
                    if (cmdResponse.length() >= 0) {
                        resultMap.put("readStatus", "false");
                        resultMap.put("readResult", "");
                        log.error("cmd:{}  run error,{}", cmd, cmdResponse);
                    }
                } else {
                    resultMap.put("readResult", cmdResponse.toString());
                    resultMap.put("readStatus", "true");
                }

            } catch (Exception exception) {
                log.error("cmd:{}  run error,{}", cmd, ErrorUtil.getStackMsg(exception));
                resultMap.put("readStatus", "false");
                resultMap.put("readResult", "");
            } catch (Error Error) {
                log.error("cmd:{}  run error,{}", cmd, ErrorUtil.getStackMsg(Error));
                resultMap.put("readStatus", "false");
                resultMap.put("readResult", "");
            } finally {
                if (errorReader != null) {
                    IOUtil.closeQuietly(errorReader);
                }

                if (bufferedReader != null) {
                    IOUtil.closeQuietly(bufferedReader);
                }
                if (process != null) {
                    process.destroy();
                }
            }
            if (resultMap.get("readStatus") == null || "false".equals(resultMap.get("readStatus"))) {
                resultMap.put("readInx", "" + index);
                length = 0;
            }
            if (cache) {
                //缓存一天
                FileCmdCache.set(cmd, length, 1000 * 60 * 60 * 24);
            }
            if (testEnable) {
                log.info("cmdResponse.length() :{},limit: {},readStatus:{},cmd {}", cmdResponse.length(), limit, resultMap.get("readStatus"), cmd);
            }
            resultMap.put("totalLength", "" + length);
            return resultMap;
        } catch (Exception e) {
            log.error("cmd:{}  run error,{}", cmd, ErrorUtil.getStackMsg(e));
        } catch (Error er) {
            log.error("cmd:{}  run error,{}", cmd, ErrorUtil.getStackMsg(er));
        }
        return resultMap;

    }

    private String doCmd(String cmd) {
        String cont = "";
        try {
            log.info("testEnable cmd:{}", cmd);
            Process process = null;
            BufferedReader bufferedReader = null;
            BufferedReader errorReader = null;
            try {
                process = Runtime.getRuntime().exec(new String[]{"/bin/sh", "-c", cmd});
                bufferedReader = new BufferedReader(new InputStreamReader(process.getInputStream()));
                errorReader = new BufferedReader(new InputStreamReader(process.getErrorStream()));
                boolean proresult = process.waitFor(4, TimeUnit.SECONDS);
                cont = bufferedReader.readLine();

            } catch (Exception exception) {
                log.error("cmd:{}  run error,{}", cmd, ErrorUtil.getStackMsg(exception));
            } catch (Error Error) {
                log.error("cmd:{}  run error,{}", cmd, ErrorUtil.getStackMsg(Error));
            } finally {
                if (errorReader != null) {
                    IOUtil.closeQuietly(errorReader);
                }
                if (bufferedReader != null) {
                    IOUtil.closeQuietly(bufferedReader);
                }
                if (process != null) {
                    process.destroy();
                }
            }
            if (testEnable) {
                log.info("testEnable cont :{},cmd {}", cont, cmd);
            }
        } catch (Exception e) {
            log.error("cmd:{}  run error,{}", cmd, ErrorUtil.getStackMsg(e));
        } catch (Error er) {
            log.error("cmd:{}  run error,{}", cmd, ErrorUtil.getStackMsg(er));
        }
        return cont;
    }

    private Set<String> forbidCmds = new HashSet<String>() {{
        add("rm ");
        add("rmdir ");
        add("-delete");
        add("awk ");
        add("rmdir ");
    }};

    private String allowCmdPattern = "^(ls|cd|grep|cat|zcat|pwd).*";

    Pattern pattern = Pattern.compile(allowCmdPattern);

    private boolean validateCmd(String cmd) {
        if (StringUtils.isBlank(cmd)) {
            return false;
        }

        for (String forbidCmd : forbidCmds) {
            if (cmd.contains(forbidCmd)) {
                return false;
            }
        }
        Matcher matcher = pattern.matcher(cmd);
        return matcher.find();
    }


    private String deployLibDir() {
        String workDir = "";
        try {
            //第五种
            ApplicationHome h = new ApplicationHome(getClass());
            File jarF = h.getSource();
            workDir = jarF.getParentFile().toString();
            log.info("checklogFiles, path:{} ", workDir);
        } catch (Exception e) {
            log.info("checklogFiles, path  error:{} ", ErrorUtil.getStackMsg(e));
            workDir = "";
        }
        return workDir;
    }

    private static final String log_folder = "logs";
    private static final String no_zip_log_folder = "unZipLogs";
    private volatile LogFileDescribeCacheDto logFileDescribeCache = new LogFileDescribeCacheDto();
    private volatile List<String> address = new LinkedList<>();

    private String deployJarDir() {
        String workDir = "";
        try {
            workDir = System.getProperty("user.dir");
            return workDir;
        } catch (Exception e) {
            log.info("checklogFiles, path  error:{} ", ErrorUtil.getStackMsg(e));
            workDir = "";
        }
        return workDir;
    }

    public String getLogsDir() {
        //配置的目录优先
        if (StringUtils.isNotBlank(httpConfig.getCusLogDir())) {
            return httpConfig.getCusLogDir();
        }
        try {
            Logger rootLogger = Logger.getRootLogger();
            Enumeration allAppenders = rootLogger.getAllAppenders();
            while (allAppenders.hasMoreElements()) {
                Appender appender = (Appender) allAppenders.nextElement();
                if (appender instanceof FileAppender) {
                    FileAppender fileAppender = (FileAppender) appender;
                    String fileName = fileAppender.getFile();
                    File file = new File(fileName);
                    String canonicalPath = file.getCanonicalPath();
                    canonicalPath = canonicalPath.substring(0, canonicalPath.lastIndexOf(File.separator));
                    log.info("canonicalPath= {}", canonicalPath);
                    return canonicalPath;
                }
            }
        } catch (Exception e) {
            log.error(ErrorUtil.getStackMsg(e));
        } catch (Error e) {
            log.error(ErrorUtil.getStackMsg(e));
        }
        return deployJarDir() + File.separator + log_folder;
    }

    private String osName = "";
    private final String Windows = "Windows";
    private final String Linux = "Linux";
    private final String noSupport = "noSupport";

    private String loadOsName() {
        if ("".equals(osName)) {
            String name = System.getProperty("os.name");
            log.info("已检测到您当前使用的系统为：" + name);
            if (osName != null) {
                if (name.startsWith("Windows")) {
                    osName = Windows;
                } else if (name.startsWith("Linux") || name.startsWith("Mac")) {
                    osName = Linux;
                } else {
                    osName = "noSupport";
                    log.info("您当前使用的系统不支持日志查询");
                }
            } else {
                osName = "noSupport";
                log.info("您当前使用的系统不支持日志查询");
            }

        }
        return osName;

    }

    /**
     * 每两分钟查一下日志目录下的文件
     */

    public void checklogFiles() {
        if (!httpConfig.isRemoteLog()) {
            return;
        }
        new Thread(() -> {
            while (true) {
                try {

                    if (testEnable) {
                        TimeUnit.SECONDS.sleep(10);
                    } else {
                        TimeUnit.MINUTES.sleep(3);
                    }
                    String os_name = loadOsName();
                    if (os_name == null || os_name.equals("") || os_name.equals(noSupport)) {
                        log.info("您当前使用的系统不支持日志查询");
                        return;
                    }
                    doChecklogFiles();

                } catch (Exception e) {
                    log.error("checklogFiles error {}", ErrorUtil.getStackMsg(e));
                } finally {
                    if (testEnable) {
                        try {
                            TimeUnit.MINUTES.sleep(2);
                        } catch (InterruptedException e) {
                            e.printStackTrace();
                        }
                    } else {
                        try {
                            TimeUnit.MINUTES.sleep(18);
                        } catch (InterruptedException e) {
                            e.printStackTrace();
                        }
                    }
                }
            }
        }).start();
    }


    @Override
    public void afterPropertiesSet() {
        String os_name = loadOsName();
        this.checklogFiles();
        if (os_name == null || os_name.equals("") || os_name.equals(noSupport)) {
            log.info("您当前使用的系统不支持日志查询");
        } else {
            new Thread(() -> {
                while (true) {
                    try {
                        boolean hasMessage = false;
                        SealedMessage sealedMessage = tasks.take();
                        if (sealedMessage != null) {
                            hasMessage = true;
                            doHandlerSealedMessage(sealedMessage);
                        }
                        if (!hasMessage) {
                            TimeUnit.SECONDS.sleep(1);
                        }
                    } catch (Exception e) {
                        log.info(ErrorUtil.getStackMsg(e));
                    }
                }
            }, "SysCmdTaskExecutor").start();
        }
    }


    private void doChecklogFiles() {
        address = IPUtils.getIpAddress();
        String os_name = loadOsName();
        if (os_name == null || os_name.equals("") || os_name.equals(noSupport)) {
            log.info("您当前使用的系统不支持日志查询");
            return;
        }
        long start = System.currentTimeMillis();
        String logFolder = getLogsDir();
        File folder = new File(logFolder);
        if (!folder.exists()) {
            folder.mkdirs();
        }
        String logMoveFolder = deployJarDir() + File.separator + no_zip_log_folder;
        File moveFolder = new File(logMoveFolder);
        if (!moveFolder.exists()) {
            moveFolder.mkdirs();
        }
        File[] oldMoveFiles = moveFolder.listFiles();
        log.info("logOrgFolder,{}", logFolder);
        log.info("logMoveFolder,{}", logMoveFolder);
        String logFileDescribeCachePath = logMoveFolder + File.separator + "logFileDescribeCache.txt";
        LogFileDescribeCacheDto oldCache = MyFileUtils.readCache(logFileDescribeCachePath);
        LogFileDescribeCacheDto newCache = new LogFileDescribeCacheDto();
        File[] files = folder.listFiles();
        List<File> logFiles = new ArrayList<>();
        List<String> fileTimes = new ArrayList<>();
        if (files != null && files.length > 0) {
            for (File file : files) {
                try {
                    logFiles.add(file);
                    BasicFileAttributes attr1 = Files.readAttributes(file.toPath(), BasicFileAttributes.class);
                    long creationTime = attr1.creationTime().toMillis();
                    long lastModifiedTime = attr1.lastModifiedTime().toMillis();
                    String format = "yyyy-MM-dd HH:mm:ss";
                    SimpleDateFormat sf = new SimpleDateFormat(format);
                    String createTime = sf.format(new Date(creationTime));
                    String modifiedTime = sf.format(new Date(lastModifiedTime));
                    fileTimes.add(createTime);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
            //创建时间倒序
            Collections.sort(fileTimes, (String ct1, String ct2) -> {
                return ct2.compareTo(ct1);
            });
            Collections.sort(logFiles, (File file1, File file2) -> {
                try {
                    BasicFileAttributes attr1 = Files.readAttributes(file1.toPath(), BasicFileAttributes.class);
                    BasicFileAttributes attr2 = Files.readAttributes(file2.toPath(), BasicFileAttributes.class);
                    long creationTime1 = attr1.creationTime().toMillis();
                    long lastModifiedTime1 = attr1.lastModifiedTime().toMillis();
                    long creationTime2 = attr2.creationTime().toMillis();
                    long lastModifiedTime2 = attr2.lastModifiedTime().toMillis();
                    if (lastModifiedTime2 != lastModifiedTime1) {
                        return lastModifiedTime2 > lastModifiedTime1 ? 1 : -1;
                    } else if (creationTime2 != creationTime1) {
                        return creationTime2 > creationTime1 ? 1 : -1;
                    } else {
                        return file2.getAbsoluteFile().compareTo(file1.getAbsoluteFile());
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
                return file2.getAbsoluteFile().compareTo(file1.getAbsoluteFile());
            });
            //每次最多解析20个文件
            int maxParseFileSize = 20;
            int hasParseFileSize = 0;
            for (File file : logFiles) {
                if (hasParseFileSize >= maxParseFileSize) {
                    continue;
                }
                try {
                    if (file.isFile()) {
                        String fileName = file.getName();
                        String filePath = file.getAbsolutePath();
                        LogFileDescribeDto newLogFileDescribe = oldCache.getTargetFile(filePath);
                        BasicFileAttributes attr = Files.readAttributes(file.toPath(), BasicFileAttributes.class);
                        long creationTime = attr.creationTime().toMillis();
                        long lastModifiedTime = attr.lastModifiedTime().toMillis();
                        //压缩文件 就检测一下文件中日志内容的起始时间和截止时间
                        if (fileName.indexOf(".gz") > 0) {
                            if (newLogFileDescribe == null) {
                                hasParseFileSize++;
                                if (os_name.equals(Windows)) {
                                    newLogFileDescribe = buildLogFileDescribeInWindows(logMoveFolder, file, creationTime, lastModifiedTime, fileTimes);
                                } else {
                                    newLogFileDescribe = buildLogFileDescribeInLinux(logMoveFolder, file, creationTime, lastModifiedTime, fileTimes);
                                }
                            }
                        } else {
                            if (newLogFileDescribe == null) {
                                newLogFileDescribe = new LogFileDescribeDto();
                                newLogFileDescribe.setLogFileFullPath(filePath);
                                newLogFileDescribe.setChangeFlag(1);
                                newLogFileDescribe.setCreationTime(creationTime);
                                newLogFileDescribe.setLastModifiedTime(lastModifiedTime);
                                hasParseFileSize++;
                                if (httpConfig.isDecisionContentTimeUseFileTime()) {
                                    decisionContentTimeUseFileTime(creationTime, lastModifiedTime, fileTimes, newLogFileDescribe);
                                } else {
                                    if (os_name.equals(Windows)) {
                                        detectionContentTimeInWindows(newLogFileDescribe);
                                    } else {
                                        detectionContentTimeInLinux(newLogFileDescribe);
                                    }
                                }
                                String today = DateUtils.format(new Date(), DateUtils.DATE_PATTERN);
                                List<String> dateTimes = newLogFileDescribe.getDateTimes();
                                if (dateTimes != null && dateTimes.size() > 0) {
                                    String lastBusinessDay = dateTimes.get(dateTimes.size() - 1);
                                    //最后的业务日志非当天 判断此日志文件 不会在变更
                                    boolean isToday = today.equals(lastBusinessDay);
                                    if (!isToday) {
                                        newLogFileDescribe.setChangeFlag(2);
                                    }
                                }

                            } else {
                                // 创建时间或者修改时间有变动，需要重新探测文件内容的起始时间和截止时间
                                if (newLogFileDescribe.getCreationTime() != creationTime || newLogFileDescribe.getLastModifiedTime() != lastModifiedTime) {
                                    hasParseFileSize++;
                                    newLogFileDescribe.setCreationTime(creationTime);
                                    newLogFileDescribe.setLastModifiedTime(lastModifiedTime);
                                    newLogFileDescribe.setChangeFlag(1);
                                    if (httpConfig.isDecisionContentTimeUseFileTime()) {
                                        decisionContentTimeUseFileTime(creationTime, lastModifiedTime, fileTimes, newLogFileDescribe);
                                    } else {
                                        if (os_name.equals(Windows)) {
                                            detectionContentTimeInWindows(newLogFileDescribe);
                                        } else {
                                            detectionContentTimeInLinux(newLogFileDescribe);
                                        }
                                    }
                                    String today = DateUtils.format(new Date(), DateUtils.DATE_PATTERN);
                                    List<String> dateTimes = newLogFileDescribe.getDateTimes();
                                    if (dateTimes != null && dateTimes.size() > 0) {
                                        String lastBusinessDay = dateTimes.get(dateTimes.size() - 1);
                                        //最后的业务日志非当天 判断此日志文件 不会在变更
                                        boolean isToday = today.equals(lastBusinessDay);
                                        if (!isToday) {
                                            newLogFileDescribe.setChangeFlag(2);
                                        }
                                    }
                                }
                            }
                        }
                        newCache.putTargetFile(filePath, newLogFileDescribe);
                    }
                } catch (Exception e) {
                    log.info("checklogFiles error: {} ", ErrorUtil.getStackMsg(e));
                } catch (Error e) {
                    log.info("checklogFiles error: {} ", ErrorUtil.getStackMsg(e));
                }
            }
        }
        try {
            if (oldMoveFiles != null && oldMoveFiles.length > 0) {
                for (File oldMoveFile : oldMoveFiles) {
                    if (oldMoveFile.getAbsolutePath().indexOf("logFileDescribeCache") > -1) {
                        continue;
                    }
                    String srcPath = newCache.getDec2SrcMap().get(oldMoveFile.getAbsolutePath());
                    if (StringUtils.isBlank(srcPath) && oldMoveFile != null && oldMoveFile.exists() && oldMoveFile.isFile()) {
                        oldMoveFile.delete();
                    }
                }

            }
        } catch (Exception e) {
            log.info("checklogFiles error: {} ", ErrorUtil.getStackMsg(e));
        } catch (Error e) {
            log.info("checklogFiles error: {} ", ErrorUtil.getStackMsg(e));
        }
        MyFileUtils.writeCache(logFileDescribeCachePath, newCache);
        logFileDescribeCache = newCache;
        long end = System.currentTimeMillis();
        long cost = end - start;
        log.info("checklogFiles cost:{} ", cost);
    }

    private LogFileDescribeDto buildLogFileDescribeInWindows(String logMoveFolder, File file, long creationTime, long lastModifiedTime, List<String> fileTimes) {
        LogFileDescribeDto oldLogFileDescribe = new LogFileDescribeDto();
        String newFileName = System.currentTimeMillis() + UUID.randomUUID().toString().replaceAll("-", "");
        String newFilePath = logMoveFolder + File.separator + newFileName + ".txt";
        MyFileUtils.writeUnGzip(file.getAbsolutePath(), newFilePath);
        oldLogFileDescribe.setLogFileFullPath(newFilePath);
        oldLogFileDescribe.setChangeFlag(2);
        oldLogFileDescribe.setCreationTime(creationTime);
        oldLogFileDescribe.setLastModifiedTime(lastModifiedTime);
        //使用日志文件的创建时间来确定打印日志时间
        if (httpConfig.isDecisionContentTimeUseFileTime()) {
            decisionContentTimeUseFileTime(creationTime, lastModifiedTime, fileTimes, oldLogFileDescribe);
        } else {
            detectionContentTimeInWindows(oldLogFileDescribe);
        }
        return oldLogFileDescribe;
    }

    private static void detectionContentTimeInWindows(LogFileDescribeDto oldLogFileDescribe) {
        try {
            List<String> dateTimes = new ArrayList<>();
            oldLogFileDescribe.setDateTimes(dateTimes);
            Scanner scanner = new Scanner(new File(oldLogFileDescribe.getLogFileFullPath()));
            String firstLine = "";
            String lastLine = "";
            String startDateTime = "";
            String endDateTime = "";
            int readIndex = 0;
            while (scanner.hasNextLine()) {
                lastLine = scanner.nextLine();
                if (readIndex == 0) {
                    firstLine = lastLine;
                }
                readIndex++;
            }
            scanner.close();
            if (readIndex > 0 && StringUtils.isNotBlank(firstLine)) {
                Pattern p = Pattern.compile("(\\d{4})-(\\d{1,2})-(\\d{1,2})");
                Matcher m = p.matcher(firstLine);
                while (m.find()) {
                    startDateTime = m.group(0);
                    break;
                }
                if (readIndex > 1 && StringUtils.isNotBlank(lastLine)) {
                    m = p.matcher(lastLine);
                    while (m.find()) {
                        endDateTime = m.group(0);
                        break;
                    }
                }
            }
            if (StringUtils.isNotBlank(startDateTime) && StringUtils.isNotBlank(endDateTime)) {
                if (startDateTime.equals(endDateTime)) {
                    dateTimes.add(startDateTime);
                } else {
                    SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
                    List<String> dates = DateUtils.getRangeDate(format.parse(startDateTime), format.parse(endDateTime), Calendar.DAY_OF_MONTH, DateUtils.DATE_PATTERN);
                    if (CollectionUtils.isNotEmpty(dates)) {
                        oldLogFileDescribe.setDateTimes(dates);
                    } else {
                        String businessDate = new SimpleDateFormat("yyyy-MM-dd").format(new Date(oldLogFileDescribe.getLastModifiedTime() - 2000));
                        dateTimes.add(businessDate);
                    }
                }
            } else if (StringUtils.isNotBlank(startDateTime)) {
                dateTimes.add(startDateTime);
            } else if (StringUtils.isNotBlank(endDateTime)) {
                dateTimes.add(endDateTime);
            } else {
                String businessDate = new SimpleDateFormat("yyyy-MM-dd").format(new Date(oldLogFileDescribe.getLastModifiedTime() - 2000));
                dateTimes.add(businessDate);
            }
        } catch (Exception e) {
            log.error("{}", ErrorUtil.getStackMsg(e));
            oldLogFileDescribe.setDateTimes(new ArrayList<>());
        }
    }

    private LogFileDescribeDto buildLogFileDescribeInLinux(String logMoveFolder, File file, long creationTime, long lastModifiedTime, List<String> fileTimes) {
        LogFileDescribeDto oldLogFileDescribe = new LogFileDescribeDto();
        if (httpConfig.isCopLogFile()) {
            String newFileName = System.currentTimeMillis() + UUID.randomUUID().toString().replaceAll("-", "");
            String newFilePath = logMoveFolder + File.separator + newFileName + ".gz";
            MyFileUtils.copyFile(file, new File(newFilePath));
            oldLogFileDescribe.setLogFileFullPath(newFilePath);
        } else {
            oldLogFileDescribe.setLogFileFullPath(file.getAbsolutePath());
        }
        oldLogFileDescribe.setChangeFlag(2);
        oldLogFileDescribe.setCreationTime(creationTime);
        oldLogFileDescribe.setLastModifiedTime(lastModifiedTime);
        //使用日志文件的创建时间来确定打印日志时间
        if (httpConfig.isDecisionContentTimeUseFileTime()) {
            decisionContentTimeUseFileTime(creationTime, lastModifiedTime, fileTimes, oldLogFileDescribe);
        } else {
            detectionContentTimeInLinux(oldLogFileDescribe);
        }
        return oldLogFileDescribe;
    }

    private void decisionContentTimeUseFileTime(long creationTime, long lastModifiedTime, List<String> fileTimes, LogFileDescribeDto oldLogFileDescribe) {
        long now = System.currentTimeMillis();
        //如果文件的创建时间==文件的修改时间，认为是日志分片文件，以后都不会变化
        if (lastModifiedTime - creationTime <= 2000 && now - lastModifiedTime >= 2000) {
            String format = "yyyy-MM-dd HH:mm:ss";
            SimpleDateFormat sf = new SimpleDateFormat(format);
            String createTime = sf.format(new Date(creationTime));
            String createDay = createTime.substring(0, 10);
            //找到 在当前日志文件之前并且不是同一天 最近的日志文件,计算两个日志文件之间的日期范围
            boolean hasEarlierFile = false;
            for (String fileTime : fileTimes) {
                String fileDay = fileTime.substring(0, 10);
                if (fileTime.compareTo(createTime) < 0 && fileDay.compareTo(createDay) < 0) {
                    try {
                        List<String> dates = DateUtils.getRangeDate(sf.parse(fileTime), new Date(creationTime), Calendar.DAY_OF_MONTH, DateUtils.DATE_PATTERN);
                        oldLogFileDescribe.setDateTimes(dates);
                    } catch (Exception e) {
                        log.error(ErrorUtil.getStackMsg(e));
                    }
                    hasEarlierFile = true;
                    break;
                }
            }
            // 没有更早的文件
            if (!hasEarlierFile) {
                String os_name = loadOsName();
                if (os_name.equals(Windows)) {
                    detectionContentTimeInWindows(oldLogFileDescribe);
                } else {
                    detectionContentTimeInLinux(oldLogFileDescribe);
                }
            }
        } else {
            List<String> dates = DateUtils.getRangeDate(new Date(creationTime), new Date(lastModifiedTime), Calendar.DAY_OF_MONTH, DateUtils.DATE_PATTERN);
            if (CollectionUtils.isNotEmpty(dates)) {
                oldLogFileDescribe.setDateTimes(dates);
            }
        }
        if (CollectionUtils.isEmpty(oldLogFileDescribe.getDateTimes())) {
            String businessDate = new SimpleDateFormat("yyyy-MM-dd").format(new Date(lastModifiedTime - 2000));
            List<String> dateTimes = new ArrayList<>();
            dateTimes.add(businessDate);
            oldLogFileDescribe.setDateTimes(dateTimes);
        }
    }

    private void detectionContentTimeInLinux(LogFileDescribeDto oldLogFileDescribe) {
        try {
            List<String> dateTimes = new ArrayList<>();
            oldLogFileDescribe.setDateTimes(dateTimes);
            String firstLine = "";
            String lastLine = "";
            StringBuilder sb = new StringBuilder();
            String newFilePath = oldLogFileDescribe.getLogFileFullPath();
            long lastModifiedTime = oldLogFileDescribe.getLastModifiedTime();
            String firstCmd = sb.append("zcat ").append(newFilePath).append(" | head -1 ").toString();
            if (newFilePath.indexOf(".gz") == -1) {
                firstCmd = sb.append("cat ").append(newFilePath).append(" | head -1 ").toString();
            }
            String lastCmd = sb.append("zcat ").append(newFilePath).append(" | tail -1 ").toString();
            if (newFilePath.indexOf(".gz") == -1) {
                lastCmd = sb.append("cat ").append(newFilePath).append(" | tail -1 ").toString();
            }
            firstLine = doCmd(firstCmd);
            lastLine = doCmd(lastCmd);
            String startDateTime = "";
            String endDateTime = "";
            if (StringUtils.isNotBlank(firstLine)) {
                Pattern p = Pattern.compile("(\\d{4})-(\\d{1,2})-(\\d{1,2})");
                Matcher m = p.matcher(firstLine);
                while (m.find()) {
                    startDateTime = m.group(0);
                    break;
                }
                if (StringUtils.isNotBlank(lastLine)) {
                    m = p.matcher(lastLine);
                    while (m.find()) {
                        endDateTime = m.group(0);
                        break;
                    }
                }
            }
            if (StringUtils.isNotBlank(startDateTime) && StringUtils.isNotBlank(endDateTime)) {
                if (startDateTime.equals(endDateTime)) {
                    dateTimes.add(startDateTime);
                } else {
                    SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
                    List<String> dates = DateUtils.getRangeDate(format.parse(startDateTime), format.parse(endDateTime), Calendar.DAY_OF_MONTH, DateUtils.DATE_PATTERN);
                    if (CollectionUtils.isNotEmpty(dates)) {
                        oldLogFileDescribe.setDateTimes(dates);
                    } else {
                        String businessDate = new SimpleDateFormat("yyyy-MM-dd").format(new Date(lastModifiedTime - 2000));
                        dateTimes.add(businessDate);
                    }
                }
            } else if (StringUtils.isNotBlank(startDateTime)) {
                dateTimes.add(startDateTime);
            } else if (StringUtils.isNotBlank(endDateTime)) {
                dateTimes.add(endDateTime);
            } else {
                String businessDate = new SimpleDateFormat("yyyy-MM-dd").format(new Date(lastModifiedTime - 2000));
                dateTimes.add(businessDate);
            }
        } catch (Exception e) {
            log.error("{}", ErrorUtil.getStackMsg(e));
        }
    }

    public static void main(String[] args) {
        String logFolder = "D:\\pyt\\gateway\\aided-api\\src\\main\\resources\\dev\\logs";
        File folder = new File(logFolder);
        if (!folder.exists()) {
            folder.mkdirs();
        }
        File[] files = folder.listFiles();
        List<File> logFiles = new ArrayList<>();
        if (files != null && files.length > 0) {
            for (File file : files) {
                logFiles.add(file);
            }
            Collections.sort(logFiles, (File file1, File file2) -> {
                try {
                    BasicFileAttributes attr1 = Files.readAttributes(file1.toPath(), BasicFileAttributes.class);
                    BasicFileAttributes attr2 = Files.readAttributes(file2.toPath(), BasicFileAttributes.class);
                    long creationTime1 = attr1.creationTime().toMillis();
                    long lastModifiedTime1 = attr1.lastModifiedTime().toMillis();
                    long creationTime2 = attr2.creationTime().toMillis();
                    long lastModifiedTime2 = attr2.lastModifiedTime().toMillis();
                    if (lastModifiedTime2 != lastModifiedTime1) {
                        return lastModifiedTime2 > lastModifiedTime1 ? 1 : -1;
                    } else if (creationTime2 != creationTime1) {
                        return creationTime2 > creationTime1 ? 1 : -1;
                    } else {
                        return file2.getAbsoluteFile().compareTo(file1.getAbsoluteFile());
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
                return file2.getAbsoluteFile().compareTo(file1.getAbsoluteFile());
            });
            for (int i = 0; i < logFiles.size(); i++) {
                try {
                    File file = logFiles.get(i);
                    BasicFileAttributes attr1 = Files.readAttributes(file.toPath(), BasicFileAttributes.class);
                    long creationTime = attr1.creationTime().toMillis();
                    long lastModifiedTime = attr1.lastModifiedTime().toMillis();
                    String format = "yyyy-MM-dd HH:mm:ss";
                    SimpleDateFormat sf = new SimpleDateFormat(format);
                    String createTime = sf.format(new Date(creationTime));
                    String modifiedTime = sf.format(new Date(lastModifiedTime));
                    log.info("filename:{},createTime:{} ,modifiedTime{} ", file.getName(), createTime, modifiedTime);
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

    }

}
