package com.xforceplus.janus.framework.event;

import com.xforceplus.apollo.msg.SealedMessage;
import com.xforceplus.apollo.utils.ErrorUtil;
import com.xforceplus.apollo.utils.FileBytesUtil;
import com.xforceplus.apollo.utils.JacksonUtil;
import com.xforceplus.janus.config.core.config.ActionConfig;
import com.xforceplus.janus.config.core.config.HttpConfig;
import com.xforceplus.janus.config.core.util.Constant;
import com.xforceplus.janus.config.core.util.JanusHttpUtil;
import com.xforceplus.janus.framework.event.dto.BigDataTmpDto;
import com.xforceplus.janus.framework.record.domain.BigRequestTmpDto;
import com.xforceplus.janus.framework.record.portal.BigDataTmpRepository;
import com.xforceplus.janus.framework.util.MyFileUtils;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpStatus;
import org.springframework.context.ApplicationEventPublisher;

import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.util.Base64;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @Author: xuchuanhou
 * @Date:2023/6/30下午4:40
 */
@Slf4j
public class BigDataDispatcher {
    public static final BlockingQueue<BigDataTmpDto> BIG_DATA_TMP_QUEUE = new ArrayBlockingQueue<BigDataTmpDto>(1000);

    @Setter
    private ApplicationEventPublisher eventPublisher;

    @Setter
    private BigDataTmpRepository bigDataTmpRepository;

    @Setter
    private HttpConfig httpConfig;

    private final static int MAX_RETRY_TIMES = 100;

    public void run() {

        while (true) {
            try {
                BigDataTmpDto bigDataTmpDto = BIG_DATA_TMP_QUEUE.take();
                if (bigDataTmpDto.getRetryTimes() > MAX_RETRY_TIMES) {
                    continue;
                }
                int existCount = bigDataTmpRepository.countByTaskId(bigDataTmpDto.getTaskId());
                if (bigDataTmpDto.getSplitCount() == existCount) {
                    buildAndSendMsg(bigDataTmpDto);
                } else {
                    TimeUnit.SECONDS.sleep(1);
                    //数据缺失,主动查询缺失的数据
                    List<BigRequestTmpDto> dataDetails = bigDataTmpRepository.getByTaskId(bigDataTmpDto.getTaskId(), false);
                    if (CollectionUtils.isEmpty(dataDetails)) {
                        bigDataTmpDto.setRetryTimes(bigDataTmpDto.getRetryTimes() + 1);
                        BIG_DATA_TMP_QUEUE.offer(bigDataTmpDto);
                        continue;
                    }
                    List<Integer> existIndexs = dataDetails.stream().map(d -> d.getTaskIndex()).collect(Collectors.toList());
                    for (int i = 0; i < bigDataTmpDto.getSplitCount(); i++) {
                        if (existIndexs.contains(Integer.valueOf(i))) {
                            continue;
                        }
                        //
                        BigRequestTmpDto bigRequestTmpDto = fetchBigData(bigDataTmpDto.getTaskId(), i);
                        if (bigRequestTmpDto != null) {
                            bigDataTmpRepository.saveBigDataTmp(bigRequestTmpDto);
                        }
                    }
                    existCount = bigDataTmpRepository.countByTaskId(bigDataTmpDto.getTaskId());
                    if (bigDataTmpDto.getSplitCount() == existCount) {
                        buildAndSendMsg(bigDataTmpDto);
                    } else {
                        bigDataTmpDto.setRetryTimes(bigDataTmpDto.getRetryTimes() + 1);
                        BIG_DATA_TMP_QUEUE.offer(bigDataTmpDto);
                        continue;
                    }
                }
            } catch (Exception ex) {

            }
        }
    }

    private void buildAndSendMsg(BigDataTmpDto bigDataTmpDto) {
        SealedMessage.Header header = JacksonUtil.getInstance().fromJson(bigDataTmpDto.getHeader(), SealedMessage.Header.class);
        //文件与报文分开处理
        String fileName = header.getOthers().get(MyFileUtils.TRANSFER_FILE_NAME);
        if (StringUtils.isNotBlank(fileName)) {
            List<BigRequestTmpDto> dataDetails = bigDataTmpRepository.getByTaskId(bigDataTmpDto.getTaskId(), false);
            BigRequestTmpDto tmpDto;
            String bPath = "tempfile";
            File fb = new File(bPath);
            if (!fb.isDirectory() || !fb.exists()) {
                fb.mkdir();
            }
            String outputPath =  bPath + File.separator + fileName;
            for (BigRequestTmpDto bigData:
                    dataDetails) {
                tmpDto = bigDataTmpRepository.queryOne(bigData.getTaskId(), bigData.getTaskIndex());
                byte[] fileBytes = Base64.getDecoder().decode(tmpDto.getContent());
                try (OutputStream outputStream = new FileOutputStream(outputPath, true)) {
                    outputStream.write(fileBytes);
                } catch (Exception e) {
                    log.error(ErrorUtil.getStackMsg(e));
                    return;
                }
            }
            SealedMessage realSealedMessage = new SealedMessage(header, new SealedMessage.Payload(outputPath));
            SealedMessageEvent sealedMessageEvent = new SealedMessageEvent(this, realSealedMessage);
            eventPublisher.publishEvent(sealedMessageEvent);
        } else {
            //全部数据已经接收，读取组装数据
            List<BigRequestTmpDto> dataDetails = bigDataTmpRepository.getByTaskId(bigDataTmpDto.getTaskId());
            StringBuilder stringBuilder = new StringBuilder();
            dataDetails.forEach(bq -> stringBuilder.append(bq.getContent()));

            byte[] decodeChar = FileBytesUtil.hexStringToByte(stringBuilder.toString());
            byte[] ungzipBody = FileBytesUtil.unjzlib(decodeChar);
            String realBody = new String(ungzipBody);
            SealedMessage realSealedMessage = new SealedMessage(header, new SealedMessage.Payload(realBody));
            SealedMessageEvent sealedMessageEvent = new SealedMessageEvent(this, realSealedMessage);
            eventPublisher.publishEvent(sealedMessageEvent);
            log.info("业务代码:{} ID:{}:事件发送完成", realSealedMessage.getHeader().getPayLoadId(), realSealedMessage.getHeader().getMsgId());
        }
    }


    //从远程拉取缺失的数据
    private BigRequestTmpDto fetchBigData(String taskId, int index) {
        Map<String, String> headers = new HashMap<>();
        headers.put("action", httpConfig.getAction().get(ActionConfig.ACTION_FETCH_BIGDATA_BLOCK));
        headers.put("Authentication", httpConfig.getAuthentication());
        headers.put(Constant.KEY_SERIALNO, taskId);
        headers.put("taskid", taskId);
        headers.put("index", ""+index);

        try {
            JanusHttpUtil.ResponseCus responseCus = JanusHttpUtil.doGetEntire(httpConfig.getUrl(), null, true, headers);
            if (responseCus != null && responseCus.getStatus() == HttpStatus.SC_OK && StringUtils.isNotBlank(responseCus.getBody())) {
                Map<String, Object> respMap = JacksonUtil.getInstance().fromJson(responseCus.getBody(), Map.class);
                if (respMap.get("result") != null) {
                    return JacksonUtil.getInstance().fromJson(respMap.get("result").toString(), BigRequestTmpDto.class);
                }
            }
        } catch (Exception ex) {
            log.error("获取大报文分片异常taskId:{},index:{},msg:{}", taskId, index, ex.getMessage());
            return null;
        }
        return null;
    }


}
