package com.xforceplus.janus.framework.record.interceptor;

import com.xforceplus.apollo.utils.Base64Util;
import com.xforceplus.apollo.utils.code.GZipUtils;
import com.xforceplus.janus.config.core.util.Constant;
import com.xforceplus.janus.framework.record.domain.BigRequestTmpDto;
import com.xforceplus.janus.framework.record.portal.BigDataTmpRepository;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateFormatUtils;
import org.apache.http.HttpStatus;
import org.springframework.web.servlet.HandlerInterceptor;

import java.io.BufferedReader;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Date;
import java.util.List;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import lombok.Setter;
import lombok.extern.slf4j.Slf4j;

/**
 * 大报文拆分拦截器，大报文拆分： 将不同任务号的数据存放到数据库临时表，请求结尾标记进行数据查询组装，将组装后的完整数据响应返回
 */
@Slf4j
public class BigRequestInterceptor implements HandlerInterceptor {

    @Setter
    private BigDataTmpRepository bigDataTmpRepository;

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {

        String taskId = request.getHeader(Constant.FIELD_BIG_DATA_TASK_ID);
        // 大报文拆分请求
        if (StringUtils.isNotBlank(taskId)) {
            int splitCount = Integer.valueOf(request.getHeader(Constant.FIELD_BIG_DATA_COUNT));

            //**所有所有分片上传完成，发起完整调用
            String onlyInvoke = request.getHeader("big-data-split-only-invoke");

            String body = null;
            if (request instanceof CustomHttpServletRequestWrapper) {
                body = getRequestBody((CustomHttpServletRequestWrapper) request).toString();
            }
            int existCount = bigDataTmpRepository.countByTaskId(taskId);
            if (Boolean.TRUE.toString().equals(onlyInvoke)) {
                //组装完整报文并调用下游业务接口
                if (existCount == splitCount) {
                    List<BigRequestTmpDto> dataDetails = bigDataTmpRepository.getByTaskId(taskId);
                    StringBuilder stringBuilder = new StringBuilder();
                    dataDetails.forEach(bq -> stringBuilder.append(bq.getContent()));

                    byte[] decodeChar = Base64Util.decode(stringBuilder.toString().getBytes(StandardCharsets.UTF_8));
                    if (request instanceof CustomHttpServletRequestWrapper) {
                        byte[] ungzipBody = GZipUtils.ungzip(decodeChar);
                        ((CustomHttpServletRequestWrapper) request).setBody(new String(ungzipBody).getBytes());
                    }
                    response.addHeader("big-request-finish", "true");
                    return true;
                } else {
                    //数据不完整需要发送端重新发起请求或者补全数据
                    String sjycBody = "{\"code\":\"0\",\"message\":\"大报文拆分异常数据缺失\",\"big-request-data-miss\":true}";
                    response.getWriter().write(sjycBody);
                    response.setStatus(HttpStatus.SC_INTERNAL_SERVER_ERROR);
                    response.addHeader("big-request-data-miss", "true");
                    return false;
                }
            } else {
                int splitIndex = Integer.valueOf(request.getHeader("big-data-split-index"));
                //保存请求的拆分数据
                BigRequestTmpDto requestBody = new BigRequestTmpDto(taskId, splitIndex, body, DateFormatUtils.format(new Date(), "yyyyMMddHHmmss"));
                String sjycBody = "{\"code\":\"1\",\"message\":\"已经上传\"}";
                try {
                    boolean result = bigDataTmpRepository.saveBigDataTmp(requestBody);
                    if (result) {
                        response.setStatus(HttpStatus.SC_OK);
                        response.getWriter().write(sjycBody);
                        return false;
                    }
                } catch (Exception ex) {
                    log.error("大报文片段保存失败:{} ,index:{}", taskId,splitIndex);
                }
                sjycBody = "{\"code\":\"0\",\"message\":\"上传失败\"}";
                response.getWriter().write(sjycBody);
                response.setStatus(HttpStatus.SC_INTERNAL_SERVER_ERROR);
                return false;
            }

        } else {
            return true;
        }
    }

    private StringBuilder getRequestBody(CustomHttpServletRequestWrapper request) {
        StringBuilder requestBodySB = new StringBuilder();
        try (BufferedReader br = request.getReader()) {
            String line = null;
            while ((line = br.readLine()) != null) {
                requestBodySB.append(line);
            }
            br.close();
            return requestBodySB;
        } catch (IOException ex) {
        }

        return requestBodySB;
    }

}
