
package com.xforceplus.janus.framework.util;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.FastDateFormat;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

public class DateUtils {

    /**
     * 时间格式(yyyy-MM-dd)
     */
    public final static String DATE_PATTERN      = "yyyy-MM-dd";
    public final static String DATE_PATTERN_NO   = "yyyyMMdd";
    /**
     * 时间格式(yyyy-MM-dd HH:mm:ss)
     */
    public final static String DATE_TIME_PATTERN = "yyyy-MM-dd HH:mm:ss";
    public final static String DATE_TIME_PATTERN1 = "yyyyMMddHHmmssSSS";
    public static final String DATE_PATTERN_H       = "yyyyMMddHH";
    public final static String DATE_PATTERN_H2 = "yyyy-MM-dd HH";

    /**
     * 日期格式化 日期格式为：yyyy-MM-dd
     *
     * @param date 日期
     * @return 返回yyyy-MM-dd格式日期
     */
    public static String format(Date date) {
        return format(date, DATE_PATTERN);
    }

    /**
     * 日期格式化 日期格式为：yyyy-MM-dd
     *
     * @param date    日期
     * @param pattern 格式，如：DateUtils.DATE_TIME_PATTERN
     * @return 返回yyyy-MM-dd格式日期
     */
    public static String format(Date date, String pattern) {
        if (date == null) {
            return null;
        }

        String datePattern = pattern;
        if (StringUtils.isEmpty(pattern)) {
            datePattern = DATE_TIME_PATTERN;
        }

        FastDateFormat dateFormat = FastDateFormat.getInstance(datePattern);
        return dateFormat.format(date);
    }

    /**
     * 字符串转换成日期
     *
     * @param strDate 日期字符串
     * @param pattern 日期的格式，如：DateUtils.DATE_TIME_PATTERN
     */
    public static Date stringToDate(String strDate, String pattern) {
        if (StringUtils.isBlank(strDate)) {
            return null;
        }

        DateTimeFormatter fmt = DateTimeFormat.forPattern(pattern);
        return fmt.parseLocalDateTime(strDate).toDate();
    }


    /**
     * 对日期的【秒】进行加/减
     *
     * @param date    日期
     * @param seconds 秒数，负数为减
     * @return 加/减几秒后的日期
     */
    public static Date addDateSeconds(Date date, int seconds) {
        DateTime dateTime = new DateTime(date);
        return dateTime.plusSeconds(seconds).toDate();
    }

    /**
     * 对日期的【分钟】进行加/减
     *
     * @param date    日期
     * @param minutes 分钟数，负数为减
     * @return 加/减几分钟后的日期
     */
    public static Date addDateMinutes(Date date, int minutes) {
        DateTime dateTime = new DateTime(date);
        return dateTime.plusMinutes(minutes).toDate();
    }

    /**
     * 对日期的【小时】进行加/减
     *
     * @param date  日期
     * @param hours 小时数，负数为减
     * @return 加/减几小时后的日期
     */
    public static Date addDateHours(Date date, int hours) {
        DateTime dateTime = new DateTime(date);
        return dateTime.plusHours(hours).toDate();
    }

    /**
     * 对日期的【天】进行加/减
     *
     * @param date 日期
     * @param days 天数，负数为减
     * @return 加/减几天后的日期
     */
    public static Date addDateDays(Date date, int days) {
        DateTime dateTime = new DateTime(date);
        return dateTime.plusDays(days).toDate();
    }

    /**
     * 对日期的【周】进行加/减
     *
     * @param date  日期
     * @param weeks 周数，负数为减
     * @return 加/减几周后的日期
     */
    public static Date addDateWeeks(Date date, int weeks) {
        DateTime dateTime = new DateTime(date);
        return dateTime.plusWeeks(weeks).toDate();
    }

    /**
     * 对日期的【月】进行加/减
     *
     * @param date   日期
     * @param months 月数，负数为减
     * @return 加/减几月后的日期
     */
    public static Date addDateMonths(Date date, int months) {
        DateTime dateTime = new DateTime(date);
        return dateTime.plusMonths(months).toDate();
    }

    /**
     * 对日期的【年】进行加/减
     *
     * @param date  日期
     * @param years 年数，负数为减
     * @return 加/减几年后的日期
     */
    public static Date addDateYears(Date date, int years) {
        DateTime dateTime = new DateTime(date);
        return dateTime.plusYears(years).toDate();
    }

    /**
     * 根据周数，获取开始日期、结束日期
     *
     * @param week 周期  0本周，-1上周，-2上上周，1下周，2下下周
     * @return 返回date[0]开始日期、date[1]结束日期
     */
    public static Date[] getWeekStartAndEnd(int week) {
        DateTime dateTime = new DateTime();
        LocalDate date = new LocalDate(dateTime.plusWeeks(week));

        date = date.dayOfWeek().withMinimumValue();
        Date beginDate = date.toDate();
        Date endDate = date.plusDays(6).toDate();
        return new Date[]{beginDate, endDate};
    }

    /**
     * 获取时间当天开始结束时间
     *
     * @param date
     * @return
     */
    public static String[] getDayStartAndEnd(Date date) {
        String endDate = DateUtils.format(endOfDay(date), DATE_TIME_PATTERN);
        String beginDate = DateUtils.format(beginOfDay(date), DATE_TIME_PATTERN);

        return new String[]{beginDate, endDate};
    }

    /**
     * 获取时间当天小时开始结束时间
     *
     * @param date
     * @return
     */
    public static String[] getHourStartAndEnd(Date date) {
        String beginDate = DateUtils.format(beginOfHour(date), DATE_TIME_PATTERN);
        String endDate = DateUtils.format(endOfHour(date), DATE_TIME_PATTERN);

        return new String[]{beginDate, endDate};
    }

    public static Date beginOfHour(Date date) {
        Calendar calendar = calendar(date);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return new Date(calendar.getTimeInMillis());
    }

    public static Date endOfHour(Date date) {
        Calendar calendar = calendar(date);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        calendar.set(Calendar.MILLISECOND, 999);
        return new Date(calendar.getTimeInMillis());
    }


    public static Date beginOfDay(Date date) {
        Calendar calendar = calendar(date);
        calendar.set(Calendar.HOUR, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
        return new Date(calendar.getTimeInMillis());
    }

    public static Date endOfDay(Date date) {

        Calendar calendar = calendar(date);
        calendar.set(Calendar.HOUR, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        calendar.set(Calendar.MILLISECOND, 999);
        return new Date(calendar.getTimeInMillis());
    }

    /**
     * 转换为Calendar对象
     *
     * @param date 日期对象
     * @return Calendar对象
     */
    public static Calendar calendar(Date date) {
        return calendar(date.getTime());
    }

    /**
     * 转换为Calendar对象
     *
     * @param millis 时间戳
     * @return Calendar对象
     */
    public static Calendar calendar(long millis) {
        final Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(millis);
        return cal;
    }


    /**
     * 获取开始日期与结束日期区间的所有时间列表 例如 2021-01,2021-04 返回2021-01，2021-02，2021-03，2021-04
     *
     * @param field  @see Calendar 日期字段 常见支持  HOUR_OF_DAY   ，分，时，日，月
     * @param format 返回日期格式
     */
    public static List<String> getRangeDate(Date beginDate, Date endDate, int field, String format) {

        List<String> result = new ArrayList<>();
        result.add(format(beginDate, format));
        while (beginDate.before(endDate)) {
            Calendar calendar = calendar(beginDate);
            calendar.add(field, 1);
            beginDate = new Date(calendar.getTimeInMillis());
            result.add(format(beginDate, format));
        }
        return result;
    }


    //获取前月的第一天
    public static String getfirstDayOfPrevMonth() {
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        Calendar cal_1 = Calendar.getInstance();//获取当前日期
        cal_1.add(Calendar.MONTH, -1);
        cal_1.set(Calendar.DAY_OF_MONTH, 1);//设置为1号,当前日期既为本月第一天
        String firstDay = format.format(cal_1.getTime());
//        System.out.println("------------firstDay:" + firstDay);
        return firstDay;

    }

    //获取当前月第一天
    public static String getfirstDayOfCurMonth() {
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        Calendar c = Calendar.getInstance();
        c.add(Calendar.MONTH, 0);
        c.set(Calendar.DAY_OF_MONTH, 1);//设置为1号,当前日期既为本月第一天
        String first = format.format(c.getTime());
//        System.out.println("===============first:" + first);
        return first;

    }

    //获取指定月的最后一天
    public static String getLastDay(String ym) {
        String last = "";
        try {
            SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
            String ymd = ym + "01";
            Calendar ca = calendar(format.parse(ymd));
            ca.set(Calendar.DAY_OF_MONTH, ca.getActualMaximum(Calendar.DAY_OF_MONTH));
            last = format.format(ca.getTime());
//            System.out.println("===============last:" + last);
        } catch (Exception e) {

        }

        return last;
    }
    //获取指定日所在月的最后一天
    public static String getLastDay(String ymd ,String pattern) {
        String last = "";
        try {
            SimpleDateFormat format = new SimpleDateFormat(pattern);
            Calendar ca = calendar(format.parse(ymd));
            ca.set(Calendar.DAY_OF_MONTH, ca.getActualMaximum(Calendar.DAY_OF_MONTH));
            last = format.format(ca.getTime());
//            System.out.println("===============last:" + last);
        } catch (Exception e) {

        }

        return last;
    }


    public static String getlastDayOfPrevMonth() {
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        //获取前月的最后一天
        Calendar cale = Calendar.getInstance();
        //设置为0号,当前日期既为本月第0天
        cale.set(Calendar.DAY_OF_MONTH, 0);
        String lastDay = format.format(cale.getTime());
//        System.out.println("------------lastDay:" + lastDay);
        return lastDay;
    }

    //获取当前月最后一天
    public static String getlastDayOfCurMonth() {
        SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");
        Calendar ca = Calendar.getInstance();
        ca.set(Calendar.DAY_OF_MONTH, ca.getActualMaximum(Calendar.DAY_OF_MONTH));
        String last = format.format(ca.getTime());
//        System.out.println("===============last:" + last);
        return last;
    }
}
