package com.xforceplus.janus.generator.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.xforceplus.apollo.utils.JacksonUtil;
import com.xforceplus.janus.common.constants.Constant;
import com.xforceplus.janus.commons.util.PageUtils;
import com.xforceplus.janus.generator.domain.GenModel;
import com.xforceplus.janus.generator.dto.ModelPropertyDto;
import com.xforceplus.janus.generator.mapper.GenModelMapper;
import com.xforceplus.janus.generator.param.GenModelParam;
import com.xforceplus.janus.generator.service.IGenModelService;
import com.xforceplus.janus.generator.util.GenConstants;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import lombok.extern.slf4j.Slf4j;

/**
 * 业务 服务层实现
 *
 * @author just
 */
@Slf4j
@Component
public class GenModelServiceImpl extends ServiceImpl<GenModelMapper, GenModel> implements IGenModelService {

    /**
     * @see com.xforceplus.janus.generator.service.IGenModelService#getRichModel(String, String,
     * boolean)
     */
    @Cacheable(cacheNames = "modelByPrjId", key = "#projectId+#className")
    @Override
    public GenModel getRichModel(String projectId, String className, boolean cascade) {
        GenModel model = this.getOne(new QueryWrapper<GenModel>()
                .eq("class_name", className)
                .eq(StringUtils.isNotBlank(projectId), "project_id", projectId)
                .eq(Constant.IS_VALID, Constant.IS_VALIDED));

        if (model == null || StringUtils.isBlank(model.getProperty())) {
            return model;
        }
        List<ModelPropertyDto> properties = null;
        if (cascade) {
            properties = genPropertyFromProperty(projectId, model.getProperty());
        } else {
            properties = JacksonUtil.getInstance().fromJsonToList(model.getProperty(), ModelPropertyDto.class);
        }

        model.setProperties(properties);
        return model;
    }


    /**
     * @see com.xforceplus.janus.generator.service.IGenModelService#findSimpleModel(String, String)
     */
    @Override
    public List<GenModel> findSimpleModel(String projectId, String excludClassName) {
        List<GenModel> models = this.list(new QueryWrapper<GenModel>()
                .notIn(StringUtils.isNotBlank(excludClassName), "class_name", excludClassName)
                .eq(StringUtils.isNotBlank(projectId), "project_id", projectId)
                .eq(Constant.IS_VALID, Constant.IS_VALIDED));
        return models;
    }

    @Override
    public Map<String, Object> getMockFromModeDefine(String projectId, String className) {
        GenModel model = this.getRichModel(projectId, className, true);
        if (model == null) {
            return null;
        }
        return mockData(model);
    }

    private Map<String, Object> mockData(GenModel model) {
        Map<String, Object> mockData = new HashMap<>();
        for (ModelPropertyDto col : model.getProperties()) {
            switch (col.getType()) {
                case GenConstants.TYPE_INTEGER:
                case GenConstants.TYPE_LONG:
                case GenConstants.TYPE_DOUBLE:
                case GenConstants.TYPE_BIGDECIMAL:
                    mockData.put(col.getName(), StringUtils.isBlank(col.getDeValue()) ? 1 : col.getDeValue());
                    continue;
                case GenConstants.TYPE_BOOLEAN:
                    mockData.put(col.getName(), StringUtils.isBlank(col.getDeValue()) ? false : Boolean.valueOf(col.getDeValue()));
                    continue;
                case GenConstants.TYPE_ARRAY:
                    List<Map<String, Object>> subs = new ArrayList<>();
                    GenModel modelsub = new GenModel(col.getRefClassName(), col.getChildren());

                    Map<String, Object> mockMap = this.mockData(modelsub);
                    subs.add(mockMap);
                    mockData.put(col.getName(), subs);
                    continue;
                case GenConstants.TYPE_OBJ:
                    GenModel modelsubObj = new GenModel(col.getRefClassName(), col.getChildren());
                    Map<String, Object> mocksubMap = this.mockData(modelsubObj);
                    mockData.put(col.getName(), mocksubMap);
                    continue;
                default:
                    mockData.put(col.getName(), StringUtils.isBlank(col.getDeValue()) ? "aa" : col.getDeValue());
                    continue;
            }
        }
        return mockData;
    }

    @Override
    public List<ModelPropertyDto> genPropertyFromProperty(String projectId, String property) {
        List<ModelPropertyDto> properties = JacksonUtil.getInstance().fromJsonToList(property, ModelPropertyDto.class);
        paddingProperty(projectId,properties);
        return properties;
    }

    public void paddingProperty(String projectId, List<ModelPropertyDto> properties) {
        if (CollectionUtils.isEmpty(properties)) {
            return;
        }

        properties.forEach(proper -> {
            if (StringUtils.isNotBlank(proper.getRefClassName())) {
                GenModel subModel = getRichModel(projectId, proper.getRefClassName(), true);
                if (subModel != null) {
                    proper.setChildren(subModel.getProperties());
                }
            } else if (CollectionUtils.isNotEmpty(proper.getChildren())) {
                paddingProperty(projectId, proper.getChildren());
            }
        });

    }


    @Override
    public PageUtils queryPager(GenModelParam param) {
        Page<GenModel> pageParam =new Page<GenModel>(param.getPage(),param.getSize());

        IPage<GenModel> page = this.page(pageParam,
                new QueryWrapper<GenModel>()
                        .eq(StringUtils.isNotBlank(param.getClassName()), "class_name", param.getClassName())
                        .like(StringUtils.isNotBlank(param.getClassNameLike()), "class_name", param.getClassName())
                        .eq(StringUtils.isNotBlank(param.getProjectId()), "project_id", param.getProjectId())
                        .eq(Constant.IS_VALID, Constant.IS_VALIDED)
        );
        return new PageUtils(page);
    }

    @Override
    public boolean deleteById(String id) {
        GenModel model = new GenModel();
        model.setId(id);
        model.setIsValid(Constant.IS_NOT_VALIDED);
        return this.updateById(model);
    }

    @Override
    public GenModel info(String id) {
        GenModel model = this.getById(id);
        if (model != null && StringUtils.isNotBlank(model.getProperty())) {
            model.setProperties(JacksonUtil.getInstance().fromJsonToList(model.getProperty(), ModelPropertyDto.class));
            model.setProperty(null);
        }
        return model;
    }

    @Override
    public boolean saveOrUpdateEntity(GenModel modelDefine) {
        if (CollectionUtils.isNotEmpty(modelDefine.getProperties())) {
            modelDefine.getProperties().forEach(field -> {
                //清空children 避免级联
                field.setChildren(null);
            });
            modelDefine.setProperty(JacksonUtil.getInstance().toJson(modelDefine.getProperties()));
        }
        return this.saveOrUpdate(modelDefine);
    }
}