package com.xforceplus.janus.message.common.utils;

import com.google.common.collect.Sets;
import com.xforceplus.janus.message.common.dto.SysUserToken;

import io.jsonwebtoken.Claims;
import io.jsonwebtoken.Jws;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;
import lombok.extern.slf4j.Slf4j;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DateUtils;

import java.util.Date;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.TimeUnit;

/**
 * JWT utility
 *
 * @author liuyao@xforceplus.com
 * @version 1.0
 * @date 2021/3/8 14:13
 */
@Slf4j
public abstract class JwtUtils {
    public static final String HEADER_MB_TOKEN_NAME            = "mbToken";
    public static final String HEADER_COMPATIBLE_MB_TOKEN_NAME = "MB_TOKEN";
    /**
     * Expired time(After 50 years)
     */
    public static final long   EXPIRED_TIME                    = TimeUnit.DAYS.toMillis(365L * 50);
    public static final String SIGN_KEY                        = "ukc8BDbRigUDaY6pZFfWus2jZWLPHO17L8s";
    public static final long   SYS_USER_EXPIRED_TIME           = TimeUnit.DAYS.toMillis(1L);

    public static final Set<String> ERROR_TOKEN = Sets.newConcurrentHashSet();
    /**
     * Generate JWT token
     *
     * @param appKey appKey
     * @return JWT token
     */
    public static String generateJwtToken(String env, String appKey, String appSecret) {
        return Jwts.builder()
                .setHeaderParam("typ", "JWT")
                .setHeaderParam("alg", "HS256")
                .setSubject("message-bus")
                .setIssuedAt(new Date())
                .setExpiration(new Date(System.currentTimeMillis() + EXPIRED_TIME))
                .claim("env", env)
                .claim("appKey", appKey)
                .claim("appSecret", appSecret)
                .signWith(SignatureAlgorithm.HS256, SIGN_KEY)
                .compact();
    }

    /**
     * Generate sys user JWT token
     *
     * @return JWT token
     */
    public static String generateSysUserJwtToken(String userId, String userName) {
        return Jwts.builder()
                .setHeaderParam("typ", "JWT")
                .setHeaderParam("alg", "HS256")
                .setSubject("message-bus-sys-user")
                .setIssuedAt(new Date())
                .setExpiration(new Date(System.currentTimeMillis() + SYS_USER_EXPIRED_TIME))
                .claim("userId", userId)
                .claim("userName", userName)
                .signWith(SignatureAlgorithm.HS256, SIGN_KEY)
                .compact();
    }

    /**
     * Get app key from JWT token
     *
     * @param jwtToken JWT token
     * @return app key value
     */
    public static String getAppKeyFromJwtToken(String jwtToken) {
        if (StringUtils.isEmpty(jwtToken)) {
            return StringUtils.EMPTY;
        }
        try {
            Jws<Claims> claimsJws = Jwts.parser().setSigningKey(SIGN_KEY).parseClaimsJws(jwtToken);
            Claims claims = claimsJws.getBody();
            return (String) claims.get("appKey");
        } catch (Exception e) {
            if (!ERROR_TOKEN.contains(jwtToken)) {
                log.error("Failed to get app key from JWT token! {} --- exception {}", jwtToken, e);
                ERROR_TOKEN.add(jwtToken);
            }
            return StringUtils.EMPTY;
        }
    }

    /**
     * Get app key from JWT token
     *
     * @param jwtToken JWT token
     * @return app key value
     */
    public static String getEnvFromJwtToken(String jwtToken) {
        if (StringUtils.isEmpty(jwtToken)) {
            return StringUtils.EMPTY;
        }
        try {
            Jws<Claims> claimsJws = Jwts.parser().setSigningKey(SIGN_KEY).parseClaimsJws(jwtToken);
            Claims claims = claimsJws.getBody();
            return (String) claims.get("env");
        } catch (Exception e) {
            log.error("Failed to get env from JWT token!", e);
            return StringUtils.EMPTY;
        }
    }

    /**
     * Get app key from JWT token
     *
     * @param jwtToken JWT token
     * @return app key value
     */
    public static SysUserToken getSysUserByJwtToken(String jwtToken) {
        if (StringUtils.isEmpty(jwtToken)) {
            return null;
        }
        try {
            Jws<Claims> claimsJws = Jwts.parser().setSigningKey(SIGN_KEY).parseClaimsJws(jwtToken);
            Claims claims = claimsJws.getBody();

            SysUserToken userToken = new SysUserToken();
            userToken.setUserId((String) claims.get("userId"));
            userToken.setUserName((String) claims.get("userName"));
            Integer exp = (Integer) claims.get("exp");


            final Date expireTime = new Date(exp * 1000L);

            userToken.setExpireTime(expireTime);

            return userToken;
        } catch (Exception e) {
            log.error("Failed to get env from JWT token!", e);
            return null;
        }
    }

}
