package com.xforceplus.tech.logger.helper;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.xforceplus.tech.logger.domain.LogRecord;
import net.logstash.logback.argument.DeferredStructuredArgument;
import net.logstash.logback.argument.StructuredArgument;
import net.logstash.logback.argument.StructuredArguments;
import org.apache.commons.lang3.time.StopWatch;

import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.TimeUnit;

/**
 * private String type;
 * <p>
 * private String subType;
 * <p>
 * private String key;
 * <p>
 * private String msg;
 * <p>
 * private Map<String, Object> params;
 * <p>
 * private Map<String, Object> ext;
 * logger
 */
public class LoggerBuilder {

    private String type;

    private String subType;

    private String key;

    private String name;

    private String status;

    private long latency;

    private StopWatch stopWatch;

    private TimeUnit timeUnit = TimeUnit.NANOSECONDS;

    private Map<String, Object> params = new HashMap<>();

    private Map<String, Object> ext = new HashMap<>();

    public static LoggerBuilder inBound() {
        return new LoggerBuilder().type("in");
    }

    public static LoggerBuilder outBound() {
        return new LoggerBuilder().type("out");
    }

    public static LoggerBuilder span() {
        return new LoggerBuilder().type("span");
    }

    public LoggerBuilder timed(TimeUnit timeUnit) {
        this.timeUnit = timeUnit;
        return timed();
    }

    //Stateful
    public LoggerBuilder timed() {
        if(stopWatch == null){
            stopWatch = new StopWatch(UUID.randomUUID().toString());
        }
        if(!stopWatch.isStarted()) {
            stopWatch.start();
        }

        return this;
    }

    public static LoggerBuilder event() {
        return new LoggerBuilder().type("evt");
    }

    public LoggerBuilder type(String type){
        this.type = type;
        return this;
    }

    public LoggerBuilder http() {
        this.subType = "http";
        return this;
    }

    public LoggerBuilder method() {
        this.subType = "method";
        Map<Thread, StackTraceElement[]> allStackTraces = Thread.getAllStackTraces();
        StackTraceElement[] stackTraceElements = allStackTraces.get(Thread.currentThread());
        if(stackTraceElements.length > 3) {
            this.name =  stackTraceElements[3].getMethodName();
        }
        return this;
    }

    public LoggerBuilder name(String name) {
        this.name = name;
        return this;
    }

    public LoggerBuilder subType(String subType) {
        this.subType = subType;
        return this;
    }

    public LoggerBuilder key(String key) {
        this.key = key;
        return this;
    }

    public LoggerBuilder params(String key, Object obj) {
        this.params.put(key, obj);
        return this;
    }

    public LoggerBuilder status(String status) {
        this.status = status;
        return this;
    }

    public LoggerBuilder ext(String key, Object obj) {
        this.ext.put(key, obj);
        return this;
    }

    public LogRecord build() {

        if(stopWatch != null) {
            if(stopWatch.isStarted()) {
                stopWatch.stop();
                this.latency = stopWatch.getNanoTime();
            }
        }

        LogRecord record = new LogRecord();
        record.setType(type);
        record.setSubType(subType);
        record.setKey(key);
        record.setName(name);
        record.setStatus(status);
        record.setElapsedTime(latency);
        record.setParams(new HashMap<>(params));
        record.setExt(new HashMap<>(ext));
        record.setTimeUnit(timeUnit);
        return record;
    }

    public StructuredArgument kv() {
        return new DeferredStructuredArgument(() -> StructuredArguments.kv("x-log", this.build()));
    }

    public String str(ObjectMapper mapper){
        LogRecord build = this.build();
        try {
            return mapper.writeValueAsString(build);
        } catch (JsonProcessingException e) {
            //TODO error
        }
        //TODO
        return "{}";
    }
}