package com.xforceplus.tenantsecurity.interceptor;

import com.auth0.jwt.exceptions.TokenExpiredException;
import com.google.common.collect.Lists;
import com.xforceplus.tenantsecurity.annotation.Authorization;
import com.xforceplus.tenantsecurity.annotation.NeedExtraInfo;
import com.xforceplus.tenantsecurity.annotation.WithoutAuth;
import com.xforceplus.tenantsecurity.config.AuthorityProperties;
import com.xforceplus.tenantsecurity.domain.AuthorizedUser;
import com.xforceplus.tenantsecurity.domain.IAuthorizedUser;
import com.xforceplus.tenantsecurity.domain.UserInfoHolder;
import com.xforceplus.tenantsecurity.domain.UserType;
import com.xforceplus.tenantsecurity.feign.client.UserExtraInfoClientService;
import com.xforceplus.tenantsecurity.feign.client.UserResourceClientService;
import com.xforceplus.tenantsecurity.feign.model.*;
import com.xforceplus.tenantsecurity.jwt.JwtConstants;
import com.xforceplus.tenantsecurity.jwt.JwtUtils;
import com.xforceplus.tenantsecurity.utils.CompressionUtils;
import com.xforceplus.tenantsecurity.utils.JsonUtils;
import com.xforceplus.tenantsecurity.utils.RequestUrlUtils;
import com.xforceplus.tenantsecurity.utils.RequestUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.env.Environment;
import org.springframework.http.MediaType;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.HandlerMapping;
import org.springframework.web.servlet.resource.ResourceHttpRequestHandler;
import org.springframework.web.util.UriComponents;
import org.springframework.web.util.UriComponentsBuilder;
import org.springframework.web.util.WebUtils;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * 微服务中 拦截http请求，解析请求头中的用户信息
 *
 * @author geewit
 * @since 2019-05-13
 */
public class UserContextInterceptor implements HandlerInterceptor {

    private static final Logger logger = LoggerFactory.getLogger(UserContextInterceptor.class);

    @Autowired
    private Environment environment;

    @Autowired(required = false)
    private AuthorityProperties authorityProperties;

    @Value("${xforce.tenant_security.adapter.interceptors.default_user_info:}")
    private String defaultUserInfoJson;

    @Autowired
    private UserExtraInfoClientService userExtraInfoClientService;
    @Autowired
    private UserResourceClientService userResourceClientService;

    @Value("${xforce.tenant_security.appid:}")
    private String appid;


    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws IOException {
        if (handler instanceof ResourceHttpRequestHandler) {
            return true;
        }
        boolean isStaticPage = RequestUrlUtils.isStaticPage(request.getRequestURI());
        if (isStaticPage) {
            return true;
        }
        HandlerMethod method = ((HandlerMethod) handler);

        boolean isWithoutAuth = method.hasMethodAnnotation(WithoutAuth.class);
        if (isWithoutAuth) {
            if (StringUtils.isNotEmpty(defaultUserInfoJson)) {
                setUserInfoHolder(defaultUserInfoJson);
            }
            return true;
        }
        Cookie tokenCookie = WebUtils.getCookie(request, UserType.USER.tokenKey());
        String token;
        if (tokenCookie != null) {
            token = tokenCookie.getValue();
        } else {
            logger.warn("Cookie中无token");
            token = request.getHeader(UserType.USER.tokenKey());
            if (StringUtils.isEmpty(token)) {
                logger.warn("Header中无token");
                token = WebUtils.findParameterValue(request, UserType.USER.tokenKey());
            }
        }

        if (StringUtils.isBlank(token)) {
            String message = "访问失败，没有登录";
            logger.warn("token isBlank, " + message);

        }
        Map<String, String> claims = null;
        if (StringUtils.isNotBlank(token)) {
            try {
                claims = JwtUtils.verifyToken(token);
            } catch (TokenExpiredException e) {
                logger.error("token过期异常TokenExpiredException,token=={}", token);
                String message = "token过期，请重新登录";
                responseUnauthorized(message, request, response);
                return false;
            }
        }

        if (claims == null) {
            String message = "访问失败，没有登录";
            logger.warn("claims == null, " + message);
            responseUnauthorized(message, request, response);
            return false;
        }
        String systemType = claims.get(JwtConstants.TYPE_KEY);
        if (!UserType.USER.value().equals(systemType)) {
            String message = "token非法，请重新登录";
            logger.warn("claims == null, " + message);
            responseUnauthorized(message, request, response);
            return false;
        }
        String encodedUserInfo = claims.get(JwtConstants.USERINFO_KEY);  //被压缩的用户信息
        if (StringUtils.isBlank(encodedUserInfo)) {
            String message = "访问失败，无效令牌";
            logger.warn("userinfo isBlank, " + message);
            responseUnauthorized(message, request, response);
            return false;
        }
        String userinfo = CompressionUtils.decode(encodedUserInfo);
        if (StringUtils.isBlank(userinfo)) {
            String message = "访问失败，无效令牌";
            logger.warn("userinfo == null, " + message);
            responseUnauthorized(message, request, response);
            return false;
        }
        Authorization authorization = method.getMethodAnnotation(Authorization.class);
        if (authorization != null) {
            String[] resourceCodes = authorization.value();
            String checkStr;
            try {
                checkStr = checkResourceCode(token, resourceCodes);
            } catch (Exception e) {
                String message = "资源码校验发生异常";
                logger.error(message, e);
                responseUnauthorized(message, request, response);
                return false;
            }
            logger.info("资源码校验结果,msg = {}", checkStr);
            if (StringUtils.isNotBlank(checkStr)) {
                logger.warn("资源码校验没有通过, msg = {}", checkStr);
                responseUnauthorized(checkStr, request, response);
                return false;
            }
        }
        String tenantIdStr = request.getHeader("tenantId");
        if(tenantIdStr == null) {
            Map<String, String> pathVariables = (Map<String, String>) request.getAttribute(HandlerMapping.URI_TEMPLATE_VARIABLES_ATTRIBUTE);
            tenantIdStr = pathVariables.get("tenantId");
        }
        Long tenantId;
        try {
            tenantId = Long.parseLong(tenantIdStr);
        } catch (NumberFormatException e) {
            logger.warn(e.getMessage() + ", tenantId = " + tenantIdStr);
            tenantId = null;
        }
        try {
            return setUserInfoHolder(userinfo, token, tenantId, method);
        } catch (Exception e) {
            String message = "解析用户上下文发生异常";
            logger.error(message, e);
            responseUnauthorized(message, request, response);
            return false;
        }
    }

    private String checkResourceCode(String token, String[] resourceCodes) {
        if (resourceCodes == null || resourceCodes.length == 0) {
            return null;
        }
        MsCheckExistsResourceRequest msCheckExistsResourceRequest = new MsCheckExistsResourceRequest();
        msCheckExistsResourceRequest.setAppId(Integer.valueOf(appid));
        msCheckExistsResourceRequest.setResourceCodes(Lists.newArrayList(resourceCodes));
        MsCheckExistsResourceResponse msCheckExistsResourceResponse = userResourceClientService.userExtraInfoClient(token)
                .checkExistsResource(msCheckExistsResourceRequest);
        if (msCheckExistsResourceResponse.getCode() != 1) {
            String message = "资源码校验失败";
            if (StringUtils.isBlank(msCheckExistsResourceResponse.getMessage())) {
                message = msCheckExistsResourceResponse.getMessage();
            }
            return message;
        }
        if (!msCheckExistsResourceResponse.getCheck()) {
            return "资源码校验没通过";
        }
        return null;
    }

    private void setUserInfoHolder(String userinfo) {
        IAuthorizedUser userInfo = JsonUtils.fromJson(userinfo, AuthorizedUser.class);
        UserInfoHolder.put(userInfo);
    }

    private boolean setUserInfoHolder(String userinfo, String token, Long tenantId, HandlerMethod method) {
        AuthorizedUser userInfo = JsonUtils.fromJson(userinfo, AuthorizedUser.class);
        if(userInfo == null) {
            logger.warn("userinfo = " + userinfo);
            return false;
        }
        if (tenantId != null) {
            userInfo.setTenantId(tenantId);
        }

        UserInfoHolder.put(userInfo);
        NeedExtraInfo needExtraInfo = method.getMethodAnnotation(NeedExtraInfo.class);
        if (needExtraInfo != null) {
            if (needExtraInfo.resources() || needExtraInfo.orgs() || needExtraInfo.companies()
                    || needExtraInfo.currentOrgs() || needExtraInfo.parentCompanies()) {
                MsGetUserExtraInfoRequest msGetUserExtraInfoRequest = new MsGetUserExtraInfoRequest();
                try {
                    int appId = Integer.parseInt(appid);
                    msGetUserExtraInfoRequest.setAppId(appId);
                } catch (NumberFormatException e) {
                    logger.warn("为设置appid");
                    return false;
                }
                msGetUserExtraInfoRequest.setResources(needExtraInfo.resources());
                msGetUserExtraInfoRequest.setOrgs(needExtraInfo.orgs());
                msGetUserExtraInfoRequest.setParentCompanies(needExtraInfo.parentCompanies());
                msGetUserExtraInfoRequest.setCurrentOrgs(needExtraInfo.currentOrgs());
                msGetUserExtraInfoRequest.setCompanies(needExtraInfo.companies());
                MsGetUserExtraInfoResponse msGetUserExtraInfoResponse = userExtraInfoClientService.userExtraInfoClient(token).extraInfo(msGetUserExtraInfoRequest);
                ExtraInfoModel extraInfoModel =
                        JsonUtils.fromJson(msGetUserExtraInfoResponse.getInfoJson(), ExtraInfoModel.class);
                userInfo.setResourceCodes(extraInfoModel.getResourceCodes());
                userInfo.setOrgs(extraInfoModel.getOrgs());
                userInfo.setParentCompanies(extraInfoModel.getParentCompanies());
                userInfo.setCurrentOrgs(extraInfoModel.getCurrentOrgs());
                userInfo.setCompanies(extraInfoModel.getCompanies());
            }

        }
        userInfo.setToken(token);
        UserInfoHolder.put(userInfo);
        return true;
    }

    /**
     * 响应未认证请求
     *
     * @param response
     */
    private void responseUnauthorized(String message, HttpServletRequest request, HttpServletResponse response) throws IOException {
        boolean isAjaxRequest = RequestUtils.isAjaxRequest(request);
//        String redirectUri = request.getHeader("Referer");
//        if(redirectUri == null) {
//            redirectUri = request.getRequestURL().toString();
//        }
        String loginUrl = environment.getProperty("tenant.security.authority.login_url", authorityProperties.getLoginUrl());
        UriComponents uriComponents = UriComponentsBuilder.fromUriString(loginUrl)/*.queryParam("redirect", redirectUri)*/.build();
        String loginURI = uriComponents.toUriString();
        if (isAjaxRequest) {
            response.setCharacterEncoding(StandardCharsets.UTF_8.name());
            response.setContentType(MediaType.APPLICATION_JSON_UTF8_VALUE);
            response.setStatus(HttpServletResponse.SC_UNAUTHORIZED);
            Map<String, Object> error = new LinkedHashMap<>();
            error.put("code", "0");
            error.put("message", message);
            error.put("loginUrl", loginURI);
            response.getWriter().write(JsonUtils.toJson(error));
        } else {
            response.sendRedirect(loginURI);
        }
    }


    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) {
        UserInfoHolder.clearContext();
    }
}
