package com.xforceplus.tenantsecurity.interceptor;

import com.xforceplus.tenantsecurity.annotation.NeedExtraInfo;
import com.xforceplus.tenantsecurity.annotation.WithoutAuth;
import com.xforceplus.tenantsecurity.domain.AuthorizedUser;
import com.xforceplus.tenantsecurity.domain.UserInfoHolder;
import com.xforceplus.tenantsecurity.domain.UserType;
import com.xforceplus.tenantsecurity.feign.client.UserExtraInfoClientService;
import com.xforceplus.tenantsecurity.feign.model.ExtraInfoModel;
import com.xforceplus.tenantsecurity.feign.model.MsGetUserExtraInfoRequest;
import com.xforceplus.tenantsecurity.feign.model.MsGetUserExtraInfoResponse;
import com.xforceplus.tenantsecurity.utils.CompressionUtils;
import com.xforceplus.tenantsecurity.utils.JsonUtils;
import com.xforceplus.tenantsecurity.utils.RequestUrlUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.resource.ResourceHttpRequestHandler;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

/**
 * 微服务中 拦截http请求，解析请求头中的用户信息
 * @author        geewit
 * @since         2019-05-13
 */
public class UserContextInterceptor implements HandlerInterceptor {

    private static final Logger logger = LoggerFactory.getLogger(UserContextInterceptor.class);

    @Value("${xforce.tenant_security.starter.interceptors.default_user_info:}")
    private String defaultUserInfoJson;

    @Value("${xforce.tenant_security.appid:}")
    private String appid;

    @Autowired
    private UserExtraInfoClientService userExtraInfoClientService;

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) {
        if(handler instanceof ResourceHttpRequestHandler) {
            logger.info("handler instanceof ResourceHttpRequestHandler");
            return true;
        }
        boolean isStaticPage = RequestUrlUtils.isStaticPage(request.getRequestURI());
        logger.info("isStaticPage: " + isStaticPage);
        if(isStaticPage) {
            return true;
        }
        String userinfo = request.getHeader(UserType.USER.userinfoKey());
        logger.info("header.userinfo: " + userinfo);
        String deCompressedUserInfo;
        HandlerMethod method = ((HandlerMethod) handler);
        if (StringUtils.isEmpty(userinfo)) {
            boolean isWithoutAuth = method.hasMethodAnnotation(WithoutAuth.class);
            if (isWithoutAuth || StringUtils.isEmpty(defaultUserInfoJson)) {
                return true;
            }
            deCompressedUserInfo = defaultUserInfoJson;
            userinfo = CompressionUtils.encode(defaultUserInfoJson);
        } else {
            try {
                deCompressedUserInfo = CompressionUtils.decode(userinfo);
            } catch (Exception e) {
                logger.warn(e.getMessage());
                return false;
            }

        }
        logger.info("deCompressedUserInfo:" + deCompressedUserInfo);
        try {
            if (StringUtils.isNotEmpty(deCompressedUserInfo)) {
                return setUserInfoHolder(userinfo, deCompressedUserInfo, method);
            } else {
                logger.warn("deCompressedUserInfo == null");
                return true;
            }
        } catch (Exception e) {
            logger.error("解析用户上下文发生异常", e);
            return false;
        }
    }

    private boolean setUserInfoHolder(String userinfo, String deCompressedUserInfo, HandlerMethod method) {
        AuthorizedUser userInfo;
        try {
            userInfo = JsonUtils.fromJson(deCompressedUserInfo, AuthorizedUser.class);
        } catch (Exception e) {
            return true;
        }
        //region NeedExtraInfo
        NeedExtraInfo needExtraInfo = method.getMethodAnnotation(NeedExtraInfo.class);
        if(needExtraInfo != null) {
            if(needExtraInfo.resources()
                    || needExtraInfo.orgs() || needExtraInfo.companies()
                    || needExtraInfo.currentOrgs() || needExtraInfo.parentCompanies()) {
                MsGetUserExtraInfoRequest msGetUserExtraInfoRequest = new MsGetUserExtraInfoRequest();
                try {
                    int appId = Integer.parseInt(appid);
                    logger.info("appId:" + appId);
                    msGetUserExtraInfoRequest.setAppId(appId);
                } catch (NumberFormatException e) {
                    logger.warn("未设置appid");
                    return false;
                }
                msGetUserExtraInfoRequest.setResources(needExtraInfo.resources());
                msGetUserExtraInfoRequest.setOrgs(needExtraInfo.orgs());
                msGetUserExtraInfoRequest.setParentCompanies(needExtraInfo.parentCompanies());
                msGetUserExtraInfoRequest.setCurrentOrgs(needExtraInfo.currentOrgs());
                msGetUserExtraInfoRequest.setCompanies(needExtraInfo.companies());
                MsGetUserExtraInfoResponse msGetUserExtraInfoResponse = userExtraInfoClientService.userExtraInfoClient(userinfo).extraInfo(msGetUserExtraInfoRequest);
                ExtraInfoModel extraInfoModel =
                        JsonUtils.fromJson(msGetUserExtraInfoResponse.getInfoJson(), ExtraInfoModel.class);
                userInfo.setResourceCodes(extraInfoModel.getResourceCodes());
                userInfo.setOrgs(extraInfoModel.getOrgs());
                userInfo.setParentCompanies(extraInfoModel.getParentCompanies());
                userInfo.setCurrentOrgs(extraInfoModel.getCurrentOrgs());
                userInfo.setCompanies(extraInfoModel.getCompanies());
            }
        }
        //endregion
        UserInfoHolder.put(userInfo);
        return true;
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) {
        UserInfoHolder.clearContext();
    }
}
