package com.xforceplus.route.api;

import com.xforceplus.domain.resource.ResourceDto;
import com.xforceplus.domain.route.ServiceApiDto;
import com.xforceplus.route.api.common.Uri;
import com.xforceplus.route.api.common.model.ServiceApiModel.*;
import com.xforceplus.tenant.security.core.api.response.ResponseEntity;
import org.springframework.cloud.openfeign.SpringQueryMap;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import javax.validation.constraints.Min;
import java.util.Map;
import java.util.Set;

/**
 * @author geewit
 */
public interface ServiceApiApi {
    interface Path extends Uri {
        String API_PREFIX = PATH_PREFIX + PATH_GLOBAL + API_VERSION + "/service-apis";
        String LIST = API_PREFIX + "/list";
        String PAGE = API_PREFIX;
        String CREATE = API_PREFIX;
        String UPDATE = API_PREFIX + "/{id}";
        String INFO = API_PREFIX + "/{id}";
        String DELETE = API_PREFIX + "/{id}";
        String RESOURCES = API_PREFIX + "/{id}/resources";
        String BIND_RESOURCES = API_PREFIX + "/{id}/resources";
        String UNBIND_RESOURCES = API_PREFIX + "/resources/unbind";
    }

    /**
     * 服务包接口分页列表
     *
     * @param query
     * @param pageable
     * @param <T>
     * @return
     */
    @ResponseBody
    @RequestMapping(name = "服务包接口分页列表", value = Path.PAGE, method = RequestMethod.GET)
    <T extends ServiceApiDto> ResponseEntity<Page<T>> page(@SpringQueryMap Request.Query query,
                                                           Pageable pageable);

    /**
     * 新增服务包接口
     *
     * @param model
     * @param <T>
     * @return
     */
    @ResponseBody
    @RequestMapping(name = "新增服务包接口", value = Path.CREATE, method = RequestMethod.POST)
    <T extends ServiceApiDto> ResponseEntity<T> create(@RequestBody @Valid Request.Save model);

    /**
     * 服务包接口信息
     *
     * @param id
     * @param <T>
     * @return
     */
    @ResponseBody
    @RequestMapping(name = "服务包接口信息", value = Path.INFO, method = RequestMethod.GET)
    <T extends ServiceApiDto> ResponseEntity<T> info(@PathVariable("id") @Valid @Min(1) long id);

    /**
     * 更新服务包接口
     *
     * @param id
     * @param model
     * @param <T>
     * @return
     */
    @ResponseBody
    @RequestMapping(name = "更新服务包接口", value = Path.UPDATE, method = RequestMethod.PUT)
    <T extends ServiceApiDto> ResponseEntity<T> update(@PathVariable("id") @Valid @Min(1) long id,
                                                       @RequestBody Request.Update model);

    /**
     * 删除服务包接口
     *
     * @param id
     * @return
     */
    @ResponseBody
    @RequestMapping(name = "删除服务包接口", value = Path.DELETE, method = RequestMethod.DELETE)
    ResponseEntity<String> delete(@PathVariable("id") @Valid @Min(1) long id);

    /**
     * 服务包接口对应的资源码分页列表
     *
     * @param id
     * @param pageable
     * @param <T>
     * @return
     */
    @ResponseBody
    @RequestMapping(name = "服务包接口对应的资源码分页列表", value = Path.RESOURCES, method = RequestMethod.GET)
    <T extends ResourceDto> ResponseEntity<Page<T>> resources(@PathVariable("id") @Valid @Min(1) long id,
                                                                                   Pageable pageable);

    /**
     * 批量绑定功能集接口
     *
     * @param id
     * @param bindResources
     * @return
     */
    @ResponseBody
    @RequestMapping(name = "批量绑定功能集接口", value = Path.BIND_RESOURCES, method = RequestMethod.POST)
    ResponseEntity<String> bindResources(@PathVariable("id") @Valid @Min(1) long id,
                                         @RequestBody Request.BindResources bindResources);

    /**
     * 反绑定功能集接口
     *
     * @param serviceApiId
     * @param resourceId
     * @return
     */
    @ResponseBody
    @RequestMapping(name = "反绑定功能集接口", value = Path.UNBIND_RESOURCES, method = RequestMethod.DELETE)
    ResponseEntity<String> unbindResource(@RequestParam("serviceApiId") @Valid @Min(1) long serviceApiId,
                                          @RequestParam("resourceId") @Valid @Min(1) long resourceId);

    @ResponseBody
    @RequestMapping(name = "服务包接口列表", value = Path.LIST, method = RequestMethod.GET)
    ResponseEntity<Map<Long, Set<Response.ServiceApiWithResource>>> list();
}
