package com.xforceplus.utils;

import com.google.common.hash.Hashing;
import com.xforceplus.domain.route.RouteDto;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.server.PathContainer;
import org.springframework.web.util.UriComponentsBuilder;
import org.springframework.web.util.pattern.PathPattern;
import org.springframework.web.util.pattern.PathPatternParser;

import java.net.URI;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author geewit
 */
public class RouteUtils {
    private final static Logger logger = LoggerFactory.getLogger(RouteUtils.class);

    private final static PathPatternParser PATH_PATTERN_PARSER = PathPatternParser.defaultInstance;

    private final static String PATH_VARIABLE_NAME_PREFIX = "variable";

    public static String hash(String path) {
        path = reRenderPath(path);
        logger.info("route.RenderPath = {}", path);
        String hash = Hashing.sha1().newHasher().putString(path, StandardCharsets.UTF_8).hash().toString();
        return hash;
    }

    public static String reRenderPath(String path) {
        PathContainer pathContainer = PathContainer.parsePath(path);
        PathPattern pathPattern = PATH_PATTERN_PARSER.parse(path);
        PathPattern.PathMatchInfo pathMatchInfo = pathPattern.matchAndExtract(pathContainer);
        if(pathMatchInfo == null) {
            return path;
        }
        Map<String, String> pathVariables = pathMatchInfo.getUriVariables();
        Map<String, String> variables = new HashMap<>(pathVariables.size());
        List<Map.Entry<String, String>> pathEntries = null;
        if(!pathVariables.isEmpty()) {
            logger.debug("route.path.variables: {} ", pathVariables);
            Map.Entry<String, String>[] pathEntriesArray = new Map.Entry[path.length()];
            for (Map.Entry<String, String> entry : pathVariables.entrySet()) {
                int index = StringUtils.indexOf(path, entry.getValue());
                logger.debug("route.path entry.value = {}, index = {}", entry.getValue(), index);
                pathEntriesArray[index] = entry;
            }
            pathEntries = Arrays.stream(pathEntriesArray).filter(Objects::nonNull).collect(Collectors.toList());
            logger.debug("pathEntries = {}", pathEntries);
            for (int i = 0; i < pathEntries.size(); i++) {
                Map.Entry<String, String> pathEntry = pathEntries.get(i);
                variables.put(pathEntry.getKey(), "{" + PATH_VARIABLE_NAME_PREFIX + i + "}");
            }
        }
        if(pathEntries == null || pathEntries.isEmpty()) {
            return path;
        }
        try {
            URI uri = UriComponentsBuilder.fromPath(path).build(variables);
            return URLDecoder.decode(uri.toString(), StandardCharsets.UTF_8.name());
        } catch (Exception e) {
            logger.warn(e.getMessage());
            return path;
        }
    }

    public static void reRenderRoute(RouteDto route) {
        if(route == null) {
            return;
        }
        String path = route.getPath();
        if(path == null) {
            return;
        }
        String url = route.getUrl();
        if(url == null) {
            return;
        }
        PathContainer pathContainer = PathContainer.parsePath(path);
        PathPattern pathPattern = PATH_PATTERN_PARSER.parse(path);
        PathPattern.PathMatchInfo pathMatchInfo = pathPattern.matchAndExtract(pathContainer);
        if(pathMatchInfo == null) {
            return;
        }
        Map<String, String> pathVariables = pathMatchInfo.getUriVariables();
        Map<String, String> variables = new HashMap<>(pathVariables.size());
        List<Map.Entry<String, String>> pathEntries = null;
        if(!pathVariables.isEmpty()) {
            logger.debug("route.path.variables: {} ", pathVariables);
            Map.Entry<String, String>[] pathEntriesArray = new Map.Entry[path.length()];
            for (Map.Entry<String, String> entry : pathVariables.entrySet()) {
                int index = StringUtils.indexOf(path, entry.getValue());
                logger.debug("route.path entry.value = {}, index = {}", entry.getValue(), index);
                pathEntriesArray[index] = entry;
            }
            pathEntries = Arrays.stream(pathEntriesArray).filter(Objects::nonNull).collect(Collectors.toList());
            logger.debug("pathEntries = {}", pathEntries);
            for (int i = 0; i < pathEntries.size(); i++) {
                Map.Entry<String, String> pathEntry = pathEntries.get(i);
                variables.put(pathEntry.getKey(), "{" + PATH_VARIABLE_NAME_PREFIX + i + "}");
            }
        }
        if(pathEntries == null || pathEntries.isEmpty()) {
            return;
        }
        try {
            URI pathUri = UriComponentsBuilder.fromPath(path).build(variables);
            path = URLDecoder.decode(pathUri.toString(), StandardCharsets.UTF_8.name());
            route.setPath(path);
            URI urlUri = UriComponentsBuilder.fromHttpUrl(url).build(variables);
            url = URLDecoder.decode(urlUri.toString(), StandardCharsets.UTF_8.name());
            route.setUrl(url);
        } catch (Exception e) {
            logger.warn(e.getMessage());
        }
    }

    public static String reRenderUrl(String url) {
        PathContainer urlContainer = PathContainer.parsePath(url);
        PathPattern urlPattern = PATH_PATTERN_PARSER.parse(url);
        PathPattern.PathMatchInfo pathMatchInfo = urlPattern.matchAndExtract(urlContainer);
        if(pathMatchInfo == null) {
            return url;
        }
        Map<String, String> urlVariables = pathMatchInfo.getUriVariables();
        Map<String, String> variables = new HashMap<>(urlVariables.size());
        List<Map.Entry<String, String>> urlEntries = null;
        if(!urlVariables.isEmpty()) {
            logger.debug("url.variables: {} ", urlVariables);
            Map.Entry<String, String>[] pathEntriesArray = new Map.Entry[url.length()];
            for (Map.Entry<String, String> entry : urlVariables.entrySet()) {
                int index = StringUtils.indexOf(url, entry.getValue());
                logger.debug("url entry.value = {}, index = {}", entry.getValue(), index);
                pathEntriesArray[index] = entry;
            }
            urlEntries = Arrays.stream(pathEntriesArray).filter(Objects::nonNull).collect(Collectors.toList());
            logger.debug("urlEntries = {}", urlEntries);
            for (int i = 0; i < urlEntries.size(); i++) {
                Map.Entry<String, String> pathEntry = urlEntries.get(i);
                variables.put(pathEntry.getKey(), "{" + PATH_VARIABLE_NAME_PREFIX + i + "}");
            }
        }
        if(urlEntries == null || urlEntries.isEmpty()) {
            return url;
        }
        try {
            URI uri = UriComponentsBuilder.fromHttpUrl(url).build(variables);
            return URLDecoder.decode(uri.toString(), StandardCharsets.UTF_8.name());
        } catch (Exception e) {
            logger.warn(e.getMessage());
            return url;
        }
    }
}
