package com.xforceplus.jpa.listener;


import com.xforceplus.entity.ServiceApi;
import com.xforceplus.entity.ServiceApi_;
import com.xforceplus.utils.ApiUtils;
import org.apache.commons.lang3.StringUtils;
import org.hibernate.engine.spi.EntityEntry;
import org.hibernate.engine.spi.ManagedEntity;
import org.springframework.data.history.RevisionMetadata;
import org.springframework.web.bind.annotation.RequestMethod;

import javax.persistence.PostLoad;
import javax.persistence.PrePersist;
import javax.persistence.PreUpdate;
import java.util.Objects;


/**
 * 自动保存/更新的JPA 监听器
 *
 * @author geewit
 * @since 2020-01-15
 */
public class ServiceApiListener implements OperatorListener<ServiceApi> {
    @PrePersist
    public void prePersist(ServiceApi entity) {
        if (entity.getStatus() == null) {
            entity.setStatus(1);
        }
        if (StringUtils.isBlank(entity.getHash())) {
            this.setHash(RevisionMetadata.RevisionType.INSERT, entity);
        }
        OperatorListener.super.beforeInsert(entity);
        OperatorListener.super.beforeUpdate(entity);
        this.cleanRelatedEntities(entity);
    }

    @PreUpdate
    public void preUpdate(ServiceApi entity) {
        this.setHash(RevisionMetadata.RevisionType.UPDATE, entity);
        OperatorListener.super.beforeUpdate(entity);
        this.cleanRelatedEntities(entity);
    }

    private void setHash(RevisionMetadata.RevisionType revisionType, ServiceApi entity) {
        boolean newHash = false;
        if (RevisionMetadata.RevisionType.INSERT.equals(revisionType)) {
            newHash = true;
        } else if (entity instanceof ManagedEntity) {
            EntityEntry entityEntry = ((ManagedEntity) entity).$$_hibernate_getEntityEntry();
            if (entityEntry != null) {
                String preUrl = (String) entityEntry.getLoadedValue(ServiceApi_.SERVICE_API_URL);
                RequestMethod preRequestMethod = (RequestMethod) entityEntry.getLoadedValue(ServiceApi_.REQUEST_METHOD);
                Long preRouteId = (Long) entityEntry.getLoadedValue(ServiceApi_.ROUTE_ID);
                newHash = !Objects.equals(preUrl, entity.getServiceApiUrl())
                        || !Objects.equals(preRequestMethod, entity.getRequestMethod())
                        || !Objects.equals(preRouteId, entity.getRouteId());
            } else {
                newHash = entity.getServiceApiUrl() != null && entity.getRequestMethod() != null && entity.getRouteId() != null;
            }
        }
        if(newHash) {
            String hash = ApiUtils.hash(entity.getServiceApiUrl(), entity.getRequestMethod().name(), entity.getRouteId());
            entity.setHash(hash);
        }
    }

    private void cleanRelatedEntities(ServiceApi entity) {
        entity.setApp(null);
        entity.setResourceApiRels(null);
    }

    @PostLoad
    public void postLoad(ServiceApi entity) {
        if (entity.getApp() != null) {
            entity.setAppName(entity.getApp().getAppName());
        }
    }
}