package com.xforceplus.config;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.cache.annotation.CacheConfig;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.CachePut;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.cache.caffeine.CaffeineCacheManager;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.data.redis.cache.RedisCacheManager;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

@SuppressWarnings("all")
@AutoConfigureAfter({CachingConfig.class})
@Configuration
public class CacheInitConfig {
    private final static Logger logger = LoggerFactory.getLogger(CacheInitConfig.class);

    final static Set<String> CACHE_NAMES = new HashSet<>();

    public CacheInitConfig(ApplicationContext applicationContext, CaffeineCacheManager caffeineCacheManager, RedisCacheManager.RedisCacheManagerBuilder redisCacheManagerBuilder) {
        String[] beanNames = applicationContext.getBeanDefinitionNames();
        for (String beanName : beanNames) {
            if (StringUtils.contains(beanName, ".")) {
                continue;
            }

            try {
                Object bean = applicationContext.getBean(beanName);
                Class<?> clazz = bean.getClass();
                String className = clazz.getName();
                if (StringUtils.startsWith(className, "com.xforceplus.")) {
                    this.setCacheNames(clazz);
                }
            } catch (BeansException ignore) {
            }
        }

        caffeineCacheManager.setCacheNames(CACHE_NAMES);
        redisCacheManagerBuilder.initialCacheNames(CACHE_NAMES);
    }

    private void setCacheNames(Class<?> clazz) {
        boolean foundAnnotation = false;
        CacheConfig cacheConfig = AnnotationUtils.getAnnotation(clazz, CacheConfig.class);
        if (cacheConfig != null) {
            CACHE_NAMES.addAll(Arrays.asList(cacheConfig.cacheNames()));
            foundAnnotation = true;
        }
        Method[] methods = clazz.getDeclaredMethods();
        for (Method method : methods) {
            CachePut cachePut = AnnotationUtils.findAnnotation(method, CachePut.class);
            if (cachePut != null) {
                CACHE_NAMES.addAll(Arrays.asList(cachePut.value()));
                foundAnnotation = true;
            }
            Cacheable cacheable = AnnotationUtils.findAnnotation(method, Cacheable.class);
            if (cacheable != null) {
                CACHE_NAMES.addAll(Arrays.asList(cacheable.value()));
                foundAnnotation = true;
            }
            CacheEvict cacheEvict = AnnotationUtils.findAnnotation(method, CacheEvict.class);
            if (cacheEvict != null) {
                CACHE_NAMES.addAll(Arrays.asList(cacheEvict.value()));
                foundAnnotation = true;
            }
        }
        if (foundAnnotation) {
            logger.info("beanName = {}", clazz.getName());
        }
    }
}
