package com.xforceplus.config;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.datatype.hibernate5.Hibernate5Module;
import com.github.benmanes.caffeine.cache.Caffeine;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.cache.CacheManager;
import org.springframework.cache.annotation.CachingConfigurerSupport;
import org.springframework.cache.annotation.EnableCaching;
import org.springframework.cache.caffeine.CaffeineCacheManager;
import org.springframework.cache.support.CompositeCacheManager;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.data.redis.cache.RedisCacheConfiguration;
import org.springframework.data.redis.cache.RedisCacheManager;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.GenericJackson2JsonRedisSerializer;
import org.springframework.data.redis.serializer.RedisSerializationContext;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;

import java.util.concurrent.TimeUnit;


/**
 * 缓存配置
 *
 * @author geewit
 */
@AutoConfigureBefore({CacheInitConfig.class})
@Configuration
@EnableCaching
public class CachingConfig extends CachingConfigurerSupport {
    private final static Logger logger = LoggerFactory.getLogger(CachingConfig.class);

    private final LettuceConnectionFactory lettuceConnectionFactory;

    private final ObjectMapper objectMapper;

    public CachingConfig(LettuceConnectionFactory lettuceConnectionFactory, ObjectMapper objectMapper) {
        this.lettuceConnectionFactory = lettuceConnectionFactory;
        this.objectMapper = objectMapper;
    }

    /**
     * redis缓存失效时间, 默认 3600 秒
     */
    @Value("${xforce.tenant.cache.redis.time-live-seconds:3600}")
    private Long redisTimeLiveSeconds;

    /**
     * caffeine本地缓存失效时间, 默认 3 秒
     */
    @Value("${xforce.tenant.cache.caffeine.time-live-seconds:3}")
    private Long caffeineTimeLiveSeconds;

    @Bean("redisCacheManagerBuilder")
    public RedisCacheManager.RedisCacheManagerBuilder redisCacheManagerBuilder() {
        RedisCacheConfiguration redisCacheConfiguration = RedisCacheConfiguration.defaultCacheConfig()
                .disableCachingNullValues()
                .entryTtl(java.time.Duration.ofSeconds(redisTimeLiveSeconds))
                .serializeKeysWith(RedisSerializationContext.SerializationPair.fromSerializer(this.keySerializer()))
                .serializeValuesWith(RedisSerializationContext.SerializationPair.fromSerializer(this.redisSerializer()));
        redisCacheConfiguration.usePrefix();

        RedisCacheManager.RedisCacheManagerBuilder redisCacheManagerBuilder = RedisCacheManager.RedisCacheManagerBuilder
                .fromConnectionFactory(lettuceConnectionFactory)
                .cacheDefaults(redisCacheConfiguration);
        return redisCacheManagerBuilder;
    }

    private RedisSerializer<String> keySerializer() {
        return new StringRedisSerializer();
    }


    private GenericJackson2JsonRedisSerializer redisSerializer() {
        ObjectMapper objectMapper = this.objectMapper.copy();
        Hibernate5Module hibernate5Module = new Hibernate5Module();
        hibernate5Module.disable(Hibernate5Module.Feature.USE_TRANSIENT_ANNOTATION);
        hibernate5Module.enable(Hibernate5Module.Feature.REPLACE_PERSISTENT_COLLECTIONS);
        hibernate5Module.enable(Hibernate5Module.Feature.SERIALIZE_IDENTIFIER_FOR_LAZY_NOT_LOADED_OBJECTS);
        objectMapper.registerModule(hibernate5Module);
        objectMapper.enable(MapperFeature.USE_ANNOTATIONS);
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        objectMapper.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
        objectMapper.setVisibility(PropertyAccessor.ALL, JsonAutoDetect.Visibility.ANY);
        objectMapper.enableDefaultTyping(ObjectMapper.DefaultTyping.NON_FINAL, JsonTypeInfo.As.PROPERTY);
        objectMapper.disable(SerializationFeature.FAIL_ON_EMPTY_BEANS);
        GenericJackson2JsonRedisSerializer serializer = new GenericJackson2JsonRedisSerializer(objectMapper);
        return serializer;
    }

    @Bean
    public RedisTemplate<String, Object> redisTemplate(RedisConnectionFactory connectionFactory) {
        RedisTemplate<String, Object> template = new RedisTemplate<>();
        template.setKeySerializer(new StringRedisSerializer());
        template.setValueSerializer(redisSerializer());
        template.setConnectionFactory(connectionFactory);
        return template;
    }

    @Bean("caffeineCacheManager")
    public CaffeineCacheManager caffeineCacheManager() {
        CaffeineCacheManager caffeineCacheManager = new CaffeineCacheManager();
        caffeineCacheManager.setAllowNullValues(true);
        caffeineCacheManager.setCaffeine(Caffeine.newBuilder().recordStats()
                .expireAfterWrite(caffeineTimeLiveSeconds, TimeUnit.SECONDS)
                .maximumSize(2000L));
        return caffeineCacheManager;
    }

    @Override
    @Primary
    @Bean
    public CacheManager cacheManager() {
        CompositeCacheManager cacheManager = new CompositeCacheManager(caffeineCacheManager(), redisCacheManagerBuilder().build());
        cacheManager.setFallbackToNoOpCache(true);
        return cacheManager;
    }

}
