package com.xforceplus.route.controller;


import com.xforceplus.entity.RouteApply;
import com.xforceplus.route.HttpCheckUtils;
import com.xforceplus.route.api.common.model.RouteModel;
import com.xforceplus.route.service.RouteApplyService;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.api.response.ResponseEntity;
import com.xforceplus.utils.RouteUtils;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.geewit.data.jpa.essential.domain.SortFactory;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Controller;
import org.springframework.util.Assert;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.annotation.Resource;
import java.util.List;

/**
 * @author geewit
 */
@Controller
@RequestMapping(value = "/api/global/route-apply")
public class RouteApplyController {
    private final static Logger logger = LoggerFactory.getLogger(RouteApplyController.class);

    @Resource
    private RouteApplyService routeApplyService;

    @AuthorizedDefinition(authorization = false)
    @ResponseBody
    @RequestMapping(name = "路由列表", value = "", method = RequestMethod.GET)
    public ResponseEntity<List<RouteApply>> list(Sort sort) {
        Sort currentSort = SortFactory.ofDefaultSort(sort, Sort.by(Sort.Direction.DESC, "path"));
        List<RouteApply> list = routeApplyService.findAll(currentSort);
        return ResponseEntity.ok(list);
    }

    @AuthorizedDefinition(authorization = false)
    @ResponseBody
    @RequestMapping(name = "路由分页列表", value = "/page", method = {RequestMethod.GET})
    public ResponseEntity<Page<RouteApply>> page(Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "path"));
        Page<RouteApply> page = routeApplyService.findAll(currentPageable);
        return ResponseEntity.ok(page);
    }

    @AuthorizedDefinition(authorization = false)
    @ResponseBody
    @RequestMapping(name = "新增路由申请", value = "", method = RequestMethod.POST)
    public ResponseEntity<RouteApply> create(RouteModel.Request.Save routeForm) {
        this.checkCreatePara(routeForm);
        if (StringUtils.isBlank(routeForm.getPath())) {
            throw new IllegalArgumentException("匹配路径不能为空");
        } else {
            String hash = RouteUtils.hash(routeForm.getPath());
            routeForm.setHash(hash);
            long count = routeApplyService.countByHash(hash);
            if (count > 0) {
                String message = "重复的匹配路径(" + routeForm.getPath() + ")";
                throw new IllegalArgumentException(message);
            }
        }
        RouteApply route = new RouteApply();
        BeanUtils.copyProperties(routeForm, route);
        route = routeApplyService.create(route);
        return ResponseEntity.ok(route);
    }

    @AuthorizedDefinition(authorization = false)
    @ResponseBody
    @RequestMapping(name = "更新路由", value = "/{id}", method = RequestMethod.PUT)
    public ResponseEntity<RouteApply> update(Long id, RouteModel.Request.Save routeForm) {
        this.checkUpdatePara(routeForm);
        RouteApply route = new RouteApply();
        BeanUtils.copyProperties(routeForm, route);
        route = routeApplyService.update(id, route);
        return ResponseEntity.ok(route);
    }

    @AuthorizedDefinition(authorization = false)
    @ResponseBody
    @RequestMapping(name = "删除路由", value = "/{id}", method = RequestMethod.DELETE)
    public ResponseEntity<String> delete(Long id) {
        routeApplyService.delete(id);
        return ResponseEntity.ok("删除成功");
    }

    @AuthorizedDefinition(authorization = false)
    @ResponseBody
    @RequestMapping(name = "更改状态", value = "/{id}/status/{status}", method = {RequestMethod.PUT})
    public ResponseEntity<RouteApply> status(@PathVariable("id") Long id,
                                             @PathVariable("status") Integer status) {
        RouteApply routeApply = routeApplyService.changeStatus(id, status);
        return ResponseEntity.ok(routeApply);
    }

    private void checkCreatePara(RouteModel.Request.Save routeForm) {
        Assert.notNull(routeForm.getAppId(), "appId不能为空。");
        Assert.notNull(routeForm.getPath(), "匹配路径不能为空。");
        Assert.notNull(routeForm.getRouteName(), "路由名称不能为空。");

        HttpCheckUtils.checkHttpUrl(routeForm.getUrl());
    }

    private void checkUpdatePara(RouteModel.Request.Save routeForm) {
        if (StringUtils.isNotBlank(routeForm.getUrl())) {
            HttpCheckUtils.checkHttpUrl(routeForm.getUrl());
        }
    }

}
