package com.xforceplus.route.controller;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.xforceplus.entity.AutoscanRecord;
import com.xforceplus.route.api.common.model.RouteModel.Request.*;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.api.response.ResponseEntity;
import com.xforceplus.route.api.RouteApi;
import com.xforceplus.api.global.autoscan.AutoscanApi;
import com.xforceplus.route.constants.AutoscanStatus;
import com.xforceplus.route.service.AutoscanRecordService;
import com.xforceplus.route.service.RouteService;
import com.xforceplus.entity.Route;
import com.xforceplus.tenant.security.autoscan.model.AutoScanBody;
import com.xforceplus.utils.RouteUtils;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.geewit.data.jpa.essential.domain.SortFactory;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Controller;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.*;

import static com.xforceplus.route.api.RouteApi.Path.*;

/**
 * @author geewit
 */
@Validated
@Controller
public class RouteController implements RouteApi, AutoscanApi {
    private final static Logger logger = LoggerFactory.getLogger(RouteController.class);

    ThreadFactory namedThreadFactory = new ThreadFactoryBuilder().setNameFormat("tenant-autoscan-pool-%d").build();

    private final AutoscanRecordService autoscanRecordService;

    private final RouteService routeService;

    public RouteController(RouteService routeService, AutoscanRecordService autoscanRecordService) {
        this.routeService = routeService;
        this.autoscanRecordService = autoscanRecordService;
    }

    /***
     * 线程池
     */
    private final ExecutorService executorService = new ThreadPoolExecutor(1, 1,
            0L, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<>(),
            namedThreadFactory, new ThreadPoolExecutor.AbortPolicy());

    @AuthorizedDefinition(authorization = false)
    @Override
    public ResponseEntity<List<Route>> list(Query query, Sort sort) {
        Sort currentSort = SortFactory.ofDefaultSort(sort, Sort.by(Sort.Direction.DESC, "path"));
        List<Route> result = routeService.findAll(query, currentSort);
        for (Route route : result) {
            RouteUtils.reRenderRoute(route);
        }
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(authorization = false)
    @Override
    public ResponseEntity<Page<Route>> page(Query query, Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "path"));
        Page<Route> result = routeService.findAll(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(authorization = false)
    @Override
    public ResponseEntity<Route> create(Save routeForm) {
        if(StringUtils.isBlank(routeForm.getPath())) {
            throw new IllegalArgumentException("匹配路径不能为空");
        } else {
            routeForm.setPath(StringUtils.trim(routeForm.getPath()));
            routeForm.setUrl(StringUtils.trim(routeForm.getUrl()));
            String hash = RouteUtils.hash(routeForm.getPath());
            long count = routeService.countByHash(hash);
            if(count > 0) {
                String message = "重复的匹配路径(" + routeForm.getPath() + ")";
                throw new IllegalArgumentException(message);
            }
            routeForm.setHash(hash);
        }
        Route result = new Route();
        BeanUtils.copyProperties(routeForm, result);
        result = routeService.create(result);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(authorization = false)
    @Override
    public ResponseEntity<Route> update(Long id, Save routeForm) {
        Route result = new Route();
        if (StringUtils.isNotBlank(routeForm.getPath())) {
            routeForm.setPath(StringUtils.trim(routeForm.getPath()));
        }
        if (StringUtils.isNotBlank(routeForm.getUrl())) {
            routeForm.setUrl(StringUtils.trim(routeForm.getUrl()));
        }
        BeanUtils.copyProperties(routeForm, result);
        result = routeService.update(id, result);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(authorization = false)
    @Override
    public ResponseEntity<String> delete(Long id) {
        routeService.delete(id);
        return ResponseEntity.ok("删除成功");
    }

    @AuthorizedDefinition(authorization = false)
    @ResponseBody
    @RequestMapping(name = "刷新路由Hash", value = API_PREFIX + "/refresh/hash", method = { RequestMethod.POST })
    public ResponseEntity<String> refreshHash() {
        routeService.refreshHash();
        return ResponseEntity.ok("刷新成功");
    }

    @AuthorizedDefinition(authorization = false)
    @ResponseBody
    @RequestMapping(name = "刷新路由", value = API_PREFIX + "/refresh", method = { RequestMethod.POST })
    public String refresh() {
        return routeService.refresh();
    }

    @AuthorizedDefinition(authorization = false)
    @ResponseBody
    @RequestMapping(name = "刷新路由", value = API_PREFIX + "/refresh/{refreshId}", method = { RequestMethod.GET })
    public String refreshCount(@PathVariable("refreshId") String refreshId) {
        return routeService.getRefreshCount(refreshId);
    }

    /**
     * 根据路由id扫描接口
     *
     * @return
     */
    @AuthorizedDefinition(authorization = false)
    @Override
    @ApiOperation(value = "根据路由id扫描接口", notes = "根据路由id扫描接口")
    public ResponseEntity<String> scan(@PathVariable Long routeId) {
        executorService.submit(() -> {
            try {
                logger.info("开始执行。");
                routeService.scanResource(routeId);
            } catch (Exception e) {
                logger.error("执行报错。", e);
            }
        });
        return ResponseEntity.okWithCode("成功");
    }

    /**
     * 根据路由id扫描接口
     *
     * @return
     */
    @AuthorizedDefinition(authorization = false)
    @Override
    @ApiOperation(value = "批量扫描接口", notes = "批量扫描接口")
    public ResponseEntity<String> batchScan(@RequestBody AutoScanBody autoScanBody) {
        if (StringUtils.isBlank(autoScanBody.getRouteIds())) {
            return ResponseEntity.ok("入参有误");
        }
        AutoscanRecord autoscanRecord = autoscanRecordService.save(autoScanBody.getRouteIds(), autoScanBody.getIsServicePackage());
        if (AutoscanStatus.CREATED.equals(autoscanRecord.getStatus())) {
            executorService.submit(() -> {
                try {
                    Set<Long> routeIds = new HashSet<>();
                    if (StringUtils.isNotBlank(autoScanBody.getRouteIds())) {
                        Arrays.stream(StringUtils.split(autoScanBody.getRouteIds(), ",")).forEach(id -> {
                            try {
                                Long parseLong = Long.parseLong(id);
                                routeIds.add(parseLong);
                            } catch (NumberFormatException e) {
                                String message = e.getMessage() + ", id: " + id;
                                logger.warn(message);
                            }
                        });
                    }
                    if (!routeIds.isEmpty()) {
                        try {
                            routeService.scanResourceByRequest(routeIds, autoScanBody);
                        } catch (Exception e) {
                            logger.warn(e.getMessage(), e);
                        }
                    }

                    autoscanRecord.setStatus(AutoscanStatus.DONE);
                    autoscanRecordService.updateStatus(autoscanRecord);
                } catch (Exception e) {
                    autoscanRecord.setStatus(AutoscanStatus.FAILED);
                    autoscanRecordService.updateStatus(autoscanRecord);
                    logger.error("执行报错", e);
                }
            });
        }
        return ResponseEntity.okWithCode("成功");
    }
}
