package com.xforceplus.route.controller;


import com.xforceplus.entity.Resource;
import com.xforceplus.entity.ServiceApi;
import com.xforceplus.route.api.ServiceApiApi;
import com.xforceplus.route.api.common.model.ResourceModel;
import com.xforceplus.route.api.common.model.ServiceApiModel.*;
import com.xforceplus.route.service.ResourceService;
import com.xforceplus.tenant.security.autoscan.annotation.AuthorizedDefinition;
import com.xforceplus.tenant.security.core.api.response.ResponseEntity;
import com.xforceplus.route.service.ServiceApiService;
import com.xforceplus.utils.ApiUtils;
import io.geewit.data.jpa.essential.domain.PageableFactory;
import io.geewit.data.jpa.essential.search.DynamicSpecifications;
import io.geewit.data.jpa.essential.search.SearchFilter;
import io.geewit.data.jpa.essential.web.servlet.Servlets;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseBody;
import org.springframework.web.context.request.WebRequest;
import springfox.documentation.annotations.ApiIgnore;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

import static com.xforceplus.route.api.ServiceApiApi.Path.API_PREFIX;

/**
 * @author geewit
 */
@Controller
public class ServiceApiController implements ServiceApiApi {
    private final static Logger logger = LoggerFactory.getLogger(ServiceApiController.class);

    private final ServiceApiService serviceApiService;

    private final ResourceService resourceService;

    public ServiceApiController(ServiceApiService serviceApiService, ResourceService resourceService) {
        this.serviceApiService = serviceApiService;
        this.resourceService = resourceService;
    }

    @AuthorizedDefinition(authorization = false)
    @Override
    public ResponseEntity<Map<Long, Set<Response.ServiceApiWithResource>>> list() {
        Map<Long, Set<Response.ServiceApiWithResource>> result = serviceApiService.all();
        return ResponseEntity.ok(result);
    }

    /**
     * 获取功能集分页列表
     *
     * @param request
     * @param pageable
     * @return
     */
    @AuthorizedDefinition(resources = {"xforce:operation:serviceApi:read"})
    @ApiIgnore
    @ApiOperation(value = "获取服务接口分页", notes = "通过分页参数，获取服务接口分页")
    @ResponseBody
    @RequestMapping(name = "服务接口分页列表查询", value = {API_PREFIX + "/query"}, method = RequestMethod.GET)
    public Page<ServiceApi> page(WebRequest request, @ApiParam(value = "pageable") Pageable pageable) {
        Map<String, Object> searchParams = Servlets.getParametersStartingWith(request, "search_");
        Collection<SearchFilter> filters = SearchFilter.parse(searchParams);
        Specification<ServiceApi> specification = DynamicSpecifications.bySearchFilter(filters);
        Page<ServiceApi> page = serviceApiService.page(specification, pageable);
        return page;
    }

    @AuthorizedDefinition(resources = {"xforce:operation:serviceApi:read"})
    @ApiOperation(value = "获取服务接口分页", notes = "通过分页参数，获取服务接口分页")
    @Override
    public ResponseEntity<Page<ServiceApi>> page(@ApiParam(value = "request") Request.Query query,
                                                 @ApiParam(value = "pageable") Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));
        Page<ServiceApi> result = serviceApiService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:serviceApi:save"})
    @ApiOperation(value = "新建服务接口")
    @Override
    public ResponseEntity<ServiceApi> create(Request.Save model) {
        //Fix bug: 不能为空
        if (model.getServiceApiPath() == null) {
            throw new IllegalArgumentException("ServiceApiPath 不能为空");
        }
        if (StringUtils.isBlank(model.getServiceApiUrl())) {
            throw new IllegalArgumentException("ServiceApiUrl 不能为空");
        }
        if (model.getRequestMethod() == null) {
            throw new IllegalArgumentException("RequestMethod 不能为空");
        }
        String hash = model.getHash();
        if (StringUtils.isBlank(hash)) {
            hash = ApiUtils.hash(model.getServiceApiUrl(), model.getRequestMethod().name(), model.getRouteId());
        }

        long count = serviceApiService.countByHash(hash);
        if (count > 0) {
            String message = "存在重复的ServiceApi(path:" + model.getServiceApiUrl() + ", method:" + model.getRequestMethod() + ", routeId:" + model.getRouteId() + ")";
            throw new IllegalArgumentException(message);
        }
        model.setHash(hash);
        ServiceApi result = serviceApiService.save(model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:serviceApi:read"})
    @ApiOperation(value = "根据id获取服务接口")
    @Override
    public ResponseEntity<ServiceApi> info(@ApiParam(value = "id", required = true) long id) {
        ServiceApi result = serviceApiService.findById(id);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:serviceApi:save"})
    @ApiOperation(value = "根据id更新服务接口")
    @Override
    public ResponseEntity<ServiceApi> update(@ApiParam(value = "id", required = true) long id,
                                             @ApiParam(value = "model", required = true) Request.Update model) {
        ServiceApi result = serviceApiService.update(id, model);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:serviceApi:save"})
    @ApiOperation(value = "根据id删除服务接口")
    @Override
    public ResponseEntity<String> delete(@ApiParam(value = "id", required = true) long id) {
        serviceApiService.deleteById(id);
        return ResponseEntity.ok("删除成功");
    }

    @AuthorizedDefinition(resources = {"xforce:operation:serviceApi:read"})
    @ApiOperation(value = "服务包接口对应的资源码分页列表")
    @Override
    public ResponseEntity<Page<Resource>> resources(@ApiParam("接口id") long id, Pageable pageable) {
        Pageable currentPageable = PageableFactory.ofDefaultSort(pageable, Sort.by(Sort.Direction.DESC, "createTime"));

        ServiceApi serviceApi = serviceApiService.findById(id);
        ResourceModel.Request.Query query = new ResourceModel.Request.Query();
        query.setServiceApiId(serviceApi.getServiceApiId());
        Page<Resource> result = resourceService.page(query, currentPageable);
        return ResponseEntity.ok(result);
    }

    @AuthorizedDefinition(resources = {"xforce:operation:serviceApi:read"})
    @ApiOperation(value = "接口批量绑定资源码集合")
    @Override
    public ResponseEntity<String> bindResources(@ApiParam("接口id") long id, Request.BindResources bindResources) {
        serviceApiService.bindResources(id, bindResources);
        return ResponseEntity.ok("绑定成功");
    }

    @AuthorizedDefinition(resources = {"xforce:operation:serviceApi:save"})
    @ApiOperation(value = "接口反绑定资源码")
    @Override
    public ResponseEntity<String> unbindResource(long serviceApiId, long resourceId) {
        serviceApiService.unbindServiceApiResourceRel(serviceApiId, resourceId);
        return ResponseEntity.ok("反绑定成功");
    }

    @AuthorizedDefinition(authorization = false)
    @ResponseBody
    @RequestMapping(name = "刷新接口Hash", value = API_PREFIX + "/refresh/hash", method = {RequestMethod.POST})
    public ResponseEntity<String> refreshHash(@RequestParam(value = "routeId", required = false) Long routeId) {
        if (routeId != null && routeId > 0) {
            serviceApiService.refreshHashByRouteId(routeId);
        } else {
            serviceApiService.refreshHash();
        }
        return ResponseEntity.ok("刷新成功");
    }
}
