package com.xforceplus.route.service;

import com.google.common.collect.Lists;
import com.xforceplus.dao.RouteApplyDao;
import com.xforceplus.entity.RouteApply;
import com.xforceplus.route.ApplyStatus;
import io.geewit.core.utils.reflection.BeanUtils;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

/**
 * @author geewit
 */
@Service
public class RouteApplyService {
    private final RouteApplyDao routeApplyDao;
    private final RouteService routeService;

    public RouteApplyService(RouteApplyDao routeApplyDao, RouteService routeService) {
        this.routeApplyDao = routeApplyDao;
        this.routeService = routeService;
    }

    @Transactional(rollbackFor = Exception.class)
    public RouteApply create(RouteApply route) {
        route.setRouteApplyId(null);
        route.setStatus(ApplyStatus.APPLY);
        return routeApplyDao.save(route);
    }

    @Transactional(rollbackFor = Exception.class)
    public RouteApply update(Long id, RouteApply route) {
        RouteApply existRouteApply = routeApplyDao.findById(id).orElseThrow(() -> new IllegalArgumentException("未找到该路由申请(" + id + ")"));
        if (existRouteApply.getStatus() == null || existRouteApply.getStatus() != ApplyStatus.APPLY) {
            throw new IllegalArgumentException("该申请记录已经受理，无法更改。");
        }
        BeanUtils.copyProperties(route, existRouteApply);
        return routeApplyDao.save(existRouteApply);
    }

    @Transactional(rollbackFor = Exception.class)
    public void delete(Long routeApplyId) {
        routeApplyDao.deleteById(routeApplyId);
    }

    public List<RouteApply> findAll(Sort sort) {
        return Lists.newArrayList(routeApplyDao.findAll(sort));
    }

    public Page<RouteApply> findAll(Pageable pageable) {
        return routeApplyDao.findAll(pageable);
    }

    @Transactional(rollbackFor = Exception.class)
    public RouteApply changeStatus(Long id, Integer status) {
        RouteApply existRouteApply = routeApplyDao.findById(id).orElseThrow(() -> new IllegalArgumentException("未找到该路由申请(" + id + ")"));
        if (existRouteApply.getStatus() == null || existRouteApply.getStatus() != ApplyStatus.APPLY) {
            throw new IllegalArgumentException("该申请记录已经受理。");
        }
        if (status == null || (status != ApplyStatus.APPROVED && status != ApplyStatus.REJECTED)) {
            throw new IllegalArgumentException("传入状态有误。");
        }
        existRouteApply.setStatus(status);
        if (existRouteApply.getStatus() == ApplyStatus.APPROVED) {
            routeService.createByRouteApply(existRouteApply);
        }
        return routeApplyDao.save(existRouteApply);
    }

    public long countByHash(String hash) {
        return routeApplyDao.countByHash(hash);
    }
}
