package com.xforceplus.route.service;

import com.xforceplus.dao.AutoscanRecordDao;
import com.xforceplus.dao.RouteDao;
import com.xforceplus.entity.AutoscanRecord;
import com.xforceplus.route.constants.AutoscanStatus;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;


/**
 * 自动扫描Service
 *
 * @author lengmz
 */
@SuppressWarnings("all")
@Service
@Slf4j
public class AutoscanRecordService {

    private final static String REDIS_PREFIX = "XF_AUTOSCAN_";

    private final AutoscanRecordDao autoscanRecordDao;

    private final RouteDao routeDao;

    private final RedisTemplate<String, String> redisTemplate;

    public AutoscanRecordService(AutoscanRecordDao autoscanRecordDao, RouteDao routeDao, RedisTemplate<String, String> redisTemplate) {
        this.autoscanRecordDao = autoscanRecordDao;
        this.routeDao = routeDao;
        this.redisTemplate = redisTemplate;
    }

    @Transactional(rollbackFor = Exception.class)
    public AutoscanRecord save(String routeIds, String routePaths, Boolean isServicePackage) {
        AutoscanRecord autoscanRecord = new AutoscanRecord();
        autoscanRecord.setRouteIds(routeIds);
        autoscanRecord.setRoutePaths(routePaths);
        autoscanRecord.setIsServicePackage(isServicePackage);
        Set<Long> ids = Arrays.stream(StringUtils.split(routeIds, ","))
                .map(routeId -> {
                    try {
                        return Long.parseLong(routeId);
                    } catch (NumberFormatException e) {
                        log.warn("input: {}, e:{}", routeId, e.getMessage());
                        return null;
                    }
                }).filter(Objects::nonNull)
                .collect(Collectors.toSet());
        Set<String> existsPaths = routeDao.findPathByRouteIds(ids);
        Set<String> inputPaths = Arrays.stream(StringUtils.split(routePaths, ",")).collect(Collectors.toSet());
        Set<String> diffPaths = inputPaths.stream().filter(path -> !existsPaths.contains(path)).collect(Collectors.toSet());
        if (!diffPaths.isEmpty()) {
            throw new IllegalArgumentException("请检查配置中的 routeIds 和 routePaths 是否正确");
        }

        String key = REDIS_PREFIX + routeIds.trim().replace(",", "_");
        boolean result = redisTemplate.opsForValue().setIfAbsent(key, "exists", 10, TimeUnit.MINUTES);
        if (result) {
            autoscanRecord.setStatus(AutoscanStatus.CREATED);
        } else {
            autoscanRecord.setStatus(AutoscanStatus.INVALID);
        }
        autoscanRecord.setCreateTime(new Date());
        autoscanRecord.setUpdateTime(new Date());
        return autoscanRecordDao.saveAndFlush(autoscanRecord);
    }

    @Transactional(rollbackFor = Exception.class)
    public AutoscanRecord updateStatus(AutoscanRecord autoscanRecord) {
        autoscanRecord.setUpdateTime(new Date());
        return autoscanRecordDao.saveAndFlush(autoscanRecord);
    }

}
