package com.xforceplus.route.service;

import com.xforceplus.dao.ResourceApiRelDao;
import com.xforceplus.dao.ResourceDao;
import com.xforceplus.dao.RouteDao;
import com.xforceplus.dao.ServiceApiDao;
import com.xforceplus.entity.*;
import com.xforceplus.route.dto.ResourceAndPathDto;
import com.xforceplus.tenant.security.autoscan.model.AuthorizationInfo;
import com.xforceplus.tenant.security.autoscan.model.AuthorizationUri;
import com.xforceplus.tenant.security.autoscan.model.AutoScanBody;
import com.xforceplus.tenant.security.autoscan.model.ResourceItem;
import com.xforceplus.utils.ApiUtils;
import com.xforceplus.utils.RouteUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.http.server.PathContainer;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.util.UriComponentsBuilder;
import org.springframework.web.util.pattern.PathPattern;
import org.springframework.web.util.pattern.PathPatternParser;

import javax.persistence.criteria.Predicate;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author geewit
 */
@Service
@Slf4j
public class ResourceAndPathService {

    private final RouteDao routeDao;
    private final ServiceApiDao serviceApiDao;
    private final ResourceDao resourceDao;
    private final ServiceApiService serviceApiService;

    private final ResourceApiRelDao resourceApiRelDao;

    private static final String operateUserName = "自动扫描";

    private static final PathPatternParser pathPatternParser = PathPatternParser.defaultInstance;

    public ResourceAndPathService(RouteDao routeDao, ServiceApiDao serviceApiDao, ResourceDao resourceDao, ServiceApiService serviceApiService, ResourceApiRelDao resourceApiRelDao) {
        this.routeDao = routeDao;
        this.serviceApiDao = serviceApiDao;
        this.resourceDao = resourceDao;
        this.serviceApiService = serviceApiService;
        this.resourceApiRelDao = resourceApiRelDao;
    }

    @Transactional(rollbackFor = Exception.class)
    public void initItem(Route route, Boolean isServicePackage, AutoScanBody autoScanBody, List<Route> sortedRoutes) {
        List<ResourceAndPathDto> list = this.fetchResource(route, autoScanBody, sortedRoutes);
        this.saveResourceWithParent(route, isServicePackage, null, autoScanBody.getResources());
        List<ServiceApi> serviceApis = new ArrayList<>();
        this.saveResourceAndPath(list, serviceApis, route, isServicePackage);
        this.deleteServiceApi(list, serviceApis, route);
        route.setEnableAutoscan(false);
        routeDao.saveAndFlush(route);
    }

    private void saveResourceWithParent(Route route, Boolean isServicePackage, Long parentId, List<ResourceItem> resources) {
        if (CollectionUtils.isEmpty(resources)) {
            log.info("resources is empty, return");
            return;
        }
        for (ResourceItem resourceItem : resources) {
            List<Resource> list = resourceDao.findAll((Specification<Resource>) (root, query, builder) -> {
                List<Predicate> predicates = new ArrayList<>();
                predicates.add(builder.equal(root.get(Resource_.resourceCode), resourceItem.getResourceCode()));
                if (route.getAppId() != null && route.getAppId() > 0L) {
                    predicates.add(builder.equal(root.get(Resource_.appId), route.getAppId()));
                }
                query.where(predicates.toArray(new Predicate[0]));
                return query.getRestriction();
            });
            Resource sysResource;
            if (CollectionUtils.isNotEmpty(list)) {
                sysResource = list.get(0);
            } else {
                sysResource = new Resource();
            }
            this.fillResourceData(route, resourceItem.getResourceName(), resourceItem.getResourceCode(), sysResource, isServicePackage);
            if (parentId != null && parentId > 0L) {
                sysResource.setParentId(parentId);
            }
            resourceDao.saveAndFlush(sysResource);
        }
    }

    private void deleteServiceApi(List<ResourceAndPathDto> list, List<ServiceApi> insertServiceApis, Route route) {
        List<ServiceApi> existServiceApis = serviceApiDao.findAll((Specification<ServiceApi>) (root, query, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            predicates.add(builder.equal(root.get(ServiceApi_.routeId), route.getRouteId()));
            query.where(predicates.toArray(new Predicate[0]));
            return query.getRestriction();
        });
        if (CollectionUtils.isEmpty(existServiceApis)) {
            return;
        }
        Set<Long> remainServiceApiIds = insertServiceApis.stream().map(ServiceApi::getServiceApiId).collect(Collectors.toSet());
        Set<String> hashSet = this.getHashset(list);
        for (ServiceApi serviceApi : existServiceApis) {
            String hash = ApiUtils.hash(serviceApi.getServiceApiUrl(), serviceApi.getRequestMethod().name(), serviceApi.getRouteId());
            if (serviceApi.getManual() != null && serviceApi.getManual() && hashSet.contains(hash)) {
                continue;
            }

            if (remainServiceApiIds.contains(serviceApi.getServiceApiId())) {
                continue;
            }
            serviceApiDao.deleteById(serviceApi.getServiceApiId());
            List<ResourceApiRel> resourceApiRels = resourceApiRelDao.findByServiceApiId(serviceApi.getServiceApiId());
            for(ResourceApiRel resourceApiRel : resourceApiRels) {
                resourceApiRelDao.deleteById(resourceApiRel.getId());
                List<ResourceApiRel> remainResourceApiRels = resourceApiRelDao.findByResourceIdAndNotServiceApiId(resourceApiRel.getResourceId(), resourceApiRel.getServiceApiId());
                if(remainResourceApiRels.isEmpty()) {
                    resourceDao.deleteById(resourceApiRel.getResourceId());
                }
            }
        }
    }

    public Set<String> getHashset(List<ResourceAndPathDto> list) {
        Set<String> hashset = new HashSet<>(list.size());
        for (ResourceAndPathDto resourceAndPathDto : list) {
            String hash = ApiUtils.hash(resourceAndPathDto.getApiUrl(), resourceAndPathDto.getRequestMethod(), resourceAndPathDto.getRouteId());
            hashset.add(hash);
        }
        return hashset;
    }

    public List<ResourceAndPathDto> fetchResource(Route route, AutoScanBody autoScanBody, List<Route> sortedRoutes) {
        String createUserId = "";
        Map<AuthorizationUri, AuthorizationInfo> aaMap = autoScanBody.getAaMap();
        log.info("获取request_mapping size={}", aaMap.size());
        List<ResourceAndPathDto> list = new ArrayList<>(aaMap.size());
        for (Map.Entry<AuthorizationUri, AuthorizationInfo> entry : aaMap.entrySet()) {
            AuthorizationUri authorizationUri = entry.getKey();
            AuthorizationInfo authorizationInfo = entry.getValue();
            String apiPath = authorizationUri.getPath();
            Route bestMatchRoute = this.bestMatchRoute(apiPath, sortedRoutes);
            if (bestMatchRoute == null) {
                log.debug("最匹配路由(bestMatchRoute) == null, 跳过");
                continue;
            }
            if (!route.getRouteId().equals(bestMatchRoute.getRouteId())) {
                log.debug("当前路由(routeId:{})和最匹配路由(routeId:{})不一致", route.getRouteId(), bestMatchRoute.getRouteId());
                continue;
            }
            boolean hasAnyAuthorizedDefinition = false;
            if(authorizationInfo != null) {
                hasAnyAuthorizedDefinition = authorizationInfo.getSkipAuthentication() || authorizationInfo.getSkipAuthorization() || authorizationInfo.getTimeout() != null || (authorizationInfo.getResources() != null && !authorizationInfo.getResources().isEmpty());
            }
            if(!hasAnyAuthorizedDefinition) {
                log.debug("api地址(path:{}, method:{})没有配置任何@AuthorizedDefinition,跳过", apiPath, authorizationUri.getMethod());
                continue;
            }
            ResourceAndPathDto item = new ResourceAndPathDto();
            item.setAppId(route.getAppId());
            item.setApiUrl(apiPath);
            item.setRequestMethod(authorizationUri.getMethod());
            item.setName(authorizationInfo.getName());
            item.setApiPath(authorizationInfo.getMethodName());
            item.setSkipAuthentication(authorizationInfo.getSkipAuthentication());
            item.setSkipAuthorization(authorizationInfo.getSkipAuthorization());
            item.setSkipTrail(Objects.isNull(authorizationInfo.getSkipTrail())?Boolean.TRUE:authorizationInfo.getSkipTrail());
            item.setCreateUserId(createUserId);
            item.setRouteId(route.getRouteId());
            if (authorizationInfo.getResources() != null && !authorizationInfo.getResources().isEmpty()) {
                item.setResourceCodes(authorizationInfo.getResources());
            }
            list.add(item);
        }
        return list;
    }

    /**
     * 从所有路由表里找最匹配 apiPath 的路由
     * @param apiPath  service-api的 requestPath
     * @param sortedRoutes 所有路由表(已排序)
     * @return 最匹配 apiPath 的路由
     */
    private Route bestMatchRoute(String apiPath, List<Route> sortedRoutes) {
        Route bestMatchRoute = null;

        apiPath = RouteUtils.reRenderPath(apiPath);
        apiPath = StringUtils.prependIfMissing(apiPath, "/");
        PathContainer apiPathContainer = PathContainer.parsePath(apiPath);
        for (Route route : sortedRoutes) {
            String routeDestPath = UriComponentsBuilder.fromHttpUrl(route.getUrl()).build().getPath();
            String routePostPath = StringUtils.prependIfMissing(routeDestPath, "/");
            if (routePostPath == null) {
                log.warn("路由(routeId:{})的 postPath == null, continue", route.getRouteId());
                continue;
            }
            routePostPath = RouteUtils.reRenderPath(routePostPath);
            PathPattern postPathPattern = pathPatternParser.parse(routePostPath);
            PathPattern.PathRemainingMatchInfo remainingMatchInfo = postPathPattern.matchStartOfPath(apiPathContainer);
            if (remainingMatchInfo == null) {
                log.debug("api地址({})和路由(routeId:{})目的地址({})不匹配,跳过", apiPath, route.getRouteId(), routeDestPath);
                continue;
            }
            String routePrePath = route.getPath();
            routePrePath = StringUtils.prependIfMissing(routePrePath, "/");
            routePrePath = RouteUtils.reRenderPath(routePrePath);
            if (StringUtils.startsWith(apiPath, routePrePath)) {
                bestMatchRoute = route;
                break;
            } else if (bestMatchRoute == null) {
                bestMatchRoute = route;
            }
        }
        return bestMatchRoute;
    }

    private void saveResourceAndPath(List<ResourceAndPathDto> list, List<ServiceApi> serviceApis, Route route, Boolean isServicePackage) {
        String pathUri = UriComponentsBuilder.fromHttpUrl(route.getUrl()).build().getPath();
        boolean isExactMatch = route.getIsExactMatch();
        for (ResourceAndPathDto item : list) {
            if (StringUtils.isNotBlank(item.getApiUrl()) && StringUtils.isNotBlank(item.getRequestMethod())) {

                boolean matchPath = this.isMatchPath(pathUri, item, isExactMatch);
                if (!matchPath) {
                    log.info("接口无法通过该路由映射, route.path: {}, service-api.url: {}", route.getUrl(), item.getApiUrl());
                    continue;
                }
                //处理api
                ServiceApi serviceApi = this.saveServiceApi(item, route);
                serviceApis.add(serviceApi);
                //处理resource
                if (item.getResourceCodes() != null && !item.getResourceCodes().isEmpty()) {
                    item.getResourceCodes().forEach(resCode -> {
                        if (StringUtils.isNotBlank(resCode)) {
                            Resource resource = this.saveResource(route, resCode, isServicePackage);
                            if(resource != null) {
                                serviceApiService.saveResourceApiRel(resource.getResourceId(), serviceApi.getServiceApiId(), operateUserName);
                            }
                        }
                    });
                }

            }
        }
    }

    private boolean isMatchPath(String pathUri, ResourceAndPathDto item, boolean isExactMatch) {
        String pattern;
        if (!StringUtils.startsWith(pathUri, "/")) {
            pathUri = "/" + pathUri;
        }
        if (isExactMatch) {
            pattern = pathUri;
        } else {
            if (StringUtils.equals("/", pathUri)) {
                pattern = "/**";
            } else {
                pattern = pathUri + "/**";
            }
        }
        PathPatternParser pathPatternParser = new PathPatternParser();
        return pathPatternParser.parse(pattern).matches(PathContainer.parsePath(item.getApiUrl()));
    }

    public Resource saveResource(Route route, String resourceCode, Boolean isServicePackage) {
        List<Resource> list = resourceDao.findAll((Specification<Resource>) (root, query, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            predicates.add(builder.equal(root.get(Resource_.resourceCode), resourceCode));
            if (route.getAppId() != null && route.getAppId() > 0L) {
                predicates.add(builder.equal(root.get(Resource_.appId), route.getAppId()));
            }
            query.where(predicates.toArray(new Predicate[0]));
            return query.getRestriction();
        });
        Resource sysResource;
        if (CollectionUtils.isNotEmpty(list)) {
            sysResource = list.get(0);
            this.fillResourceData(route, null, resourceCode, sysResource, isServicePackage);
            resourceDao.saveAndFlush(sysResource);
            return sysResource;
        } else {
            return null;
        }
    }

    private ServiceApi saveServiceApi(ResourceAndPathDto item, Route route) {
        String hash = ApiUtils.hash(item.getApiUrl(), item.getRequestMethod(), route.getRouteId());
        List<ServiceApi> serviceApis = serviceApiDao.findByHash(hash);
        ServiceApi serviceApi;
        if (CollectionUtils.isNotEmpty(serviceApis)) {
            serviceApi = serviceApis.get(0);
        } else {
            serviceApi = new ServiceApi();
            serviceApi.setHash(hash);
        }
        this.buildServiceApi(route, item, serviceApi);
        serviceApiDao.saveAndFlush(serviceApi);
        return serviceApi;
    }

    private void buildServiceApi(Route route, ResourceAndPathDto item, ServiceApi serviceApi) {
        serviceApi.setServiceApiPath(item.getApiPath());
        serviceApi.setServiceApiUrl(item.getApiUrl());
        serviceApi.setRequestMethod(RequestMethod.valueOf(item.getRequestMethod()));
        serviceApi.setAppId(route.getAppId() == null ? 1L : route.getAppId());
        if (StringUtils.isNotBlank(item.getName())) {
            serviceApi.setServiceApiName(item.getName());
        } else {
            serviceApi.setServiceApiName("没有名称");
        }
        serviceApi.setRouteId(item.getRouteId());
        serviceApi.setSkipAuthentication(item.getSkipAuthentication());
        serviceApi.setSkipAuthorization(item.getSkipAuthorization());
        serviceApi.setStatus(1);
        Date now = Calendar.getInstance().getTime();
        if (serviceApi.getServiceApiId() == null) {
            serviceApi.setCreaterName(operateUserName);
            serviceApi.setCreateTime(now);
        }
        serviceApi.setUpdaterName(operateUserName);
        serviceApi.setUpdateTime(now);
    }

    private void fillResourceData(Route route, String resourceName, String resourceCode, Resource sysResource, Boolean isServicePackage) {
        sysResource.setAppId(route.getAppId() == null ? 1 : route.getAppId());
        sysResource.setResourceCode(resourceCode);
        if (StringUtils.isNotBlank(resourceName)) {
            sysResource.setResourceName(resourceName);
        }
        if (isServicePackage != null && !isServicePackage) {
            sysResource.setIsServicePackage(false);
        } else {
            //添加这个判断是为了防止原有系统资源码字段是跟服务包不相关(0)并且没有配服务包关联，
            // 跑完该代码后用户获取不到资源码而炸锅
            if (sysResource.getResourceId() == null
                    || StringUtils.equals(sysResource.getCreaterName(), operateUserName)) {
                sysResource.setIsServicePackage(true);
            }
        }
        if (sysResource.getResourceId() != null) {
            sysResource.setUpdaterName(operateUserName);
        } else {
            sysResource.setResourcePlatform(0);
            sysResource.setResourceType(0);
            sysResource.setStatus(1);
            sysResource.setParentId(0L);
            sysResource.setCreaterName(operateUserName);
            sysResource.setCreateTime(new Date());
        }
    }
}
