package com.xforceplus.route.service;

import com.fasterxml.jackson.core.type.TypeReference;
import com.xforceplus.route.api.common.model.RouteModel.Request.*;
import com.xforceplus.route.listener.MessagePublisher;
import com.xforceplus.dao.RouteDao;
import com.xforceplus.entity.Route;
import com.xforceplus.entity.RouteApply;
import com.xforceplus.query.RouteQueryHelper;
import com.xforceplus.tenant.security.autoscan.model.AuthorizationInfo;
import com.xforceplus.tenant.security.autoscan.model.AuthorizationUri;
import com.xforceplus.tenant.security.autoscan.model.AutoScanBody;
import com.xforceplus.utils.RouteUtils;
import io.geewit.core.exception.ProcessedException;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.data.jpa.envers.domain.ComparedRevision;
import io.geewit.utils.uuid.UUIDUtils;
import io.geewit.web.utils.JsonUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.history.RevisionSort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.util.UriComponentsBuilder;
import org.springframework.web.util.pattern.PathPattern;
import org.springframework.web.util.pattern.PathPatternParser;

import java.net.URI;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author geewit
 */
@Service
public class RouteService {
    private final static Logger logger = LoggerFactory.getLogger(RouteService.class);

    private final RouteDao routeDao;

    private final MessagePublisher messagePublisher;

    private final RedisTemplate redisTemplate;

    private final ResourceAndPathService resourceAndPathService;

    private final RestTemplate restTemplate;

    public RouteService(RouteDao routeDao,
                        MessagePublisher messagePublisher,
                        RedisTemplate redisTemplate,
                        ResourceAndPathService resourceAndPathService,
                        @Qualifier("tenantRestTemplate") RestTemplate restTemplate) {
        this.routeDao = routeDao;
        this.messagePublisher = messagePublisher;
        this.redisTemplate = redisTemplate;
        this.resourceAndPathService = resourceAndPathService;
        this.restTemplate = restTemplate;
    }

    @Transactional(rollbackFor = Exception.class)
    public Route create(Route route) {
        return routeDao.save(route);
    }

    @Transactional(rollbackFor = Exception.class)
    public Route update(Long id, Route route) {
        Route existRoute = routeDao.findById(id).orElseThrow(() -> new ProcessedException("未找到该路由", HttpStatus.NOT_FOUND.value()));
        BeanUtils.copyProperties(route, existRoute);
        return routeDao.save(existRoute);
    }

    @Transactional(rollbackFor = Exception.class)
    public void delete(Long routeId) {
        routeDao.deleteById(routeId);
    }

    public List<Route> findAll(Query query, Sort sort) {
        Specification<Route> specification = RouteQueryHelper.querySpecification(query);
        return routeDao.findAll(specification, sort);
    }

    public Page<Route> findAll(Query query, Pageable pageable) {
        Specification<Route> specification = RouteQueryHelper.querySpecification(query);
        return routeDao.findAll(specification, pageable);
    }

    public String refresh() {
        logger.info("route refresh message publish");
        String refreshId = UUIDUtils.randomUUID();
        messagePublisher.publish(refreshId);
        return refreshId;
    }

    public String getRefreshCount(String refreshId) {
        Object count = redisTemplate.opsForValue().get("refreshId:" + refreshId);
        return count == null ? "null" : String.valueOf(count);
    }

    @Transactional(rollbackFor = Exception.class)
    public Route createByRouteApply(RouteApply routeApply) {
        Route route = new Route();
        BeanUtils.copyProperties(routeApply, route);
        route.setRouteId(null);
        route.setStatus(1);
        return routeDao.save(route);
    }

    @Transactional(rollbackFor = Exception.class)
    public void scanRoute(Long routeId) {
        List<Route> sortedRoutes = this.sortedRoutes(Stream.of(routeId).collect(Collectors.toSet()));
        Optional<Route> routeOptional = sortedRoutes.stream().filter(r -> r.getRouteId().equals(routeId)).findFirst();
        if(!routeOptional.isPresent()) {
            logger.warn("不存在的路由(routeId:{})", routeId);
            return;
        }
        Route route = routeOptional.get();
        AutoScanBody autoScanBody = new AutoScanBody();
        RestTemplate restTemplate = new RestTemplate();
        UriComponentsBuilder uriComponentsBuilder = UriComponentsBuilder.fromHttpUrl(route.getUrl());
        String url = uriComponentsBuilder.replacePath("/request_mappings").build().toString();
        logger.info("url = {}", url);
        ResponseEntity<String> resp = restTemplate.getForEntity(url, String.class);
        String json = resp.getBody();
        Map<AuthorizationUri, AuthorizationInfo> aaMap = JsonUtils.fromJson(json, new TypeReference<Map<AuthorizationUri, AuthorizationInfo>>() {
        });
        logger.info("获取request_mapping size = {}", aaMap.size());
        resourceAndPathService.initItem(route, autoScanBody.getIsServicePackage(), autoScanBody, sortedRoutes);
    }

    public void scanResourceByRequest(Set<Long> routeIds, AutoScanBody autoScanBody) {
        List<Route> sortedRoutes = this.sortedRoutes(routeIds);
        for (Long routeId : routeIds) {
            Optional<Route> routeOptional = sortedRoutes.stream().filter(r -> r.getRouteId().equals(routeId)).findFirst();
            if(!routeOptional.isPresent()) {
                logger.warn("不存在的路由(routeId:{}), 跳过", routeId);
                continue;
            }
            Route route = routeOptional.get();
//            if (route.getEnableAutoscan() == null || !route.getEnableAutoscan()) {
//                logger.info("路由(routeId:{})未开启自动扫描, 跳过", routeId);
//                continue;
//            }
            resourceAndPathService.initItem(route, autoScanBody.getIsServicePackage(), autoScanBody, sortedRoutes);
        }
    }

    private List<Route> sortedRoutes(Set<Long> routeIds) {
        Query query = new Query();
        query.setRouteIds(StringUtils.join(routeIds, ","));
        query.setStatus(1);
        query.setIsExactMatch(false);
        List<Route> routes = this.findAll(query, Sort.unsorted());
        if (routes.isEmpty()) {
            return routes;
        }
        for (Route route : routes) {
            route.setPath(RouteUtils.reRenderPath(route.getPath()));
            route.setUrl(RouteUtils.reRenderUrl(route.getUrl()));
        }
        routes.sort(Comparator.nullsLast(
                (o1, o2) -> {
                    try {
                        if (!routeIds.isEmpty()) {
                            if (routeIds.contains(o1.getRouteId()) && !routeIds.contains(o2.getRouteId())) {
                                return -1;
                            } else if (!routeIds.contains(o1.getRouteId()) && routeIds.contains(o2.getRouteId())) {
                                return 1;
                            }
                        }
                        String processedPrePath1;
                        if (o1.getIsExactMatch()) {
                            processedPrePath1 = StringUtils.removeEnd(o1.getPath(), "/");
                        } else {
                            processedPrePath1 = StringUtils.appendIfMissing(o1.getPath(), "/**");
                        }
                        processedPrePath1 = StringUtils.prependIfMissing(processedPrePath1, "/");

                        String processedPrePath2;
                        if (o2.getIsExactMatch()) {
                            processedPrePath2 = StringUtils.removeEnd(o2.getPath(), "/");
                        } else {
                            processedPrePath2 = StringUtils.appendIfMissing(o2.getPath(), "/**");
                        }
                        processedPrePath2 = StringUtils.prependIfMissing(processedPrePath2, "/");

                        PathPattern thisPathPattern = PathPatternParser.defaultInstance.parse(processedPrePath1);
                        PathPattern thatPathPattern = PathPatternParser.defaultInstance.parse(processedPrePath2);
                        int compare = PathPattern.SPECIFICITY_COMPARATOR.compare(thisPathPattern, thatPathPattern);
                        if(compare == 0) {
                            if (o1.getIsExactMatch() && !o2.getIsExactMatch()) {
                                return -1;
                            } else if (!o1.getIsExactMatch() && o2.getIsExactMatch()) {
                                return 1;
                            } else {
                                return 0;
                            }
                        }
                        return compare;
                    } catch (Exception e) {
                        logger.warn(e.getMessage(), e);
                        return 0;
                    }
                }
        ));
        return routes;
    }

    @Transactional(rollbackFor = Exception.class)
    public void refreshHash() {
        Query query = new Query();
        Specification<Route> specification = RouteQueryHelper.querySpecification(query);
        List<Route> routes = routeDao.findAll(specification);
        for (Route route : routes) {
            String hash = RouteUtils.hash(route.getPath());
            route.setHash(hash);
            routeDao.save(route);
        }
    }

    public long countByHash(String hash) {
        return routeDao.countByHash(hash);
    }

    public void checkRoute(long routeId, String healthPath) {
        Route route = routeDao.findById(routeId).orElseThrow(() -> new IllegalArgumentException("错误的路由id(" + routeId + ")"));
        if (StringUtils.isNotBlank(healthPath) && !healthPath.equals(route.getHealthPath())) {
            route.setHealthPath(healthPath);
            routeDao.saveAndFlush(route);
        }
        URI healthUri = UriComponentsBuilder.fromHttpUrl(route.getUrl()).replacePath(route.getHealthPath()).build().toUri();
        try {
            ResponseEntity<String> responseEntity = restTemplate.getForEntity(healthUri, String.class);
            if (!responseEntity.getStatusCode().is2xxSuccessful()) {
                throw new IllegalArgumentException("状态码不是200, 而是:" + responseEntity.getStatusCodeValue() + ", 错误信息:" + responseEntity.getBody());
            }
        } catch (RestClientException e) {
            throw new IllegalArgumentException(e.getMessage());
        }
    }

    public Page<ComparedRevision<Route, String>> findHistories(long id, Pageable pageable) {
        pageable = PageRequest.of(pageable.getPageNumber(), pageable.getPageSize(), pageable.getSort().and(RevisionSort.desc()));
        Page<ComparedRevision<Route, String>> page = routeDao.findComparedRevisions(id, pageable);
        return page;
    }
}
