package com.xforceplus.route.service;

import com.xforceplus.dao.ResourceApiRelDao;
import com.xforceplus.dao.ResourceDao;
import com.xforceplus.dao.ServiceApiDao;
import com.xforceplus.entity.*;
import com.xforceplus.query.ServiceApiQueryHelper;
import com.xforceplus.route.api.common.model.ServiceApiModel;
import com.xforceplus.dao.RouteDao;
import com.xforceplus.utils.ApiUtils;
import io.geewit.core.utils.reflection.BeanUtils;
import io.geewit.data.jpa.essential.domain.EntityGraphs;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.data.jpa.domain.Specification;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import javax.persistence.criteria.Predicate;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author geewit
 */
@Service
public class ServiceApiService {
    private final static Logger logger = LoggerFactory.getLogger(ServiceApiService.class);

    private final RouteDao routeDao;

    private final ServiceApiDao serviceApiDao;

    private final ResourceApiRelDao resourceApiRelDao;

    private final ResourceDao resourceDao;

    public ServiceApiService(RouteDao routeDao, ServiceApiDao serviceApiDao, ResourceApiRelDao resourceApiRelDao, ResourceDao resourceDao) {
        this.routeDao = routeDao;
        this.serviceApiDao = serviceApiDao;
        this.resourceApiRelDao = resourceApiRelDao;
        this.resourceDao = resourceDao;
    }


    public Page<ServiceApi> page(ServiceApiModel.Request.Query query, Pageable pageable) {
        Specification<ServiceApi> specification = ServiceApiQueryHelper.querySpecification(query);
        return serviceApiDao.findAll(specification, pageable, EntityGraphs.named(ServiceApi.NAMED_ENTITY_GRAPH_DEFAULT));
    }

    public Page<ServiceApi> page(Specification<ServiceApi> specification, Pageable pageable) {
        return serviceApiDao.findAll(specification, pageable, EntityGraphs.named(ServiceApi.NAMED_ENTITY_GRAPH_DEFAULT));
    }

    public List<ServiceApi> list(ServiceApiModel.Request.Query query, Sort sort) {
        Specification<ServiceApi> specification = ServiceApiQueryHelper.querySpecification(query);
        List<ServiceApi> list = serviceApiDao.findAll(specification, sort, EntityGraphs.named(ServiceApi.NAMED_ENTITY_GRAPH_DEFAULT));
        return list;
    }

    public List<ServiceApi> list(Specification<ServiceApi> specification, Sort sort) {
        return serviceApiDao.findAll(specification, sort, EntityGraphs.named(ServiceApi.NAMED_ENTITY_GRAPH_DEFAULT));
    }

    @Transactional(rollbackFor = Exception.class)
    public ServiceApi save(ServiceApiModel.Request.Save model) {
        ServiceApi entity = new ServiceApi();
        BeanUtils.copyProperties(model, entity, Stream.of(ServiceApi_.APP, ServiceApi_.RESOURCE_API_RELS).toArray(String[]::new));
        entity.setManual(true);
        entity = serviceApiDao.saveAndFlush(entity);
        this.bindResources(entity, model.getBindResources());
        return entity;
    }

    @Transactional(rollbackFor = Exception.class)
    public ServiceApi update(long id, ServiceApiModel.Request.Update model) {
        ServiceApi existEntity = this.findById(id);
        if (StringUtils.isBlank(model.getServiceApiName())) {
            model.setServiceApiName(null);
        }
        if (StringUtils.isBlank(model.getServiceApiPath())) {
            model.setServiceApiPath(null);
        }
        if (StringUtils.isBlank(model.getServiceApiUrl())) {
            model.setServiceApiUrl(null);
        }
        BeanUtils.copyProperties(model, existEntity, Stream.of(ServiceApi_.APP, ServiceApi_.RESOURCE_API_RELS).toArray(String[]::new));
        existEntity = serviceApiDao.saveAndFlush(existEntity);
        this.bindResources(existEntity, model.getBindResources());
        return existEntity;
    }

    public ServiceApi findById(Long id) {
        return serviceApiDao.findById(id).orElseThrow(() -> new IllegalArgumentException("未找到实体"));
    }

    @Transactional(rollbackFor = Exception.class)
    public void deleteById(long id) {
        Optional<ServiceApi> optionalServiceApi = serviceApiDao.findById(id);
        if (!optionalServiceApi.isPresent()) {
            String message = "不存在id:" + id + "的服务接口";
            throw new IllegalArgumentException(message);
        }

        long countRel = resourceApiRelDao.countByServiceApiId(id);
        if (countRel > 0) {
            throw new IllegalArgumentException("已经关联关系不能删除");
        }
        serviceApiDao.deleteById(id);
        resourceApiRelDao.deleteByServiceApiId(id);
    }

    @Transactional(rollbackFor = Exception.class)
    public boolean saveResourceApiRel(Long resourceId, Long serviceApiId, String operateUserName) {
        List<ResourceApiRel> list = resourceApiRelDao.findAll((Specification<ResourceApiRel>) (root, query, builder) -> {
            List<Predicate> predicates = new ArrayList<>();
            predicates.add(builder.equal(root.get(ResourceApiRel_.serviceApiId), serviceApiId));
            predicates.add(builder.equal(root.get(ResourceApiRel_.resourceId), resourceId));
            query.where(predicates.toArray(new Predicate[0]));
            return query.getRestriction();
        });
        if (list.isEmpty()) {
            ResourceApiRel resourceApiRel = new ResourceApiRel();
            resourceApiRel.setResourceId(resourceId);
            resourceApiRel.setServiceApiId(serviceApiId);
            resourceApiRel.setCreateUserName(operateUserName);
            resourceApiRelDao.saveAndFlush(resourceApiRel);
            return true;
        }
        return false;
    }

    /**
     * @param id            接口id
     * @param bindResources
     */
    @Transactional(rollbackFor = Exception.class)
    public void bindResources(long id, ServiceApiModel.Request.BindResources bindResources) {
        ServiceApi serviceApi = this.findById(id);
        this.bindResources(serviceApi, bindResources);
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindResources(ServiceApi serviceApi, ServiceApiModel.Request.BindResources bindResources) {
        if (bindResources == null) {
            return;
        }
        List<Long> resourceIds = bindResources.getResourceIds();
        if (resourceIds == null) {
            return;
        }
        logger.info("resourceIds = " + resourceIds.stream().filter(Objects::nonNull).map(Object::toString).collect(Collectors.joining(",")));
        List<ResourceApiRel> existRels = resourceApiRelDao.findByServiceApiId(serviceApi.getServiceApiId());
        logger.info("exist api-resource-rels.size = " + existRels.size());
        //region 往绑定不在数据库中但是报文 resourcesetIds 中有的服务包-功能集关系
        Set<ResourceApiRel> insertingRels = resourceIds.stream().filter(resourceId -> existRels.stream().map(ResourceApiRel::getResourceId).noneMatch(existId -> existId.equals(resourceId))).map(resourceId -> {
            Optional<Resource> resourceOptional = resourceDao.findById(resourceId);
            if (resourceOptional.isPresent()) {
                ResourceApiRel rel = new ResourceApiRel();
                rel.setResourceId(resourceId);
                rel.setServiceApiId(serviceApi.getServiceApiId());
                return rel;
            } else {
                return null;
            }
        }).filter(Objects::nonNull).collect(Collectors.toSet());
        this.bindResources(insertingRels);
        //endregion
        if (bindResources.isOverwrite()) {
            //region 从数据库中解绑不在报文 resourcesetIds 中的服务包-功能集关系
            existRels.stream().filter(rel -> resourceIds.stream().noneMatch(resourceId -> resourceId.equals(rel.getResourceId()))).forEach(rel -> {
                logger.info("deleting Resource-Api-Rel record, {}", rel);
                resourceApiRelDao.deleteById(rel.getId());
            });
            //endregion
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void bindResources(Collection<ResourceApiRel> rels) {
        if (!rels.isEmpty()) {
            rels.forEach(resourceApiRelDao::saveAndFlush);
        }
    }

    @Transactional(rollbackFor = Exception.class)
    public void unbindServiceApiResourceRel(long serviceApiId, long resourceId) {
        resourceApiRelDao.deleteByServiceApiIdAndResourceId(serviceApiId, resourceId);
    }

    public long countByHash(String hash) {
        return serviceApiDao.countByHash(hash);
    }

    /**
     * 刷新所有ServiceApi的hash值
     */
    public void refreshHash() {
        this.refreshHashByRouteId(null);
    }

    /**
     * 根据routeId刷新ServiceApi的hash值
     *
     * @param routeId
     */
    public void refreshHashByRouteId(Long routeId) {
        ServiceApiModel.Request.Query query = new ServiceApiModel.Request.Query();
        if (routeId != null && routeId > 0) {
            query.setRouteId(routeId);
        }
        Specification<ServiceApi> specification = ServiceApiQueryHelper.querySpecification(query);
        Pageable pageable = Pageable.ofSize(1000);
        Page<ServiceApi> page;
        do {
            page = serviceApiDao.findAll(specification, pageable);
            for (ServiceApi serviceApi : page) {
                String hash = ApiUtils.hash(serviceApi.getServiceApiUrl(), serviceApi.getRequestMethod().name(), serviceApi.getRouteId());
                serviceApi.setHash(hash);
                serviceApi.setApp(null);
                serviceApi.setResourceApiRels(null);
                serviceApiDao.save(serviceApi);
            }
            pageable = page.nextPageable();
        } while (page.hasNext());
    }

    public Map<Long, Set<ServiceApiModel.Response.ServiceApiWithResource>> all() {
        try {
            return routeDao.all();
        } catch (Exception e) {
            logger.warn(e.getMessage());
            return Collections.emptyMap();
        }
    }
}
