package com.xforceplus.tenant.security.autoscan.config;

import com.xforceplus.api.global.autoscan.AutoscanApi;
import com.xforceplus.tenant.security.autoscan.listener.AutoScanApplicationContextListener;
import com.xforceplus.tenant.security.autoscan.servlet.AutoScanRequestMappingsServlet;
import com.xforceplus.tenant.security.client.feign.config.XforceDataWebProperties;
import com.xforceplus.tenant.security.client.feign.service.ClientService;
import com.xforceplus.tenant.security.client.feign.utils.FeignUtils;
import feign.Feign;
import feign.RequestInterceptor;
import feign.okhttp.OkHttpClient;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.AutoConfigureAfter;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.data.web.SpringDataWebProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.servlet.ServletRegistrationBean;
import org.springframework.context.ApplicationListener;
import org.springframework.context.ResourceLoaderAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.io.ResourceLoader;

/**
 * @author geewit
 */
@Slf4j
@ConditionalOnMissingBean(TenantAutoScanConfiguration.class)
@Configuration(value = "tenantAutoScanConfiguration")
@AutoConfigureAfter(name = {"tenantFeignConfiguration"})
@EnableConfigurationProperties({AutoScanProperties.class, XforceDataWebProperties.class})
public class TenantAutoScanConfiguration implements ResourceLoaderAware {

    private final SpringDataWebProperties springDataWebProperties;
    private final XforceDataWebProperties xforceDataWebProperties;

    private final AutoScanProperties autoScanProperties;

    @Value("${xforce.tenant.service.route_service:http://route-mgmt:8080}")
    private String routeServiceUrl;

    public TenantAutoScanConfiguration(@Autowired(required = false) XforceDataWebProperties xforceDataWebProperties,
                                       @Autowired(required = false) SpringDataWebProperties springDataWebProperties,
                                       AutoScanProperties autoScanProperties) {
        this.xforceDataWebProperties = xforceDataWebProperties;
        this.springDataWebProperties = springDataWebProperties;
        this.autoScanProperties = autoScanProperties;
        log.info("TenantAutoScanConfiguration initialized");
    }

    @Override
    public void setResourceLoader(ResourceLoader resourceLoader) {
        this.resourceLoader = resourceLoader;
    }

    private ResourceLoader resourceLoader;


    @Bean(name = "autoScanApplicationContextListener")
    public ApplicationListener<ContextRefreshedEvent> autoScanApplicationContextListener() {
        if (autoScanProperties != null && this.autoScanProperties.getEnabled() != null && this.autoScanProperties.getEnabled()) {
            log.info("Autoscan,开关已经打开。");
            return new AutoScanApplicationContextListener(this.autoScanProperties);
        } else {
            log.info("Autoscan,开关未打开。");
            return event -> {
            };
        }
    }

    @Bean(name = "autoScanRequestMappingsServlet")
    public AutoScanRequestMappingsServlet autoScanRequestMappingsServlet() {
        return new AutoScanRequestMappingsServlet();
    }

    @Bean
    public ServletRegistrationBean<AutoScanRequestMappingsServlet> getServletRegistrationBean(AutoScanRequestMappingsServlet autoScanRequestMappingsServlet) {
        ServletRegistrationBean<AutoScanRequestMappingsServlet> bean = new ServletRegistrationBean<>(autoScanRequestMappingsServlet);
        //访问路径值
        bean.addUrlMappings("/request_mappings");
        return bean;
    }

    private OkHttpClient client(okhttp3.OkHttpClient okHttpClient) {
        return new OkHttpClient(okHttpClient);
    }

    @Bean
    public AutoscanApi routeApi(@Autowired @Qualifier("tenantOkHttpClient")okhttp3.OkHttpClient tenantOkHttpClient,
                                @Autowired(required = false) @Qualifier("tenantTokenRequestInterceptor") RequestInterceptor requestInterceptor,
                                @Autowired(required = false) @Qualifier("tenantClientService") ClientService clientService) {
        String pageParameter = StringUtils.defaultString(xforceDataWebProperties.getPageable().getPageParameter(), springDataWebProperties.getPageable().getPageParameter());
        String sizeParameter = StringUtils.defaultString(xforceDataWebProperties.getPageable().getSizeParameter(), springDataWebProperties.getPageable().getSizeParameter());
        String sortParameter = StringUtils.defaultString(xforceDataWebProperties.getSort().getSortParameter(), springDataWebProperties.getSort().getSortParameter());
        Boolean oneIndexedParameters = xforceDataWebProperties.getPageable().isOneIndexedParameters();
        Feign.Builder builder = FeignUtils.of(this.client(tenantOkHttpClient),
                clientService,
                this.resourceLoader,
                pageParameter,
                sizeParameter,
                sortParameter,
                oneIndexedParameters);
        if(requestInterceptor != null) {
            builder.requestInterceptor(requestInterceptor);
        }
        return builder.target(AutoscanApi.class, this.routeServiceUrl);
    }
}
