package com.xforceplus.tenant.security.autoscan.listener;

import com.xforceplus.api.global.route.RouteApi;
import com.xforceplus.tenant.security.autoscan.config.AutoScanProperties;
import com.xforceplus.tenant.security.autoscan.model.AuthorizationInfo;
import com.xforceplus.tenant.security.autoscan.model.AuthorizationUri;
import com.xforceplus.tenant.security.autoscan.model.AutoScanBody;
import com.xforceplus.tenant.security.autoscan.model.ResourceItem;
import com.xforceplus.tenant.security.autoscan.utils.AuthorizationInfoUtils;
import com.xforceplus.tenant.security.autoscan.utils.ResourceCodeUtils;
import com.xforceplus.tenant.security.core.api.response.ResponseEntity;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.util.CollectionUtils;
import org.springframework.web.client.RestTemplate;

import java.net.InetAddress;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;


/**
 * @author geewit
 */
@SuppressWarnings("all")
public class AutoScanApplicationContextListener implements ApplicationContextAware, ApplicationListener<ContextRefreshedEvent> {

    private final static Logger logger = LoggerFactory.getLogger(AutoScanApplicationContextListener.class);

    public AutoScanApplicationContextListener(AutoScanProperties autoScanProperties) {
        this.autoScanProperties = autoScanProperties;
    }

    private ApplicationContext applicationContext;

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    private final AtomicBoolean locker = new AtomicBoolean(false);

    @Autowired
    private final AutoScanProperties autoScanProperties;

    private Map<AuthorizationUri, AuthorizationInfo> mappings;

    @Value("${server.port:8080}")
    private String localPort;

    @Autowired
    private RouteApi routeApi;

    private ExecutorService executorService = Executors.newSingleThreadExecutor();

    @Override
    public void onApplicationEvent(ContextRefreshedEvent contextRefreshedEvent) {
        boolean result = locker.compareAndSet(false, true);
        if (!result) {
            logger.info("该事件被锁定, 等待就绪......");
            return;
        }
        logger.info("进入应用监听方法");
        executorService.execute(() -> {
            try {
                logger.info("进入调用自动扫描功能方法");
                this.doAutoScan();
            } catch (Exception e) {
                logger.error("调用自动扫描功能失败", e);
            }
        });

    }

    private void doAutoScan() throws Exception {
        RestTemplate restTemplate = new RestTemplate();
        restTemplate.getMessageConverters().stream()
                .filter(httpMessageConverter -> httpMessageConverter instanceof StringHttpMessageConverter)
                .forEach(httpMessageConverter -> ((StringHttpMessageConverter) httpMessageConverter).setDefaultCharset(StandardCharsets.UTF_8));
        if (StringUtils.isBlank(autoScanProperties.getRouteIds())) {
            throw new RuntimeException("启动自动扫描，却没配置路由routeIds.");
        }
        String routeIds = autoScanProperties.getRouteIds();
        String host = InetAddress.getLocalHost().getHostAddress();
        List<ResourceItem> resources = ResourceCodeUtils.readYmlResource(autoScanProperties.getYamlPath());
        Set<String> resourceCodeSet = new HashSet<>();
        if (!CollectionUtils.isEmpty(resources)) {
            resources.stream().filter(Objects::nonNull).forEach(item -> resourceCodeSet.add(item.getResourceCode()));
        }
        mappings = AuthorizationInfoUtils.getAuthorizationUriAuthorizationInfoMap(this.applicationContext, null);
        logger.info("调用自动扫描接口");
        AutoScanBody autoScanBody = new AutoScanBody();
        autoScanBody.setAaMap(mappings);
        autoScanBody.setResources(resources);
        autoScanBody.setRouteIds(routeIds);
        autoScanBody.setHost(host);
        autoScanBody.setIsServicePackage(autoScanProperties.getIsServicePackage());
        ResponseEntity<String> result = routeApi.batchScan(autoScanBody);
        logger.info("调用自动扫描结果, result=={}", result.getResult());
    }

    private void addResourceCode(Set<String> resourceCodeSet, ResourceItem resourceItem) {
        resourceCodeSet.add(resourceItem.getResourceCode());
    }

}
