package com.xforceplus.tenant.security.autoscan.listener;

import com.xforceplus.api.global.autoscan.AutoscanApi;
import com.xforceplus.tenant.security.autoscan.config.AutoScanProperties;
import com.xforceplus.tenant.security.autoscan.model.*;
import com.xforceplus.tenant.security.autoscan.utils.AuthorizationInfoUtils;
import com.xforceplus.tenant.security.autoscan.utils.ResourceCodeUtils;
import com.xforceplus.tenant.security.core.api.response.ResponseEntity;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.http.converter.StringHttpMessageConverter;
import org.springframework.util.CollectionUtils;
import org.springframework.web.client.RestTemplate;

import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;


/**
 * @author geewit
 */
@SuppressWarnings("all")
public class AutoScanApplicationContextListener implements ApplicationContextAware, ApplicationListener<ContextRefreshedEvent> {

    private final static Logger logger = LoggerFactory.getLogger(AutoScanApplicationContextListener.class);

    public AutoScanApplicationContextListener(AutoScanProperties autoScanProperties) {
        this.autoScanProperties = autoScanProperties;
        logger.info("tenant-security.AutoScanApplicationContextListener initialized");
    }

    private ApplicationContext applicationContext;

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    private final AtomicBoolean locker = new AtomicBoolean(false);

    @Autowired
    private final AutoScanProperties autoScanProperties;

    @Autowired
    private AutoscanApi routeApi;

    private ExecutorService executorService = Executors.newSingleThreadExecutor();

    @Override
    public void onApplicationEvent(ContextRefreshedEvent contextRefreshedEvent) {
        AtomicInteger count = new AtomicInteger(0);
        while (!locker.get() && count.get() < 10) {
            if(count.get() > 0) {
                logger.info("Autoscan,重试第{}次", count.get());
            }
            locker.compareAndSet(false, true);
            logger.info("Autoscan,进入应用监听方法");
            executorService.execute(() -> {
                try {
                    logger.info("Autoscan,进入调用自动扫描功能方法");
                    this.doAutoScan();
                    return;
                } catch (Exception e) {
                    logger.error("Autoscan,调用自动扫描功能失败", e);
                    locker.set(false);
                    try {
                        int wait = 30;
                        Thread.sleep(wait * 1000);
                        logger.info("Autoscan,等待{}秒重试", wait);
                    } catch (InterruptedException interruptedException) {
                        logger.warn(interruptedException.getMessage());
                        return;
                    }
                } finally {
                    count.getAndIncrement();
                }
            });
        }
    }

    private void doAutoScan() throws Exception {
        RestTemplate restTemplate = new RestTemplate();
        restTemplate.getMessageConverters().stream()
                .filter(httpMessageConverter -> httpMessageConverter instanceof StringHttpMessageConverter)
                .forEach(httpMessageConverter -> ((StringHttpMessageConverter) httpMessageConverter).setDefaultCharset(StandardCharsets.UTF_8));
        if (StringUtils.isBlank(autoScanProperties.getRouteIds())) {
            throw new RuntimeException("Autoscan,启动自动扫描, 却没配置路由 routeIds.");
        }
        if (StringUtils.isBlank(autoScanProperties.getRoutePaths())) {
            throw new RuntimeException("Autoscan,启动自动扫描, 却没配置路由 routePaths.");
        }
        Properties mergedProperties = null;
        Properties yaml = ResourceCodeUtils.readPropertiesResource(autoScanProperties.getYamlPath());
        if (yaml != null) {
            mergedProperties = yaml;
        }
        Properties properties = ResourceCodeUtils.readPropertiesResource(autoScanProperties.getPropertiesPath());
        if (properties != null) {
            if (mergedProperties == null) {
                mergedProperties = properties;
            } else {
                mergedProperties.putAll(properties);
            }
        }
        List<ResourceItem> resources;
        if (mergedProperties != null) {
            resources = new ArrayList<>(mergedProperties.size());
            for (Map.Entry<Object, Object> entry : mergedProperties.entrySet()) {
                ResourceItem resourceItem = new ResourceItem();
                resourceItem.setResourceCode(StringUtils.replace(entry.getKey().toString(), ".", ":"));
                resourceItem.setResourceName(entry.getValue().toString());
                resources.add(resourceItem);
            }
        } else {
            resources = new ArrayList<>();
        }

        Set<String> resourceCodeSet = new HashSet<>();
        if (!CollectionUtils.isEmpty(resources)) {
            resources.stream().filter(Objects::nonNull).forEach(item -> resourceCodeSet.add(item.getResourceCode()));
        }
        String routeIds = StringUtils.trimToNull(autoScanProperties.getRouteIds());
        String routePaths = StringUtils.trimToNull(autoScanProperties.getRoutePaths());
        RequestMappingsHolder.REQUEST_MAPPINGS = AuthorizationInfoUtils.getAuthorizationUriAuthorizationInfoMap(this.applicationContext, null);
        logger.info("Autoscan,调用自动扫描接口");
        AutoScanBody autoScanBody = new AutoScanBody();
        autoScanBody.setAaMap(RequestMappingsHolder.REQUEST_MAPPINGS);
        autoScanBody.setResources(resources);
        autoScanBody.setRouteIds(routeIds);
        autoScanBody.setRoutePaths(routePaths);
        autoScanBody.setIsServicePackage(autoScanProperties.getIsServicePackage());
        ResponseEntity<String> result = routeApi.batchScan(autoScanBody);
        logger.info("Autoscan,调用自动扫描结果, result=={}", result.getResult());
    }

    private void addResourceCode(Set<String> resourceCodeSet, ResourceItem resourceItem) {
        resourceCodeSet.add(resourceItem.getResourceCode());
    }

}
