package com.xforceplus.tenant.security.client.feign.service;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.xforceplus.tenant.security.client.feign.support.OauthClientProperties;
import io.geewit.web.utils.JsonUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.Map;
import java.util.concurrent.atomic.AtomicInteger;


/**
 * @author geewit
 */
@Slf4j
public class ClientService {

    public ClientService(@Qualifier("tenantRestTemplate") RestTemplate restTemplate, OauthClientProperties clientProperties) {
        this.restTemplate = restTemplate;
        this.clientProperties = clientProperties;
        this.cache = CacheBuilder.newBuilder()
                .maximumSize(1)
                .expireAfterWrite(Duration.of(clientProperties.getExpireHours(), ChronoUnit.HOURS))
                .build(new CacheLoader<String, String>() {
                    @Override
                    public String load(@NotNull String key) {
                        return ClientService.this.fetchToken();
                    }
                });
        log.info("ClientService initialized");
    }

    private final AtomicInteger counter = new AtomicInteger(0);

    private final RestTemplate restTemplate;

    private final OauthClientProperties clientProperties;

    private final LoadingCache<String, String> cache;

    public String token() {
        if(clientProperties.getParams() == null) {
            log.warn("clientProperties.params == null, return null");
            return null;
        }
        String clientId = clientProperties.getParams().getClientId();
        if(StringUtils.isBlank(clientId)) {
            log.warn("clientProperties.params.clientId is blank, return null");
            return null;
        }
        if(StringUtils.isBlank(clientProperties.getParams().getSecret())) {
            log.warn("clientProperties.params.secret is blank, return null");
            return null;
        }
        log.debug("clientProperties.params.clientId = {}", clientId);
        return cache.getUnchecked(clientId);
    }

    public void refresh() {
        log.info("refresh cache");
        cache.invalidateAll();
    }

    private String fetchToken() {
        try {
            HttpHeaders headers = new HttpHeaders();
            headers.setContentType(MediaType.APPLICATION_JSON);
            HttpEntity<OauthClientProperties.Params> request = new HttpEntity<>(clientProperties.getParams(), headers);
            log.debug("request.url = {}", clientProperties.getRequestUrl());
            ResponseEntity<String> responseEntity = restTemplate.postForEntity(clientProperties.getRequestUrl(), request, String.class);
            if(responseEntity.getStatusCode().is2xxSuccessful()) {
                String body = responseEntity.getBody();
                Map<String, String> bodyMap = JsonUtils.fromJson(body, new TypeReference<Map<String, String>>(){});
                String token = bodyMap.get("data");
                log.info("fetch token: {}", token);
                if(StringUtils.isEmpty(token)) {
                    return this.exceptionFetch();
                } else {
                    counter.set(0);
                    return token;
                }
            } else {
                log.warn("requestUrl: {} failed, (reponse.status: {})", clientProperties.getRequestUrl(), responseEntity.getStatusCodeValue());
                return this.exceptionFetch();
            }
        } catch (RestClientException e) {
            log.warn("requestUrl: {} failed, e: {}", clientProperties.getRequestUrl(), e.getMessage());
            return this.exceptionFetch();
        }
    }

    private String exceptionFetch() {
        this.refresh();
        try {
            log.info("sleep 1s");
            Thread.sleep(clientProperties.getSleepMillis());
            log.info("fetchToken again");
            int count = counter.getAndAdd(1);
            if(count > clientProperties.getMaxFailTimes()) {
                counter.set(0);
                String message = "发生无法解决的错误";
                log.warn(message);
                throw new RuntimeException(message);
            }
            return this.fetchToken();
        } catch (InterruptedException e) {
            log.warn("sleep exception: {}", e.getMessage());
            return this.fetchToken();
        }
    }
}
